{ **********************************************************************
  *                              COMMON.INC                            *
  **********************************************************************
            This code is common to MATHFUNC.INC and MATH387.INC
  ********************************************************************** }


const
  NFACT = 33;  { The factorials of the first NFACT integers are stored
                 in a table }
var
  MathErr : Integer;  { Error code from last function evaluation }

  FactArray : array[0..NFACT] of Float;  { Table of factorials }


  function FMin(X, Y : Float) : Float;
  begin
    if X <= Y then
      FMin := X
    else
      FMin := Y;
  end;

  function FMax(X, Y : Float) : Float;
  begin
    if X >= Y then
      FMax := X
    else
      FMax := Y;
  end;

  function IMin(X, Y : Integer) : Integer;
  begin
    if X <= Y then
      IMin := X
    else
      IMin := Y;
  end;

  function IMax(X, Y : Integer) : Integer;
  begin
    if X >= Y then
      IMax := X
    else
      IMax := Y;
  end;

  procedure FSwap(var X, Y : Float);
  var
    Temp : Float;
  begin
    Temp := X;
    X := Y;
    Y := Temp;
  end;

  procedure ISwap(var X, Y : Integer);
  var
    Temp : Integer;
  begin
    Temp := X;
    X := Y;
    Y := Temp;
  end;

  function Sgn(X : Float) : Integer;
  begin
    if X >= 0.0 then
      Sgn := 1
    else
      Sgn := - 1;
  end;

  function Sgn0(X : Float) : Integer;
  begin
    if X > 0.0 then
      Sgn0 := 1
    else if X = 0.0 then
      Sgn0 := 0
    else
      Sgn0 := - 1;
  end;

  function DefaultVal(ErrCode : Integer) : Float;
  { Sets the global variable MathErr and the function default value
    according to the error code }
  begin
    MathErr := ErrCode;
    case ErrCode of
      DOMAIN : DefaultVal := 0.0;
      SING : DefaultVal := MAXNUM;
      OVERFLOW : DefaultVal := MAXNUM;
      UNDERFLOW : DefaultVal := 0.0;
    else
      DefaultVal := 0.0;
    end;
  end;

  function IntPow(X : Float; N : Integer) : Float;
  { Computes X^N by repeated multiplications }
  var
    M : Integer;
    T : Float;
  begin
    MathErr := FN_OK;

    if X = 0.0 then
      begin
        if N = 0 then       { 0^0 = lim  x^x = 1 }
          IntPow := 1.0     {       x->0         }
        else if N > 0 then
          IntPow := 0.0     { 0^N = 0 }
        else
          IntPow := DefaultVal(SING);
        Exit;
      end;

    if N = 0 then
      begin
        IntPow := 1.0;
        Exit;
      end;

    { Legendre's algorithm for minimizing the number of multiplications }
    T := 1.0;
    M := Abs(N);
    repeat
      if Odd(M) then
        begin
          Dec(M);
          T := T * X;
        end
      else
        begin
          M := M div 2;
          X := Sqr(X);
        end;
    until M = 0;

    if N > 0 then IntPow := T else IntPow := 1.0 / T;
  end;

  function Pythag(X, Y : Float) : Float;
  { Computes Sqrt(X^2 + Y^2) without destructive underflow or overflow }
  var
    AbsX, AbsY : Float;
  begin
    MathErr := FN_OK;
    AbsX := Abs(X);
    AbsY := Abs(Y);
    if AbsX > AbsY then
      Pythag := AbsX * Sqrt(1.0 + Sqr(AbsY / AbsX))
    else if AbsY = 0.0 then
      Pythag := 0.0
    else
      Pythag := AbsY * Sqrt(1.0 + Sqr(AbsX / AbsY));
  end;

  function FixAngle(Theta : Float) : Float;
  begin
    MathErr := FN_OK;
    while Theta > PI do
      Theta := Theta - TWOPI;
    while Theta <= - PI do
      Theta := Theta + TWOPI;
    FixAngle := Theta;
  end;

  function MathError : Integer;
  begin
    MathError := MathErr;
  end;

  procedure InitGlobalVar;
  { Initializes the global variables of the unit }
  var
    I : Integer;
  begin
    { Initialize MathErr }
    MathErr := FN_OK;

    { Store the factorials of the first NFACT integers in a table }
    FactArray[0] := 1.0;
    FactArray[1] := 1.0;
    FactArray[2] := 2.0;
    for I := 3 to NFACT do
      FactArray[I] := FactArray[I - 1] * I;
  end;
