{ **********************************************************************
  *                            Unit FMATH.PAS                          *
  *                              Version 2.2                           *
  *                     (c) J. Debord, October 1999                    *
  **********************************************************************
        This unit implements some mathematical functions in Pascal
  **********************************************************************
  Notes:

  1) The default real type is DOUBLE (8-byte real). Depending on the
     compiler, other types may be selected by defining the symbols:

     ---------------------------------------------------------
     Symbol        Type                    TP-BP-Delphi    FPK
     ---------------------------------------------------------
     SINGLEREAL    SINGLE   (4-byte real)        X          X
     PASCALREAL    REAL     (6-byte real)        X
     EXTENDEDREAL  EXTENDED (10-byte real)       X
     ---------------------------------------------------------

  2) Error handling: The function MathError returns the error code from
     the last function evaluation. It must be checked immediately after
     a function call:

       Y := f(X);        (* f is one of the functions of the library *)
       if MathError = FN_OK then ...

     The possible error codes, and the default values attributed to the
     function, are the following:

     ------------------------------------------------------------------
     Error code   Value  Significance            Function default value
     ------------------------------------------------------------------
     FN_OK          0    No error
     DOMAIN        -1    Argument domain error   0
     SING          -2    Function singularity    +/- MAXNUM
     OVERFLOW      -3    Overflow range error    MAXNUM
     UNDERFLOW     -4    Underflow range error   0
     ------------------------------------------------------------------

     where MAXNUM is a constant (declared in MATHCONS.INC) defining the
     highest number which may be represented within the chosen floating
     point type.

     The standard functions (Sqrt, Exp, Ln, Sin, Cos, ArcTan) have been
     redefined according to the above conventions.

  3) With Turbo Pascal (v. 6.0 or higher) and a 387 (or higher)
     coprocessor, it is possible to speed up the calculations by
     defining the symbol CPU387. This will ensure the inclusion of
     the file MATH387.INC which contains assembler routines making
     direct calls to the coprocessor functions.
  ********************************************************************** }

unit FMath;

interface

{ **********************************************************************
  Definition of numeric types according to compiler directives
  ********************************************************************** }

{$IFDEF FPK}
  {$UNDEF PASCALREAL}
  {$UNDEF EXTENDEDREAL}
{$ENDIF}

{$IFDEF PASCALREAL}
  {$IFDEF VER120}
    type Float = Real48;  { Delphi 4 }
  {$ELSE}
    type Float = Real;
  {$ENDIF}
{$ELSE}
{$IFDEF SINGLEREAL}
  type Float = Single;
{$ELSE}
{$IFDEF EXTENDEDREAL}
  type Float = Extended;
{$ELSE}
  {$DEFINE DOUBLEREAL}
  type Float = Double;
{$ENDIF}
{$ENDIF}
{$ENDIF}

{ **********************************************************************
  Include definitions of mathematical constants
  ********************************************************************** }

{$I MATHCONS.INC}

{ **********************************************************************
  Error handling
  ********************************************************************** }

function MathError : Integer;  { Error code from the last function call }

{ **********************************************************************
  Minimum, maximum and exchange of two numbers
  ********************************************************************** }

function FMin(X, Y : Float) : Float;      { Minimum of 2 reals }
function FMax(X, Y : Float) : Float;      { Maximum of 2 reals }
function IMin(X, Y : Integer) : Integer;  { Minimum of 2 integers }
function IMax(X, Y : Integer) : Integer;  { Maximum of 2 integers }

procedure FSwap(var X, Y : Float);        { Exchange 2 reals }
procedure ISwap(var X, Y : Integer);      { Exchange 2 integers }
procedure CSwap(var X, Y : Complex);      { Exchange 2 complex numbers }

{ **********************************************************************
  Sign, logarithms, exponentials and power
  ********************************************************************** }

function Sgn(X : Float) : Integer;                { Sign (returns 1 if X = 0) }
function Sgn0(X : Float) : Integer;               { Sign (returns 0 if X = 0) }
function Exp(X : Float) : Float;                  { Exponential }
function Exp2(X : Float) : Float;                 { 2^X }
function Exp10(X : Float) : Float;                { 10^X }
function Ln(X : Float) : Float;                   { Natural log }
function Log2(X : Float) : Float;                 { Log, base 2 }
function Log10(X : Float) : Float;                { Decimal log }
function LogA(X, A : Float) : Float;              { Log, base A }
function IntPow(X : Float; N : Integer) : Float;  { X^N }
function Pow(X, Y : Float) : Float;               { X^Y, X >= 0 }
function Pythag(X, Y : Float) : Float;            { Sqrt(X^2 + Y^2) }

{ **********************************************************************
  Trigonometric and inverse trigonometric functions
  ********************************************************************** }

function FixAngle(Theta : Float) : Float;  { Set Theta in -Pi..Pi }
function Sin(X : Float) : Float;           { Sinus }
function Cos(X : Float) : Float;           { Cosinus }
function Tan(X : Float) : Float;           { Tangent }
function ArcSin(X : Float) : Float;        { Arc sinus }
function ArcCos(X : Float) : Float;        { Arc cosinus }
function ArcTan(X : Float) : Float;        { Arc tangent }
function ArcTan2(Y, X : Float) : Float;    { Angle (Ox, OM) with M(X,Y) }

procedure SinCos(X : Float; var SinX, CosX : Float);  { Sin & Cos }

{ **********************************************************************
  Hyperbolic and inverse hyperbolic functions
  ********************************************************************** }

function Sinh(X : Float) : Float;     { Hyperbolic sine }
function Cosh(X : Float) : Float;     { Hyperbolic cosine }
function Tanh(X : Float) : Float;     { Hyperbolic tangent }
function ArcSinh(X : Float) : Float;  { Inverse hyperbolic sine }
function ArcCosh(X : Float) : Float;  { Inverse hyperbolic cosine }
function ArcTanh(X : Float) : Float;  { Inverse hyperbolic tangent }

procedure SinhCosh(X : Float; var SinhX, CoshX : Float);  { Sinh & Cosh }

{ **********************************************************************
  Special functions
  ********************************************************************** }

function Fact(N : Integer) : Float;         { Factorial }
function Binomial(N, K : Integer) : Float;  { Binomial coef. C(N,K) }
function Gamma(X : Float) : Float;          { Gamma function }
function SgnGamma(X : Float) : Integer;     { Sign of Gamma function }
function LnGamma(X : Float) : Float;        { Log(|Gamma(X)|) }
function IGamma(A, X : Float) : Float;      { Incomplete Gamma function }
function JGamma(A, X : Float) : Float;      { Complement of IGamma }
function Beta(X, Y : Float) : Float;        { Beta function }
function IBeta(A, B, X : Float) : Float;    { Incomplete Beta function }
function Erf(X : Float) : Float;            { Error function }
function Erfc(X : Float) : Float;           { Complement of Erf }

{ **********************************************************************
  Binomial distribution with probability P and number of repetitions N
  ********************************************************************** }

function PBinom(N : Integer; P : Float; K : Integer) : Float; { Prob(X = K) }
function FBinom(N : Integer; P : Float; K : Integer) : Float; { Prob(X <= K) }

{ **********************************************************************
  Poisson distribution with mean Mu
  ********************************************************************** }

function PPoisson(Mu : Float; K : Integer) : Float;  { Prob(X = K) }
function FPoisson(Mu : Float; K : Integer) : Float;  { Prob(X <= K) }

{ **********************************************************************
  Standard normal distribution
  ********************************************************************** }

function DNorm(X : Float) : Float;    { Density of standard normal }
function FNorm(X : Float) : Float;    { Prob(U <= X) }
function PNorm(X : Float) : Float;    { Prob(|U| >= |X|) }
function InvNorm(P : Float) : Float;  { Inverse of FNorm : returns X
                                        such that Prob(U <= X) = P}

{ **********************************************************************
  Student distribution with Nu d.o.f.
  ********************************************************************** }

function DStudent(Nu : Integer; X : Float) : Float;  { Density of t }
function FStudent(Nu : Integer; X : Float) : Float;  { Prob(t <= X) }
function PStudent(Nu : Integer; X : Float) : Float;  { Prob(|t| >= |X|) }

{ **********************************************************************
  Khi-2 distribution with Nu d.o.f.
  ********************************************************************** }

function DKhi2(Nu : Integer; X : Float) : Float;  { Density of Khi2 }
function FKhi2(Nu : Integer; X : Float) : Float;  { Prob(Khi2 <= X) }
function PKhi2(Nu : Integer; X : Float) : Float;  { Prob(Khi2 >= X) }

{ **********************************************************************
  Fisher-Snedecor distribution with Nu1 and Nu2 d.o.f.
  ********************************************************************** }

function DSnedecor(Nu1, Nu2 : Integer; X : Float) : Float;  { Density of F }
function FSnedecor(Nu1, Nu2 : Integer; X : Float) : Float;  { Prob(F <= X) }
function PSnedecor(Nu1, Nu2 : Integer; X : Float) : Float;  { Prob(F >= X) }

{ **********************************************************************
  Other probability distributions
  ********************************************************************** }

function DExpo(A, X : Float) : Float;      { Density of Exponential distrib.}
function DBeta(A, B, X : Float) : Float;   { Density of Beta distribution }
function DGamma(A, B, X : Float) : Float;  { Density of Gamma distribution }

{ **********************************************************************
  Complex number initialization and conversion
  ********************************************************************** }

procedure CSet(var Z : Complex; A, B : Float; F : ComplexForm);
{ ----------------------------------------------------------------------
  Initializes a complex number according to the form specified by F
    F = Rec  ==>  Z = A + i * B
    F = Pol  ==>  Z = A * Exp(i * B)
  ---------------------------------------------------------------------- }

procedure CConvert(var Z : Complex; F : ComplexForm);
{ Converts the complex number Z to the form specified by F }

{ **********************************************************************
  Complex functions
  ********************************************************************** }

function CReal(Z : Complex) : Float;  { Re(Z) }
function CImag(Z : Complex) : Float;  { Im(Z) }
function CAbs(Z : Complex) : Float;   { |Z| }
function CArg(Z : Complex) : Float;   { Arg(Z) }
function CSgn(Z : Complex) : Integer; { Complex sign }

procedure CNeg(A : Complex; var Z : Complex);      { Z = -A }
procedure CConj(A : Complex; var Z : Complex);     { Z = A* }
procedure CAdd(A, B : Complex; var Z : Complex);   { Z = A + B }
procedure CSub(A, B : Complex; var Z : Complex);   { Z = A - B }
procedure CDiv(A, B : Complex; var Z : Complex);   { Z = A / B }
procedure CMult(A, B : Complex; var Z : Complex);  { Z = A * B }
procedure CLn(A : Complex; var Z : Complex);       { Z = Ln(A) }
procedure CExp(A : Complex; var Z : Complex);      { Z = Exp(A) }
procedure CPow(A, B : Complex; var Z : Complex);   { Z = A^B }

procedure CIntPow(A : Complex; N : Integer; var Z : Complex);  { Z = A^N }
procedure CRealPow(A : Complex; X : Float; var Z : Complex);   { Z = A^X }
procedure CSqrt(A : Complex; var Z : Complex);                 { Z = Sqrt(A) }
procedure CRoot(A : Complex; K, N : Integer; var Z : Complex); { Z = A^(1/N) }

procedure CSin(A : Complex; var Z : Complex);      { Z = Sin(A) }
procedure CCos(A : Complex; var Z : Complex);      { Z = Cos(A) }
procedure CTan(A : Complex; var Z : Complex);      { Z = Tan(A) }

procedure CArcSin(A : Complex; var Z : Complex);   { Z = ArcSin(A) }
procedure CArcCos(A : Complex; var Z : Complex);   { Z = ArcCos(A) }
procedure CArcTan(A : Complex; var Z : Complex);   { Z = ArcTan(A) }

procedure CSinh(A : Complex; var Z : Complex);     { Z = Sinh(A) }
procedure CCosh(A : Complex; var Z : Complex);     { Z = Cosh(A) }
procedure CTanh(A : Complex; var Z : Complex);     { Z = Tanh(A) }

procedure CArcSinh(A : Complex; var Z : Complex);  { Z = ArcSinh(A) }
procedure CArcCosh(A : Complex; var Z : Complex);  { Z = ArcCosh(A) }
procedure CArcTanh(A : Complex; var Z : Complex);  { Z = ArcTanh(A) }

procedure CLnGamma(A : Complex; var Z : Complex);  { Z = Ln(Gamma(A)) }

{ ********************************************************************** }

implementation

{ **********************************************************************
  Include code for mathematical functions (Pascal or Assembler version)
  ********************************************************************** }

{$IFDEF CPU387}
  {$I MATH387.INC}
{$ELSE}
  {$I MATHFUNC.INC}
{$ENDIF}

{ **********************************************************************
  Include code for special functions
  ********************************************************************** }

{$I MATHSPEC.INC}

{ **********************************************************************
  Include code for probability functions
  ********************************************************************** }

{$I PROBA.INC}

{ **********************************************************************
  Include code for complex functions
  ********************************************************************** }

{$I COMPLEX.INC}

begin
  { Initialize the global variables defined in COMMON.INC }
  InitGlobalVar;
end.
