{ **********************************************************************
                            Unit PHARMAC.PAS
                              Version 1.1
                     (c) J. Debord, September 1998
  **********************************************************************
               Pharmacokinetics procedures and functions
       for mamillary compartmental models (up to 3 compartments)
             with elimination from the central compartment
  ********************************************************************** }

unit Pharmac;

interface

uses
  FMath,
  Matrices,
  Polynom,
  Stat,
  Regress;

const
  BOLUS = 0;               { IV bolus }
  INFUS = 1;               { IV infusion }
  EXTRA = 2;               { Extravascular administration }

const
  MAXCOMP = 3;             { Maximum number of compartments }

type
  TAdmin =                 { Drug administration parameters }
    record
      Route : Integer;
      Dose  : Float;
      T_inf : Float;       { Infusion time }
      F     : Float;       { Bioavailability factor }
    end;

  TPharModel =             { Pharmacokinetic model }
    record
      Ncomp    : Integer;  { Nb of compartments }
      Ka       : Float;    { Absorption constant }
      V1       : Float;    { Volume of central compartment }
      Ke       : Float;    { Elimination constant }
      K12, K21 : Float;    { Transfer constants }
      K13, K31 : Float;    { Transfer constants }
      CL       : Float;    { Total plasma clearance }
      Vz       : Float;    { Volume of distribution in terminal phase }
      Vss      : Float;    { Volume of distribution at steady state }
      MRT      : Float;                       { Mean residence time }
      T_half   : array[0..MAXCOMP] of Float;  { Half-lives }
    end;


function PharFit(Admin : TAdmin ; X, Y, W : PVector; N : Integer;
                   var Model : TPharModel) : Integer;
{ --------------------------------------------------------------------
  Approximate fit of a pharmacokinetic model by linear regression
  --------------------------------------------------------------------
  Input parameters : Admin = administration parameters
                     X     = times
                     Y     = concentrations
                     W     = weights
                     N     = number of points
                     Model = pharmacokinetic model (with number of
                             compartments defined)
  ----------------------------------------------------------------------
  Output parameter : Model = pharmacokinetic model (containing the
                             fitted parameters)
  ---------------------------------------------------------------------- }

procedure PharParam(var Model : TPharModel);
{ ----------------------------------------------------------------------
  Computes the pharmacokinetic parameters : CL, Vz, Vss, MRT, T1/2
  ----------------------------------------------------------------------
  Input parameters : Model = model parameters
  ----------------------------------------------------------------------
  Output parameter : Model (updated)
  ---------------------------------------------------------------------- }

function CoefExpo(Admin : TAdmin; Model : TPharModel;
                  A, Alpha : PVector) : Integer;
{ ----------------------------------------------------------------------
  Computes the coefficients (A) and exponents (Alpha) of the
  multi-exponential equation describing the plasma concentration-time
  curve, for a given pharmacokinetic model and a given administration.

                           N
  IV bolus :      C(t) =  Sum  A(j).exp[-Alpha(j).t]
                          j=1

                                 N
  IV infusion :   C(t<T_inf) =  Sum  A(j).(1 - exp[-Alpha(j).t])
                                j=1

                                 N
                  C(t>T_inf) =  Sum  A(j+N).exp[-Alpha(j).(t-T_inf)]
                                j=1

                                A(j+N) = A(j).(1 - exp[-Alpha(j).T_inf])

                           N
  Extravascular : C(t) =  Sum  A(j).exp[-Alpha(j).t]     Alpha(0) = Ka
                          j=0
  ----------------------------------------------------------------------
  Input parameters  : Admin = administration parameters
                      Model = model parameters
  ----------------------------------------------------------------------
  Output parameters : A     = coefficients
                      Alpha = exponents in descending order,
                              with Alpha(0) = Ka
  ----------------------------------------------------------------------
  Possible results :  0 if no error
                     -1 if the exponents cannot be computed
  ---------------------------------------------------------------------- }

function PharConc(Admin : TAdmin; Model : TPharModel; T : Float) : Float;
{ ----------------------------------------------------------------------
  Returns the concentration in the central compartment (plasma conc.) at
  time T, for a given administration and a given pharmacokinetic model.
  ----------------------------------------------------------------------
  Input parameters : Admin = administration parameters
                     Model = model parameters
                     T     = time at which concentration is evaluated
  ---------------------------------------------------------------------- }


implementation

var
  Coef  : PVector;  { Coef. of polynomial used to compute exponents }
  Root  : PVector;  { Roots of polynomial used to compute exponents }
  Alpha : PVector;  { Exponents of multi-exponential equation       }
  A     : PVector;  { Coefficients of multi-exponential equation    }
  A_bar : PVector;  { Coefficients of IV bolus scaled by (V1/Dose)  }


  function ModelParam(Admin : TAdmin; A, Alpha : PVector;
                      var Model : TPharModel) : Integer;
  var
    J       : Integer;
    B       : PVector;  { Save coefficients }
    S       : Float;    { Sum of IV bolus coefficients }
    P       : Float;
    ErrCode : Integer;
  begin
    ErrCode := 0;
    with Admin do
      with Model do
        begin
          DimVector(B, Ncomp);

          { Store actual coefficients in B }
          CopyVector(B, A, 0, Ncomp);
          Ka := Alpha^[0];

          { Convert A to the coefficients of IV bolus }
          case Route of
            INFUS :
              for J := 1 to Ncomp do
                begin
                  P := Alpha^[J] * T_inf;
                  A^[J] := A^[J] * P / (1.0 - Exp(- P));
                end;
            EXTRA :
              for J := 1 to Ncomp do
                A^[J] := A^[J] * (Ka - Alpha^[J]) / (Ka * F);
          end;

          S := 0.0;
          for J := 1 to Ncomp do
            S := S + A^[J];
          V1 := Dose / S;
          case Ncomp of
            1 : Ke := Alpha^[1];
            2 : begin
                  K21 := (A^[1] * Alpha^[2] + A^[2] * Alpha^[1]) / S;
                  Ke := Alpha^[1] * Alpha^[2] / K21;
                  K12 := Alpha^[1] + Alpha^[2] - K21 - Ke;
                end;
            3 : begin
                  Coef^[2] := 1.0;
                  Coef^[1] := - (Alpha^[1] * (A^[2] + A^[3]) +
                                 Alpha^[2] * (A^[1] + A^[3]) +
                                 Alpha^[3] * (A^[1] + A^[2])) / S;
                  Coef^[0] := (Alpha^[1] * Alpha^[2] * A^[3] +
                               Alpha^[1] * Alpha^[3] * A^[2] +
                               Alpha^[2] * Alpha^[3] * A^[1]) / S;
                  if RRootPol(Coef, 2, Root) = 2 then
                    begin
                      K21 := Root^[1];
                      K31 := Root^[2];
                      Ke := Alpha^[1] * Alpha^[2] * Alpha^[3] / K21 / K31;
                      K12 := (Alpha^[1] * Alpha^[2] +
                              Alpha^[2] * Alpha^[3] +
                              Alpha^[1] * Alpha^[3] -
                              K21 * (Alpha^[1] + Alpha^[2] + Alpha^[3]) -
                              Ke * K31 + K21 * K21) / (K31 - K21);
                      K13 := Alpha^[1] + Alpha^[2] + Alpha^[3] -
                             Ke - K12 - K21 - K31;
                    end
                  else
                    ErrCode := - 1;
                end;
          end;

          { Retrieve actual coefficients }
          CopyVector(A, B, 0, Ncomp);
          DelVector(B, Ncomp);
        end;
    ModelParam := ErrCode;
  end;


  function PharFit(Admin : TAdmin ; X, Y, W : PVector; N : Integer;
                   var Model : TPharModel) : Integer;
  { --------------------------------------------------------------------
    Approximate fit of a pharmacokinetic model by linear regression
    Ref. : R. GOMENI & C. GOMENI, Automod : A polyalgorithm for an
           integrated analysis of linear pharmacokinetic models
           Comput. Biol. Med., 1979, 9, 39-48
    -------------------------------------------------------------------- }
  const
    NO_REAL_ROOT = -2;  { No real exponent in multi-exponential curve }
  var
    N1, M   : Integer;  { Number of points and parameters }
    N_exp   : Integer;  { Number of exponentials }
    X1, Y1  : PVector;  { Scaled coordinates }
    W1      : PVector;  { Weights }
    U       : PMatrix;  { Variables for linear regression }
    P       : PVector;  { Linear regression parameters }
    C, Z    : PVector;  { Coefficients and roots of polynomial }
    V       : PMatrix;  { Variance-covariance matrix }
    H       : Float;    { Integration step }
    ErrCode : Integer;  { Error code }
    I, J, K : Integer;  { Loop variables }
  begin
    ErrCode := 0;
    { Find number of exponentials }
    if Admin.Route = EXTRA then
      N_exp := Succ(Model.Ncomp)
    else
      N_exp := Model.Ncomp;

    M := Pred(2 * N_exp);

    DimVector(X1, N);
    DimVector(Y1, N);
    DimVector(W1, N);
    DimMatrix(U, M, N);
    DimMatrix(V, M, M);
    DimVector(P, M);
    DimVector(C, N_exp);
    DimVector(Z, N_exp);

    with Admin do
      if Route = INFUS then
        { Use only the post-infusion part of the curve }
        begin
          J := 0;
          for K := 1 to N do
            if X^[K] >= T_inf then
              begin
                Inc(J);
                X1^[J] := X^[K] - T_inf;
                Y1^[J] := Y^[K];
                W1^[J] := W^[K];
              end;
          N1 := J;
        end
      else
        { Use whole curve }
        begin
          for K := 1 to N do
            begin
              X1^[K] := X^[K];
              Y1^[K] := Y^[K];
              W1^[K] := W^[K];
            end;
          N1 := N;
        end;

    X1^[0] := X1^[1];      { Save first X value }
    if X1^[0] <> 0.0 then  { Change scale so that the X's begin at zero }
      for K := 1 to N1 do
        X1^[K] := X1^[K] - X1^[0];

  { --------------------------------------------------------------------
    Perform linear regression on the linearized form of the equation :

    y = p(0) + p(1) * x + p(2) * x^2 + ... + p(N_exp-1) * x^(N_exp-1)

                    (x                          (x    (x
         + p(N_exp) | y dx + ... + p(2*N_exp-1) | ....| y dx
                    )0                          )0    )0
    -------------------------------------------------------------------- }

    { Compute increasing powers of X }
    if N_exp > 1 then
      for K := 2 to N1 do
        begin
          U^[1]^[K] := X1^[K];
          for I := 2 to Pred(N_exp) do
            U^[I]^[K] := U^[I - 1]^[K] * X1^[K];
        end;

    { Compute integrals by trapezoidal rule }
    for K := 2 to N1 do
      begin
        H := 0.5 * (X1^[K] - X1^[K - 1]);
        U^[N_exp]^[K] := U^[N_exp]^[K - 1] + (Y1^[K] + Y1^[K - 1]) * H;
        for I := Succ(N_exp) to M do
          U^[I]^[K] := U^[I]^[K - 1] + (U^[I - 1]^[K] + U^[I - 1]^[K - 1]) * H;
      end;

    { Fit the linearized equation }
    if WMulFit(U, Y1, W1, N1, M, True, P, V) = MAT_SINGUL then
      ErrCode := MAT_SINGUL
    else
      begin
      { ----------------------------------------------------------------
        The exponents are the real roots of the polynomial :
        x^N_exp + p(N_exp) * x^(N_exp-1) - p(N_exp+1) * x^(N_exp-2) +...
        ---------------------------------------------------------------- }

        { Compute coefficients of polynomial }
        C^[N_exp] := 1.0;
        for I := 1 to N_exp do
          if Odd(I) then
            C^[N_exp - I] := P^[N_exp + I - 1]
          else
            C^[N_exp - I] := - P^[N_exp + I - 1];

        { Solve polynomial }
        if RRootPol(C, N_exp, Z) <> N_exp then
          ErrCode := NO_REAL_ROOT
        else
          begin
            { Sort exponents in decreasing order }
            DQSort(Z, 1, N_exp);

            { Compute coefficients of exponentials by weighted
              linear regression on the exponential terms }
            for I := 1 to N_exp do
              for K := 1 to N1 do
                U^[I]^[K] := Exp(- Z^[I] * X1^[K]);
            if WMulFit(U, Y1, W1, N1, N_exp, False, P, V) = MAT_SINGUL then
              ErrCode := MAT_SINGUL
            else
              begin
                { Correct for scale change if necessary }
                if X1^[0] <> 0.0 then
                  for I := 1 to N_exp do
                    P^[I] := P^[I] * Exp(Z^[I] * X1^[0]);

                { Get coefficients and exponents }
                if Admin.Route = EXTRA then
                  for I := 1 to N_exp do
                    begin
                      A^[I - 1] := P^[I];
                      Alpha^[I - 1] := Z^[I];
                    end
                else
                  for I := 1 to N_exp do
                    begin
                      A^[I] := P^[I];
                      Alpha^[I] := Z^[I];
                    end;
                ErrCode := MAT_OK;
              end;
          end;
      end;

    { Compute model parameters from coefficients and exponents }
    if ErrCode = MAT_OK then
      ErrCode := ModelParam(Admin, A, Alpha, Model);

    PharFit := ErrCode;

    DelVector(X1, N);
    DelVector(Y1, N);
    DelVector(W1, N);
    DelMatrix(U, M, N);
    DelMatrix(V, M, M);
    DelVector(P, M);
    DelVector(C, N_exp);
    DelVector(Z, N_exp);
  end;


  function PharExpo(Model : TPharModel; Alpha : PVector) : Integer;
  { Computes the exponents of the multi-exponential equation }
  var
    J : Integer;
  begin
    with Model do
      begin
        Alpha^[0] := Ka;
        case Ncomp of
          1 : begin
                Alpha^[1] := Ke;
                PharExpo := 0;
                Exit;
              end;
          2 : begin
                Coef^[0] := Ke * K21;
                Coef^[1] := Ke + K12 + K21;
                Coef^[2] := 1.0;
              end;
          3 : begin
                Coef^[0] := Ke * K21 * K31;
                Coef^[1] := K21 * (Ke + K13) + K31 * (Ke + K12 + K21);
                Coef^[2] := Ke + K12 + K21 + K13 + K31;
                Coef^[3] := 1.0;
              end;
        end;

        if RRootPol(Coef, Ncomp, Root) = Ncomp then
          begin
            QSort(Root, 1, Ncomp);
            for J := 1 to Ncomp do
              Alpha^[J] := - Root^[J];
            PharExpo := 0;
          end
        else
          PharExpo := - 1;
      end;
  end;


  procedure UnitCoef(Model : TPharModel; Alpha, A_bar : PVector);
  { Computes the coefficients of IV bolus, scaled by V1/Dose }
  var
    D21, D31, D32 : Float;
  begin
    with Model do
      case Ncomp of
        1 : A_bar^[1] := 1.0;
        2 : begin
              D21 := Alpha^[2] - Alpha^[1];
              A_bar^[1] := (K21 - Alpha^[1]) / D21;
              A_bar^[2] := (Alpha^[2] - K21) / D21;
            end;
        3 : begin
              D21 := Alpha^[2] - Alpha^[1];
              D31 := Alpha^[3] - Alpha^[1];
              D32 := Alpha^[3] - Alpha^[2];
              A_bar^[1] := (K21 - Alpha^[1]) * (K31 - Alpha^[1]) / (D21 * D31);
              A_bar^[2] := (Alpha^[2] - K21) * (K31 - Alpha^[2]) / (D21 * D32);
              A_bar^[3] := (Alpha^[3] - K21) * (Alpha^[3] - K31) / (D31 * D32);
            end;
      end;
  end;


  function CoefExpo(Admin : TAdmin; Model : TPharModel;
                    A, Alpha : PVector) : Integer;
  const
    OldKe  : Float = 0.0;  { Static variables to save transfer constants }
    OldK12 : Float = 0.0;  {   between two function calls                }
    OldK21 : Float = 0.0;
    OldK13 : Float = 0.0;
    OldK31 : Float = 0.0;
  var
    K_Changed : Boolean;  { Checks if transfer constants have changed }
    ErrCode   : Integer;  { Error code }
    J         : Integer;  { Loop variable }
    R         : Float;    { Dose / V1 }
  begin
    with Admin do
      with Model do
        begin
          K_Changed := (Ke <> OldKe) or
          ((Ncomp > 1) and ((K12 <> OldK12) or (K21 <> OldK21))) or
          ((Ncomp > 2) and ((K13 <> OldK13) or (K31 <> OldK31)));

          if K_Changed then
            begin
              ErrCode := PharExpo(Model, Alpha);
              if ErrCode = 0 then
                begin
                  UnitCoef(Model, Alpha, A_bar);
                  OldKe := Ke;
                  OldK12 := K12;
                  OldK21 := K21;
                  OldK13 := K13;
                  OldK31 := K31;
                end;
            end;

          CoefExpo := ErrCode;
          if ErrCode <> 0 then Exit;

          A^[0] := 0.0;
          R := Dose / V1;
          for J := 1 to Ncomp do
            A^[J] := A_bar^[J] * R;
          case Route of
            INFUS : for J := 1 to Ncomp do
                      begin
                        A^[J] := A^[J] / (Alpha^[J] * T_inf);
                        A^[J + Ncomp] := A^[J] * (1.0 - Exp(- Alpha^[J] *
                                                               T_inf));
                      end;
            EXTRA : for J := 1 to Ncomp do
                      begin
                        A^[J] := A^[J] * F * Ka / (Ka - Alpha^[J]);
                        A^[0] := A^[0] - A^[J];
                      end;
          end;
        end;
  end;


  procedure PharParam(var Model : TPharModel);
  var
    I : Integer;
  begin
    with Model do
      begin
        case Ncomp of
          1 : Vss := V1;
          2 : Vss := V1 * (1.0 + K12 / K21);
          3 : Vss := V1 * (1.0 + K12 / K21 + K13 / K31);
        end;
        CL := Ke * V1;
        MRT := Vss / CL;
        if PharExpo(Model, Alpha) = 0 then
          begin
            Vz := CL / Alpha^[Ncomp];
            for I := 0 to Ncomp do
              if Alpha^[I] > 0 then
                T_half[I] := LN2 / Alpha^[I];
          end;
      end;
  end;


  function PharConc(Admin : TAdmin; Model : TPharModel; T : Float) : Float;
  var
    J     : Integer;
    T1, C : Float;
  begin
    C := 0.0;
    if CoefExpo(Admin, Model, A, Alpha) = 0 then
      with Admin do
        with Model do
          case Route of
            BOLUS : for J := 1 to Ncomp do
                      C := C + A^[J] * Exp(- Alpha^[J] * T);
            INFUS : if T <= T_inf then
                      for J := 1 to Ncomp do
                        C := C + A^[J] * (1.0 - Exp(- Alpha^[J] * T))
                    else
                      begin
                        T1 := T - T_inf;
                        for J := 1 to Ncomp do
                          C := C + A^[J + Ncomp] * Exp(- Alpha^[J] * T1);
                      end;
            EXTRA : for J := 0 to Ncomp do
                      C := C + A^[J] * Exp(- Alpha^[J] * T);
          end;
    PharConc := C;
  end;

begin
  DimVector(Coef, MAXCOMP);
  DimVector(Root, MAXCOMP);
  DimVector(Alpha, MAXCOMP);
  DimVector(A, 2 * MAXCOMP);
  DimVector(A_bar, MAXCOMP);
end.

