{ **********************************************************************
                            Program TESTCOEF.PAS
                                Version 1.0
                       (c) J. Debord, November 1996
  **********************************************************************
  This program computes the coefficients and exponents of a multi-
  exponential equation describing the plasma concentration-time curve,
  for a given pharmacokinetic model and a given administration.
  ********************************************************************** }

program TestCoef;

uses
  { Turbo Pascal unit }
  Crt,
  { TpMath units }
  Matrices,
  Pharmac;

const
{ -------------------------------------------
  Example for high dose methotrexate infusion
  ------------------------------------------- }

  Admin : TAdmin =
  (Route : INFUS;
   Dose  : 50;
   T_inf : 0.25;
   F     : 1.0);

  Model : TPharModel =
  { The model is defined by the following parameters }
  (Ncomp  : 3;
   Ka     : 0.0;
   V1     : 16.486492;
   Ke     : 2.888088;
   K12    : 3.940980;
   K21    : 0.416229;
   K13    : 2.965655;
   K31    : 0.033592;
  { Additional parameters will be determined by program }
   CL     : 0.0;
   Vz     : 0.0;
   Vss    : 0.0;
   MRT    : 0.0;
   T_half : (0.0, 0.0, 0.0, 0.0));

var
  A     : PVector;  { Coefficients of multi-exponential equation }
  Alpha : PVector;  { Exponents of multi-exponential equation    }

  procedure WriteResults;
  var
    Lbound : Integer;  { Index of first exponential term   }
                       { = 0 if extravascular, 1 otherwise }
    I      : Integer;  { Loop variable }
  begin
    if Admin.Route = EXTRA then
      Lbound := 0
    else
      Lbound := 1;

    ClrScr;

    with Model do
      begin
        case Admin.Route of
          BOLUS : begin
                    WriteLn('         N');
                    WriteLn('C(t) =  Sum  A(j).exp[-Alpha(j).t]');
                    WriteLn('        j=1');
                  end;
          INFUS : begin
                    WriteLn('               N');
                    WriteLn('C(t<T_inf) =  Sum  A(j).(1 - exp[-Alpha(j).t])');
                    WriteLn('              j=1');
                    WriteLn;
                    WriteLn('               N');
                    WriteLn('C(t>T_inf) =  Sum  A(j+N).exp[-Alpha(j).(t-T_inf)]');
                    WriteLn('              j=1');
                  end;
          EXTRA : begin
                    WriteLn('         N');
                    WriteLn('C(t) =  Sum  A(j).exp[-Alpha(j).t]     Alpha(0) = Ka');
                    WriteLn('        j=0');
                  end;
        end;

        WriteLn('-------------------------------------------------------');

        WriteLn('Coefficients : '#10);

        for I := Lbound to Ncomp do
          begin
            Write('A(', I, ')     = ', A^[I]:12:6);
            if Admin.Route = INFUS then
              Write('         A(', I + Ncomp, ')     = ', A^[I + Ncomp]:12:6);
            WriteLn;
          end;

        WriteLn(#10'Exponents and half-lives :'#10);
        for I := Lbound to Ncomp do
          WriteLn('Alpha(', I, ') = ', Alpha^[I]:12:6,
                  '         T1/2(', I, ')  = ', T_half[I]:12:6);
        WriteLn;
        WriteLn('Total plasma clearance                   = ', CL:12:6);
        WriteLn('Volume of distribution in terminal phase = ', Vz:12:6);
        WriteLn('Volume of distribution at steady state   = ', Vss:12:6);
        WriteLn('Mean residence time (IV bolus)           = ', MRT:12:6);
      end;
  end;

begin
  DimVector(A, 2 * Model.Ncomp);  { Factor 2 is required for IV infusion }
  DimVector(Alpha, Model.Ncomp);

  { Compute coefficients and exponents }
  if CoefExpo(Admin, Model, A, Alpha) = 0 then
    begin
      { Compute additional model parameters }
      PharParam(Model);
      WriteResults;
    end
  else
    WriteLn('Unable to compute coefficients and exponents!');

  DelVector(A, 2 * Model.Ncomp);
  DelVector(Alpha, Model.Ncomp);
end.
