{ **********************************************************************
                              Program WLS.PAS
                                Version 1.1
                      (c) J. Debord, September 1998
  **********************************************************************
  This program fits a pharmacokinetic model by weighted least squares,
  i.e. by minimizing the sum of weighted squared residuals :

                            n
                     SSr = Sum w(k) * [y(k) - ycalc(k)]^2
                           k=1

  where n        = the number of points
        y(k)     = observed drug concentration at point k
        ycalc(k) = estimated drug concentration
        w(k)     = weight of concentration y(k)

  w(k) is taken inversely proportional to the variance v(k) of the
  concentration y(k). v(k) is usually computed as a function of y(k) :

                       v(k) = Vr * g[y(k)] = Vr / w(k)

  where Vr is the residual variance and g is a user-specified function
  (e.g. g[y(k)] = y(k)^2 for a constant coefficient of variation).

  The following parameters are passed on the command line :

    1st parameter = Name of input file (default extension = .DAT)
    2nd parameter = Number of compartments (1 to 3, default = 1)

  Input files are ASCII files with the following structure :

    Line 1 : Title of study
    Line 2 : Number of variables (must be 2 here !)
    Line 3 : Name of variable x
    Line 4 : Name of variable y
    Line 5 : Number of points (must be > number of fitted parameters !)

    The next lines contain the coordinates (x, y) of the points (1 point
    by line). The values of x and y must be separated by spaces or tabs.

    The next line(s) contain the administration parameters :
    - Administration route, coded as 0 = IV bolus
                                     1 = IV infusion
                                     2 = Extravascular
    - Administered dose
    - Duration of infusion
    - Bioavailability factor (0..1)

  The file HDMTX.DAT is an example data file for high dose methotrexate
  infusion. Use it with 2 and 3 compartments to select the best model.

  The program can be run from Turbo Pascal's integrated environment,
  in which case the parameters (e.g. HDMTX 2) are entered through the
  "Parameters" option of the menu, or from DOS (after compilation into
  an executable file), in which case the parameters are entered on the
  command line (e.g. WLS HDMTX 2).
  ********************************************************************** }

program WLS;

uses
  { Turbo Pascal units }
  Crt,
  Graph,
  { TpMath units }
  FMath,
  Matrices,
  Optim,
  Stat,
  PaString,
  Plot,
  Pharmac;

const
  MAXITER = 500;     { Number of iterations allowed to Simplex }
  TOL     = 1.0E-3;  { Required precision for Simplex estimation }

var
  InFName : String;      { Name of input file }
  Title   : String;      { Title of study }
  XName,
  YName   : String;      { Variable names }
  N       : Integer;     { Number of points }
  X, Y    : PVector;     { Point coordinates }
  Ycalc   : PVector;     { Expected Y values }
  W       : PVector;     { Weights }
  Lbound,
  Ubound  : Integer;     { Indices of first & last fitted parameters }
  B       : PVector;     { Regression parameters }
  SSr     : Float;       { Residual sum of squares }
  ErrCode : Integer;     { Error code }
  Admin   : TAdmin;      { Drug administration parameters }
  Model   : TPharModel;  { Pharmacokinetic model }

{ **********************************************************************
  Define here the function used to compute the variance of an observed
  y value. The true variance will be : V(y) = Vr * VarFunc(y), where Vr
  is the residual variance (estimated by the program).
  For unweighted regression, simply set VarFunc to 1.0
  ********************************************************************** }

  function VarFunc(Y : Float) : Float;
  begin
    VarFunc := Sqr(Y);  { For a constant coefficient of variation }
  end;

  procedure ReadCmdLine;
  { Reads command line parameters }
  var
    I, N, ErrCode : Integer;
  begin
    { Name of input file }
    InFName := ParamStr(1);
    if Pos('.', InFName) = 0 then InFName := InFName + '.DAT';
    { Number of compartments }
    N := 0;
    Val(ParamStr(2), N, ErrCode);
    if (ErrCode <> 0) or (N < 1) then N := 1;
    if N > MAXCOMP then N := MAXCOMP;
    Model.Ncomp := N;
  end;

  function ParamName(I : Integer) : String;
  { Returns the name of the I-th parameter }
  begin
    case I of
      0 : ParamName := 'Ka ';
      1 : ParamName := 'V1 ';
      2 : ParamName := 'Ke ';
      3 : ParamName := 'K12';
      4 : ParamName := 'K21';
      5 : ParamName := 'K13';
      6 : ParamName := 'K31';
    end;
  end;

  function RegFunc(X : Float; B : PVector) : Float;
  { Computes the regression function }
  begin
    with Model do
      { Copy regression parameters into model parameters }
      begin
        Ka := B^[0];
        V1 := B^[1];
        Ke := B^[2];
        if Ncomp > 1 then
          begin
            K12 := B^[3];
            K21 := B^[4];
          end;
        if Ncomp > 2 then
          begin
            K13 := B^[5];
            K31 := B^[6];
          end;
      end;
    { Compute concentration at time X }
    RegFunc := PharConc(Admin, Model, X);
  end;

{ **********************************************************************
  The next routine defines the objective function to be minimized. It
  must be compiled in FAR mode ($F+). Don't modify the parameter list
  or function type : they must match the definitions in OPTIM.PAS
  ********************************************************************** }

  {$F+}
  function WLS_ObjFunc(B : PVector) : Float;
  var
    K : Integer;
  begin
    for K := 1 to N do
      Ycalc^[K] := RegFunc(X^[K], B);
    WLS_ObjFunc := SumWSqrDifVect(Y, Ycalc, W, 1, N);
  end;
  {$F-}

{ ********************************************************************
       Routines to read data, perform fit, write and plot results
  ******************************************************************** }

  procedure ReadInputFile;
  { Reads input file and dimensions data arrays }
  var
    InF  : Text;     { Input file }
    Nvar : Integer;  { Number of variables }
    I    : Integer;  { Loop variable }
  begin
    ErrCode := 0;
    Assign(InF, InFName);
    Reset(InF);

    ReadLn(InF, Title);
    ReadLn(InF, Nvar);

    if Nvar <> 2 then
      begin
        WriteLn('Data file must contain 2 variables!');
        ErrCode := - 1;
        Exit;
      end;

    ReadLn(InF, XName);
    ReadLn(InF, YName);
    ReadLn(InF, N);

    DimVector(X, N);
    DimVector(Y, N);
    DimVector(Ycalc, N);
    DimVector(W, N);

    for I := 1 to N do
      Read(InF, X^[I], Y^[I]);

    with Admin do
      Read(InF, Route, Dose, T_inf, F);
    Close(InF);
  end;

  procedure FitModel;
  var
    K : Integer;
  begin
    { Dimension arrays }
    if Admin.Route = EXTRA then
      Lbound := 0
    else
      Lbound := 1;
    Ubound := 2 * Model.Ncomp;
    DimVector(B, Ubound);

    { Compute weights }
    for K := 1 to N do
      W^[K] := 1.0 / VarFunc(Y^[K]);

    { Compute initial parameter estimates and stop if estimation fails }
    ErrCode := PharFit(Admin, X, Y, W, N, Model);
    if ErrCode <> MAT_OK then
      begin
        WriteLn('Unable to compute initial parameter estimates!');
        Exit;
      end;

    { Copy model parameters into regression parameters }
    with Model do
      begin
        B^[0] := Ka;
        B^[1] := V1;
        B^[2] := Ke;
        if Ncomp > 1 then
          begin
            B^[3] := K12;
            B^[4] := K21;
          end;
        if Ncomp > 2 then
          begin
            B^[5] := K13;
            B^[6] := K31;
          end;
      end;

    { Perform minimization (Simplex method) }
    ErrCode := Simplex(WLS_ObjFunc, B, Lbound, Ubound, MAXITER, TOL, SSr);

    if ErrCode <> MAT_OK then
      WriteLn('Unable to refine parameters!');
  end;

  procedure WriteOutputFile;
  var
    OutFName : String;   { Name of output file }
    OutF     : Text;     { Output file }
    Line1,
    Line2    : String;   { Separating lines }
    Ybar     : Float;    { Average Y value }
    Delta    : Float;    { Y - Ycalc }
    SSt      : Float;    { Total sum of squares }
    SSe      : Float;    { Explained sum of squares }
    Vr       : Float;    { Residual variance }
    Sr       : Float;    { Residual error }
    R2       : Float;    { Coefficient of determination }
    F        : Float;    { Variance ratio }
    AIC      : Float;    { Akaike information criterion }
    P        : Float;    { Probability }
    SY       : Float;    { Standard deviation of observation }
    Npar     : Integer;  { Number of parameters }
    Nu1, Nu2 : Integer;  { Degrees of freedom }
    I, K     : Integer;  { Loop variables }
  begin
    K := Pos('.', InFName);
    if K = 0 then
      OutFName := InFName + '.OUT'
    else
      OutFName := Copy(InFName, 1, Pred(K)) + '.OUT';
    Assign(OutF, OutFName);
    Rewrite(OutF);

    Line1 := StrChar(73, '-');
    Line2 := StrChar(73, '=');

    WriteLn(OutF, Line2);
    WriteLn(OutF, 'WLS : Weighted Least Squares Analysis of Pharmacokinetic Data');
    WriteLn(OutF, Line2);

    WriteLn(OutF, 'Data file  : ', InFName);
    WriteLn(OutF, 'Study name : ', Title);
    WriteLn(OutF, 'x variable : ', XName);
    WriteLn(OutF, 'y variable : ', YName);
    Write(OutF, 'Model      : ');
    case Admin.Route of
      BOLUS : Write(OutF, 'IV bolus');
      INFUS : Write(OutF, 'IV infusion');
      EXTRA : Write(OutF, 'Extravascular administration');
    end;
    Write(OutF, ' - ', Model.Ncomp, ' compartment');
    if Model.Ncomp > 1 then Write(OutF, 's');
    WriteLn(OutF);

    Ybar := Average(Y, 1, N);
    SSt := SumWSqrDif(Y, W, 1, N, Ybar);
    SSe := SumWSqrDif(Ycalc, W, 1, N, Ybar);
    Npar := Ubound - Lbound + 1;
    Nu1 := Npar - 1;
    Nu2 := N - Npar;
    Vr := SSr / Nu2;
    Sr := Sqrt(Vr);
    R2 := 1.0 - SSr / SSt;
    F := (SSe / Nu1) / Vr;
    P := PSnedecor(Nu1, Nu2, F);
    AIC := N * Ln(SSr) + 2 * Npar;

    WriteLn(OutF, Line1);
    WriteLn(OutF, 'Parameter    Estimated value');
    WriteLn(OutF, Line1);

    for I := Lbound to Ubound do
      WriteLn(OutF, ParamName(I), B^[I]:25:6);

    WriteLn(OutF, Line1);
    WriteLn(OutF, 'Number of points              : n   = ', N:9);
    WriteLn(OutF, 'Residual error                : s   = ', Sr:14:4);
    WriteLn(OutF, 'Coefficient of determination  : r2  = ', R2:14:4);
    WriteLn(OutF, 'Variance ratio (expl./resid.) : F   = ', F:14:4,
                  '    Prob(>F) = ', P:6:4);
    WriteLn(OutF, 'Akaike information criterion  : AIC = ', AIC:14:4);
    WriteLn(OutF, Line1);

    with Model do
      begin
        WriteLn(OutF, 'Total plasma clearance                   = ', CL:12:6);
        WriteLn(OutF, 'Volume of distribution in terminal phase = ', Vz:12:6);
        WriteLn(OutF, 'Volume of distribution at steady state   = ', Vss:12:6);
        WriteLn(OutF, 'Mean residence time (IV bolus)           = ', MRT:12:6);
        for I := Lbound to Ncomp do
          WriteLn(OutF, 'Half-life (', I, ')                            = ',
              T_half[I]:12:6);
      end;

    WriteLn(OutF, Line1);
    WriteLn(OutF, '  i        Y obs.       Y calc.      Residual      Std.dev.      Std.res.');
    WriteLn(OutF, Line1);

    for K := 1 to N do
      begin
        Delta := Y^[K] - Ycalc^[K];
        SY := Sr / Sqrt(W^[K]);
        WriteLn(OutF, K:3, Y^[K]:14:4, Ycalc^[K]:14:4, Delta:14:4,
                           SY:14:4, (Delta / SY):14:4);
      end;
    WriteLn(OutF, Line2);
    Close(OutF);
    WriteLn('Results written to file ', OutFName);
  end;

  {$F+}
  function PlotRegFunc(X : Float) : Float;
  { Defines the function to be plotted }
  begin
    PlotRegFunc := RegFunc(X, B);
  end;
  {$F-}

  procedure PlotGraph;
  var
    Ch : Char;
  begin
    { Define here the path to the graphic drivers }
    BGIPath := 'C:\BP\BGI';

    { Determine scale automatically }
    AutoScale(X, 1, N, XAxis);
    AutoScale(Y, 1, N, YAxis);

    { Set graph titles }
    GraphTitle.Text := Title;
    XTitle.Text := XName;
    YTitle.Text := YName;

    { Plot graph }
    if GraphOk then
      begin
        WriteLegend(0);
        SetClipping(True);

        { Plot points }
        PlotCurve(X, Y, 1, N, 1, 0);

        { Plot fitted function }
        PlotFunc(PlotRegFunc, XAxis.Min, XAxis.Max, 1);

        { Leave graphic mode }
        Ch := ReadKey;
        CloseGraph;
      end;
  end;

{ **********************************************************************
                                Main program
  ********************************************************************** }

begin
  ReadCmdLine;
  ReadInputFile;
  if ErrCode = 0 then
    FitModel;
  if ErrCode = MAT_OK then
    begin
      PharParam(Model);
      PlotGraph;
      WriteOutputFile;
    end;
end.
