library RanDevs;

{ This file compiles to a DLL that can be called from Microsoft Excel. Some of
  the code was written by me, but many of the functions call Pascal code in Jean
  DeBord's TPMath library (http://ourworld.compuserve.com/homepages/JDebord/).
  The file was developed using Borland Delphi 4.0 (update pack 3) and uses the
  ComObj unit. I think ComObj arrived with Delphi 4.0 and hence the code will
  not compile with earlier Delphi versions.

  The routines greatly extend the utility of Excel solving for numerical
  problems. I use them for demographic analyses and have incorporated them into
  an Excel add-in package called PopTools. This add-in allows Excel to be used
  analysis of population projection matrices, simulation of stochastic processes
  and calculation of bootstrap (resampling) statistics.

  I found it difficult to find information on how to interface Visual Basic for
  Applications and DLL's. The file is provided to the public domain in the hope
  that someone who knows more about VBA, Delphi and COM objects can improve the
  code.

  The headers in the PopTools VBA file follow ........... Greg Hood October 1999
                                                          greg.hood@dwe.csiro.au

  Declare Function dNormalInt Lib "RanDevs.dll" (ByVal X As Double, ByVal Y As Double) As Integer
  Declare Function dNormalDev Lib "RanDevs.dll" (ByVal X As Double, ByVal Y As Double) As Double
  Declare Function dRandReal Lib "RanDevs.dll" (ByVal X As Double, ByVal Y As Double) As Double
  Declare Function dRandInt Lib "RanDevs.dll" (ByVal X As Double, ByVal Y As Double) As Integer
  Declare Function dExpDev Lib "RanDevs.dll" (ByVal X As Double) As Double
  Declare Function dGamma Lib "RanDevs.dll" (ByVal X As Double) As Double
  Declare Function dGeomDev Lib "RanDevs.dll" (ByVal X As Double) As Integer
  Declare Function dNegBinomial Lib "RanDevs.dll" (ByVal N As Double, _
                                                   ByVal mu As Double, _
                                                   ByVal K As Double) As Double
  Declare Function dNegBinomialDev Lib "RanDevs.dll" (ByVal mu As Double, _
                                                   ByVal K As Double) As Integer
  Declare Function dBinomialDev Lib "RanDevs.dll" (ByVal N As Double, ByVal p As Double) As Integer
  Declare Function dLogNormalDev Lib "RanDevs.dll" (ByVal X As Double, ByVal Y As Double) As Double
  Declare Function dGammaDev Lib "RanDevs.dll" (ByVal X As Double) As Double
  Declare Function dPoissonDev Lib "RanDevs.dll" (ByVal X As Double) As Integer
  Declare Function dEigVal Lib "RanDevs.dll" (ByRef Matrix As Variant) As Variant
  Declare Function dDomEig Lib "RanDevs.dll" (ByRef Matrix As Variant) As Double
  Declare Function dEigVect Lib "RanDevs.dll" (ByRef Matrix As Variant) As Variant
  Declare Function dPCA Lib "RanDevs.dll" (ByRef DataMatrix As Variant) As Variant
  Declare Function dSVD Lib "RanDevs.dll" (ByRef Matrix As Variant) As Variant
  Declare Function dLife Lib "RanDevs.dll" (ByRef Matrix As Variant) As Variant
  Declare Function dCholesky Lib "RanDevs.dll" (ByRef Matrix As Variant) As Variant
  Declare Function dLUDecomp Lib "RanDevs.dll" (ByRef Matrix As Variant) As Variant
  Declare Function dQR Lib "RanDevs.dll" (ByRef Matrix As Variant) As Variant
  Declare Function dRoots Lib "RanDevs.dll" (ByRef Matrix As Variant) As Variant
  Declare Function dQuickSort Lib "RanDevs.dll" (ByRef Vector As Variant, ByVal Descending As Boolean) As Variant
  Declare Function dRowMatrixMult Lib "RanDevs.dll" (ByRef Matrix As Variant, ByRef Vector As Variant) As Variant
  Declare Function dMtoV Lib "RanDevs.dll" (ByRef Matrix As Variant) As Variant
  Declare Function dACF Lib "RanDevs.dll" (ByRef Vector As Variant) As Variant
}

uses
  SysUtils, Classes, Stat, ComObj,
  FMath, Matrices, Eigen, Regress, Windows;

const
  DIM_ERROR             = 7777;         { Dimensions do not match expectation }
  DLL_MEM_ERROR         = 9999;         { Error in this DLL due to memory allocation }
  MIN_NUMBER            = -1E100;       { A really negative number }
  MAXITER               = 100;          { Maximum number of iterations }
  TOL                   = 1.0E-8;       { Tolerance on the eigenvector }
  TwoPi                 = 6.28318530717959;
  Sqrt2Pi               = 2.506628274631001;

function Divide(A, B: double): double;
{ Divides A by B, but returns 0 if B = 0 }
begin
  if B <> 0 then
    result := A / B
  else
    result := 0;
end;

function dMinD(A, B: double): double;
begin
  if A < B then result := A else result := B;
end;

function GetExcelData(var V: Variant;
  var A: PMatrix;
  var Rows, Cols: integer): boolean;
{ Takes an Excel range object and puts the data into
  a PMatrix. Uses Delphi ComObj unit. Can also take an
  array of double and return data in a Pmatrix. Functions
  of RanDevs.dpr can be called from within VBA code as
  follows:

  Dim A(10,10) as double
  ...
  ...
  Result = RanDevsFunction(A)
  }
var
  NewV                  : Variant;
  i, j                  : integer;
  NoErrors              : boolean;
begin
  {Check that we have a COM object from Excel
   This should be an Excel range object }
  NoErrors := TRUE;
  if VarType(V) = VarDispatch then
  begin
    try
      Rows := V.Rows.Count;
      Cols := V.Columns.Count;
      DimMatrix(A, Rows, Cols);
      NewV := V.Value;
      for i := 1 to Rows do for j := 1 to Cols do
          if varType(NewV[i, j]) = varDouble then
            A^[i]^[j] := NewV[i, j]
          else
            NoErrors := FALSE;
    except on E: Exception do
        NoErrors := FALSE;
    end
  end
  else
  begin
    { We have an Excel value array rather than a range }
    try
      Rows := VarArrayHighBound(V, 1);
      Cols := VarArrayHighBound(V, 2);
      DimMatrix(A, Rows, Cols);
      for i := 1 to Rows do for j := 1 to Cols do
          if varType(V[i, j]) = varDouble then
            A^[i]^[j] := V[i, j]
          else
            NoErrors := FALSE;
    except on E: Exception do
        NoErrors := FALSE;
    end
  end;
  result := NoErrors;
end;

function GetExcelVector(var V: Variant;
  var A: PVector;
  var Rows, Cols: integer): boolean;
var
  NewV                  : Variant;
  i, j                  : integer;
  NoErrors              : boolean;
begin
  { Check that we have a COM object from Excel  }
  NoErrors := TRUE;                     { This should be an Excel range object }
  if VarType(V) = VarDispatch then
  try
    Rows := V.Rows.Count;
    Cols := V.Columns.Count;
    NewV := V.Value;
  except on E: Exception do
      NoErrors := FALSE;
  end
  else                                  { Alternatively we have a value array }
  try
    Rows := VarArrayHighBound(V, 1);
    Cols := VarArrayHighBound(V, 2);
    NewV := V;
  except on E: Exception do
      NoErrors := FALSE;
  end;

  if Rows > Cols then
  begin
    DimVector(A, Rows);
    for i := 1 to Rows do
      if varType(NewV[i, 1]) = varDouble then
        A^[i] := NewV[i, 1]
      else
        NoErrors := FALSE;
  end
  else                                  { Cols>Rows }
  begin
    DimVector(A, Cols);
    for j := 1 to Cols do
      if varType(NewV[1, j]) = varDouble then
        A^[j] := NewV[1, j]
      else
        NoErrors := FALSE;
  end;

  result := NoErrors;
end;

procedure WriteError(var V: Variant; ErrCode: integer);
var
  i, j                  : integer;
begin
  for i := VarArrayLowBound(V, 1) to VarArrayHighBound(V, 1) do
    for j := VarArrayLowBound(V, 2) to VarArrayHighBound(V, 2) do
      V[i, j] := ErrCode;
end;

procedure WriteVectorError(var V: Variant; ErrCode: integer);
var
  i                     : integer;
begin
  for i := VarArrayLowBound(V, 1) to VarArrayHighBound(V, 1) do
    V[i] := ErrCode;
end;

procedure FillPMatrix(A: PMatrix; Valu: double; Rows, Cols: integer);
var
  i, j                  : integer;
begin
  for i := 1 to Rows do
    for j := 1 to Cols do
      A^[I]^[J] := Valu;
end;

procedure VToPMatrix(V: Variant; A: PMatrix; Rows, Cols: integer);
var
  i, j                  : integer;
begin
  for i := 1 to Rows do
    for j := 1 to Cols do
      A^[I]^[J] := V[I, J];
end;

procedure VToPVector(V: Variant; A: PVector; Rows: integer);
var
  i                     : integer;
begin
  for i := 1 to Rows do
    A^[I] := V[I, 1];
end;

procedure PMatrixToV(A: PMatrix; var V: Variant; Rows, Cols: integer);
var
  i, j                  : integer;
begin
  for i := 1 to Rows do
    for j := 1 to Cols do
      V[I, J] := A^[I]^[J]
end;

procedure PVectorToV(A: PVector; var V: Variant; Rows: integer);
var
  i                     : integer;
begin
  for i := 1 to Rows do
    V[I, 1] := A^[I]
end;

function RealMax(Lambda_Re, Lambda_Im: PVector; N: integer): double;
var
  i                     : integer;
  temp                  : double;
begin
  temp := MIN_NUMBER;
  for i := 1 to N do
    if Lambda_re^[i] > temp then if Lambda_im^[i] = 0 then
        temp := Lambda_re^[i];
  result := temp
end;

function dACF(var V: Variant): Variant; stdcall;
{Returns the autocorrelation function. Algorithm adapted from
Box, Jenkins & Reinsel 1994 (the bible of time series analysis}
var
  counter               : integer;
  K, lag                : integer;
  num, numerator,
    denom1, denom2,
    denominator,
    x1, x2, Mean, Total : double;
  N                     : integer;
  A                     : PVector;
  rows, cols            : integer;
  R                     : Variant;

begin
  if GetExcelVector(V, A, rows, cols) then
  try
    if rows > cols then N := rows else N := Cols;
    {K := N div 4; recommended by Box-Jenkins-Reinsel 1994}
    K := N div 4;
    Total := 0;
    for counter := 1 to N do
      Total := Total + A^[counter];
    Mean := Total / N;
    if rows > cols then
    begin
      R := VarArrayCreate([0, N, 1, 1], varDouble);
      R[0, 1] := 1;
    end
    else
    begin
      R := VarArrayCreate([1, 1, 0, N], varDouble);
      R[1, 0] := 1;
    end;
    for lag := 1 to K do
    begin
      numerator := 0;
      denominator := 0;
      for counter := 1 to N do
      begin
        if counter <= (N - lag) then x1 := A^[counter + lag] - Mean else x1 := 0;
        x2 := A^[counter] - Mean;
        numerator := numerator + x1 * x2;
        denominator := denominator + x2 * x2;
      end;
      if rows > cols then R[lag, 1] := divide(numerator, denominator)
      else R[1, lag] := divide(numerator, denominator);
    end;
    Result := R;
  except
    on E: Exception do
      WriteError(R, DLL_MEM_ERROR)
  end;
end;

procedure eigsrt(Lambda_re, Lambda_im: PVector; v: PMatrix; n: integer);
var
  k, j, i               : integer;
  p, PC                 : double;
begin
  for i := 1 to n - 1 do begin
    k := i;
    p := Lambda_re^[i];
    for j := i + 1 to n do begin
      if (Lambda_re^[j] >= p) then begin
        k := j;
        p := Lambda_re^[j];
        PC := Lambda_im^[j]
      end
    end;
    if (k <> i) then begin
      Lambda_re^[k] := Lambda_re^[i];
      Lambda_re^[i] := p;
      Lambda_im^[k] := Lambda_im^[i];
      Lambda_im^[i] := PC;
      for j := 0 to n do begin
        p := v^[i, j];
        v^[i, j] := v^[k, j];
        v^[k, j] := p
      end
    end
  end
end;

procedure EigValSort(Lambda_re, Lambda_im: PVector; n: integer);
var
  k, j, i               : integer;
  p, PC                 : double;
begin
  for i := 1 to n - 1 do begin
    k := i;
    p := Lambda_re^[i];
    PC := Lambda_im^[i];
    for j := i + 1 to n do begin
      if (Lambda_re^[j] >= p) then begin
        k := j;
        p := Lambda_re^[j];
        PC := Lambda_im^[j]
      end
    end;
    if (k <> i) then begin
      Lambda_re^[k] := Lambda_re^[i];
      Lambda_re^[i] := p;
      Lambda_im^[k] := Lambda_im^[i];
      Lambda_im^[i] := PC;
    end
  end
end;

function dEigVal(var V: Variant): Variant; stdcall;
{ Calls TPMath: All eigenvalues of V }
var
  A                     : PMatrix;
  R                     : Variant;
  Lambda_Re, Lambda_Im  : PVector;
  Dimension, i, j       : integer;
begin
  if GetExcelData(V, A, Dimension, Dimension) then
  try
    { Get memory for TPMath routines }
    DimVector(Lambda_Re, Dimension);
    DimVector(Lambda_Im, Dimension);
    { Create array to hold results }
    R := VarArrayCreate([1, Dimension, 1, 2], varDouble);
    try
      //Get the real and complex eigenvalues
      if EigenVals(A, 1, Dimension, Lambda_Re, Lambda_Im) = MAT_OK then
      begin
        EigValSort(Lambda_Re, Lambda_Im, Dimension);
        for i := 1 to Dimension do
        begin
          //Real part in left column, imaginary in right
          R[i, 1] := Lambda_Re^[i];
          R[i, 2] := Lambda_Im^[i];
        end;
      end
    except
      on E: Exception do
        WriteError(R, DLL_MEM_ERROR);
    end;
  finally
    Result := R;
    DelMatrix(A, Dimension, Dimension);
    DelVector(Lambda_Re, Dimension);
    DelVector(Lambda_Im, Dimension);
  end
  else
    result := DLL_MEM_ERROR;
end;

function dMtoV(var V: Variant): Variant; stdcall;
var
  i, j                  : integer;
  A                     : PMatrix;
  R                     : Variant;
  Rows, Cols            : integer;
begin
  try
    if GetExcelData(V, A, Rows, Cols) then
      R := VarArrayCreate([1, Rows * Cols, 1, 1], varDouble);
    for i := 1 to Rows do
      for j := 1 to Cols do
        R[(i - 1) * Cols + j, 1] := A^[i]^[j]
  except
    on E: Exception do
      WriteError(R, DLL_MEM_ERROR)
  end;
  Result := R;
  DelMatrix(A, Rows, Rows);
end;

function dCholesky(var V: Variant): Variant; stdcall;
{ Calls TPMath: Cholesky decomposition }
var
  A, L                  : PMatrix;
  R                     : Variant;
  Rows, Cols            : integer;
begin
  if GetExcelData(V, A, Rows, Cols) then
    if Rows = Cols then
    try
      //Get memory for TPMath routines
      DimMatrix(L, Rows, Rows);
      //Create array to hold results
      R := VarArrayCreate([1, Rows, 1, Rows], varDouble);
      try
        case Cholesky(A, 1, Rows, L) of
          MAT_OK: PMatrixToV(L, R, Rows, Rows);
          MAT_NOT_PD: WriteError(R, MAT_NOT_PD)
        end;                            //case
      except
        on E: Exception do
          WriteError(R, DLL_MEM_ERROR)
      end;
    finally
      Result := R;
      DelMatrix(A, Rows, Rows);
      DelMatrix(L, Rows, Rows);
    end
    else
      result := DIM_ERROR
  else
    result := DLL_MEM_ERROR;
end;

function dLUDecomp(var V: Variant): Variant; stdcall;
{ Calls TPMath: LU decomposition of V }
var
  A                     : PMatrix;
  R                     : Variant;
  Rows, Cols            : integer;
begin
  if GetExcelData(V, A, Rows, Cols) then
    if Rows = Cols then
    try
      //Create array to hold results
      R := VarArrayCreate([1, Rows, 1, Rows], varDouble);
      try
        case LU_Decomp(A, 1, Rows) of
          MAT_OK: PMatrixToV(A, R, Rows, Rows);
          MAT_NOT_PD: WriteError(R, MAT_NOT_PD)
        end;                            //case
      except
        on E: Exception do
          WriteError(R, DLL_MEM_ERROR)
      end;
    finally
      Result := R;
      DelMatrix(A, Rows, Rows);
    end
    else
      result := DIM_ERROR
  else
    result := DLL_MEM_ERROR;
end;

function dRowMatrixMult(var M: Variant; var V: Variant): Variant; stdcall;
//Matrix pre-multiplication of a vector V by a square matrix M
//that is represented as a row in an Excel sheet. The vector
//must also be a column vector represented as a row in Excel
var
  PM, PV                : PMatrix;
  R                     : Variant;
  Rows, Cols, i, j      : integer;
  summ                  : double;
begin
  try
    if GetExcelData(M, PM, Rows, Cols) then
      if GetExcelData(V, PV, Rows, Cols) then //Vector defines dimension
      try
        R := VarArrayCreate([1, 1, 1, Cols], varDouble);
        for i := 1 to Cols do
        begin
          summ := 0;
          for j := 1 to Cols do
            summ := summ + PM^[1]^[(i - 1) * Cols + j] * PV^[1]^[j];
          R[1, i] := summ
        end;
      except
        on E: Exception do
          WriteError(R, DLL_MEM_ERROR)
      end
      else
        result := DLL_MEM_ERROR
    else
      result := DLL_MEM_ERROR;
  finally
    Result := R;
  end
end;

function dQuickSort(var V: Variant; Descending: boolean): Variant; stdcall;
{ Calls TPMath: Sort of V }
var
  A                     : PMatrix;
  Col1                  : PVector;
  R                     : Variant;
  Rows, Cols, i, j      : integer;
begin
  if GetExcelData(V, A, Rows, Cols) then
    if Rows = 1 then
    try
      //Create array to hold results
      R := VarArrayCreate([1, 1, 1, Cols], varDouble);
      try
        if Descending then
          DQSort(A^[1], 1, Cols)
        else
          QSort(A^[1], 1, Cols);
        for j := 1 to Cols do R[1, j] := A^[1]^[j];
      except
        on E: Exception do
          WriteError(R, DLL_MEM_ERROR)
      end;
    finally
      Result := R;
      DelMatrix(A, Rows, Cols);
    end
    else if Cols = 1 then
    try
      //Create array to hold results
      R := VarArrayCreate([1, Rows, 1, 1], varDouble);
      DimVector(Col1, Rows);
      for i := 1 to Rows do Col1^[i] := A^[i]^[1];
      try
        if Descending then
          DQSort(Col1, 1, Rows)
        else
          QSort(Col1, 1, Rows);
        for i := 1 to Rows do R[i, 1] := Col1^[i];
      except
        on E: Exception do
          WriteError(R, DLL_MEM_ERROR)
      end;
    finally
      Result := R;
      DelVector(Col1, Rows);
      DelMatrix(A, Rows, Cols);
    end
    else
      result := DIM_ERROR
  else
    result := 222;
end;

function dDomEig(var V: Variant): double; stdcall;
{ Calls TPMath: Dominant eigenvalue of matrix V}
var
  A                     : PMatrix;
  Lambda_Re, Lambda_Im  : PVector;
  Dimension, i, j       : integer;
begin
  if GetExcelData(V, A, Dimension, Dimension) then
  try
    //Get memory for TPMath routines
    DimVector(Lambda_Re, Dimension);
    DimVector(Lambda_Im, Dimension);
    try
      //Get the real and complex eigenvalues
      if EigenVals(A, 1, Dimension, Lambda_Re, Lambda_Im) = MAT_OK then
        result := RealMax(Lambda_Re, Lambda_Im, Dimension)
    except
      on E: Exception do
        result := DLL_MEM_ERROR;
    end;
  finally
    DelMatrix(A, Dimension, Dimension);
    DelVector(Lambda_Re, Dimension);
    DelVector(Lambda_Im, Dimension);
  end
  else
    result := DLL_MEM_ERROR;
end;

function dLife(var V: Variant): Variant; stdcall;
{ Calculates an eigenalysis of a population projection matrix V to determine the
  rate of increase, age structure and reproductive value. }
var
  A, ACopy, A_t         : PMatrix;
  R                     : Variant;
  Lambda_Re, Lambda_Im  : PVector;
  EigenVector           : PVector;
  Dimension, i, j       : integer;
  temp, Summ            : Float;
begin
  if GetExcelData(V, A, Dimension, Dimension) then
  try
    //Get memory for TPMath routines
    DimMatrix(ACopy, Dimension, Dimension);
    DimMatrix(A_t, Dimension, Dimension);
    DimVector(Lambda_Re, Dimension);
    DimVector(Lambda_Im, Dimension);
    DimVector(EigenVector, Dimension);
    R := VarArrayCreate([1, Dimension * 2 + 1, 1, 1], varDouble);
    try
      //Copy the values of V to a PMatrix
      //VToPMatrix(V, A, Dimension, Dimension);
      CopyMatrix(ACopy, A, 1, 1, Dimension, Dimension);
      Transpose(A, 1, 1, Dimension, Dimension, A_t);
      //Get the real and complex eigenvalues
      if EigenVals(A, 1, Dimension, Lambda_Re, Lambda_Im) = MAT_OK then
      begin
        temp := RealMax(Lambda_Re, Lambda_Im, Dimension);
        R[1, 1] := temp;
        //Eigenvectors of dominant eigenvalue
        for i := 1 to Dimension do
          EigenVector^[i] := 1;
        if EigenVect(ACopy, 1, Dimension, MAXITER, TOL, EigenVector, temp) = MAT_OK then
        begin
          R[1, 1] := temp;
          Summ := Sum(EigenVector, 1, Dimension);
          for i := 1 to Dimension do
            R[i + 1, 1] := EigenVector^[i] / Summ;
        end;
        //Eigenvectors of dominant eigenvalue of transpose
        for i := 1 to Dimension do EigenVector^[i] := 1;
        if EigenVect(A_t, 1, Dimension, MAXITER, TOL, EigenVector, temp) = MAT_OK then
        begin
          R[1, 1] := temp;
          Summ := Sum(EigenVector, 1, Dimension);
          for i := 1 to Dimension do
            R[Dimension + i + 1, 1] := EigenVector^[i] / Summ;
        end;
      end;
    except
      on E: Exception do
        WriteError(R, DLL_MEM_ERROR)
    end;
  finally
    result := R;
    DelMatrix(A, Dimension, Dimension);
    DelMatrix(ACopy, Dimension, Dimension);
    DelMatrix(A_t, Dimension, Dimension);
    DelVector(Lambda_Re, Dimension);
    DelVector(Lambda_Im, Dimension);
    DelVector(EigenVector, Dimension);
  end
  else
    result := DLL_MEM_ERROR;
end;

function dEigVect(var V: Variant): Variant; stdcall;
{ Calls TPMath: All eigenvalues and eigenvectors of V }
var
  A, ACopy              : PMatrix;      { Matrix }
  i, j                  : Integer;      { Dimension of matrix }
  Lambda_Re, Lambda_Im  : PVector;      { Vector of eigenvalues }
  EigenVectors          : PMatrix;      { Vector of eigenvectors }
  R                     : Variant;
  Summ                  : double;
  Rows                  : integer;

begin
  if GetExcelData(V, A, Rows, Rows) then
  try
    //Get memory for TPMath routines
    //DimMatrix(A, Rows, Rows);
    DimMatrix(ACopy, Rows, Rows);
    DimMatrix(EigenVectors, Rows, Rows);
    DimVector(Lambda_Re, Rows);
    DimVector(Lambda_Im, Rows);
    //Create array to hold results
    R := VarArrayCreate([1, Rows, 1, Rows + 2], varDouble);
    try
      //for i := 1 to Rows do for j := 1 to Rows do A^[i]^[j] := V[i, j];
      FillPMatrix(EigenVectors, 1, Rows, Rows);
      //Keep a copy of A
      CopyMatrix(ACopy, A, 1, 1, Rows, Rows);
      //Compute eigenvalues
      if EigenVals(A, 1, Rows, Lambda_Re, Lambda_Im) = MAT_OK then
      begin
        eigsrt(Lambda_Re, Lambda_im, EigenVectors, Rows);
        //Store eigenvalues
        for i := 1 to Rows do
        begin
          R[i, Rows + 1] := Lambda_Re^[i];
          R[i, Rows + 2] := Lambda_Im^[i];
        end;
        //Compute eigenvectors
        for i := 1 to Rows do
        begin
          //A is destroyed in EigenVect procedure
          CopyMatrix(A, ACopy, 1, 1, Rows, Rows);
          //if abs(Lambda_Re^[i]-1)<0.0001 then Lambda_Re^[i]:=Lambda_Re^[i]*1.01;
          case EigenVect(A, 1, Rows, MAXITER, TOL, EigenVectors^[i], Lambda_Re^[i]) of
            MAT_OK:
              begin
                //Summ := Sum(EigenVectors^[i], 1, Rows);
                for j := 1 to Rows do
                begin
                  //EigenVectors^[i]^[j] := Divide(EigenVectors^[i]^[j], Summ);
                  R[i, j] := EigenVectors^[i]^[j];
                end;
              end;
            MAT_SINGUL:
              WriteError(R, MAT_SINGUL);
            MAT_NON_CONV:
              WriteError(R, MAT_NON_CONV);
          end;
        end;
        eigsrt(Lambda_Re, Lambda_im, EigenVectors, Rows);
      end;
    except
      on E: Exception do
        WriteError(R, DLL_MEM_ERROR)
    end;
  finally
    for i := 1 to Rows do
    begin
      for j := 1 to Rows do
        R[i, j] := EigenVectors^[i]^[j];
      R[i, Rows + 1] := Lambda_re^[i];
      R[i, Rows + 2] := Lambda_im^[i];
    end;
    Result := R;
    //Free up pointers
    DelMatrix(A, Rows, Rows);
    DelMatrix(ACopy, Rows, Rows);
    DelMatrix(EigenVectors, Rows, Rows);
    DelVector(Lambda_Re, Rows);
    DelVector(Lambda_Im, Rows);
  end
  else
    result := DLL_MEM_ERROR;
end;

function dPCA(var ExcelVariant: Variant): Variant; stdcall;
{Calls TPMath: Principal components analysis of a data matrix }
var
  N                     : Integer;      { Number of observations }
  Nvar                  : Integer;      { Number of variables }
  XName                 : PStrVector;   { Variable names }
  X                     : PMatrix;      { Variables }
  M                     : PVector;      { Mean vector }
  S                     : PVector;      { Standard deviations }
  V                     : PMatrix;      { Variance-covariance matrix }
  R                     : PMatrix;      { Correlation matrix }
  Lambda                : PVector;      { Eigenvalues of correlation matrix }
  C                     : PMatrix;      { Eigenvectors of correlation matrix }
  Rc                    : PMatrix;      { Correlations between variables & princ. factors }
  Z                     : PMatrix;      { Scaled variables }
  F                     : PMatrix;      { Principal factors }
  Res                   : Variant;      { Results returned to Excel }
  i, j                  : integer;

begin
  if GetExcelData(ExcelVariant, X, NVar, N) then
  try
    //Get memory for TPMath routines
    DimVector(M, Nvar);
    DimVector(S, Nvar);
    DimMatrix(V, Nvar, Nvar);
    DimMatrix(R, Nvar, Nvar);
    DimVector(Lambda, Nvar);
    DimMatrix(C, Nvar, Nvar);
    DimMatrix(Rc, Nvar, Nvar);
    DimMatrix(Z, Nvar, N);
    DimMatrix(F, Nvar, N);
    //Create array to hold results
    Res := VarArrayCreate([1, NVar * 3 + N, 1, NVar], varDouble);
    try
      VecMean(X, N, 1, Nvar, M);
      VecSD(X, N, 1, Nvar, M, S);
      { Scale variables }
      ScaleVar(X, N, 1, Nvar, M, S, Z);
      { Compute variance-covariance matrix }
      MatVarCov(X, N, 1, Nvar, M, V);
      { Compute correlation matrix }
      MatCorrel(V, 1, Nvar, R);
      // Perform principal component analysis.
      // The original matrix R is destroyed.
      case PCA(R, 1, Nvar, Lambda, C, Rc) of
        MAT_OK:
          begin
            //Compute principal factors
            PrinFac(Z, N, 1, Nvar, C, F);
            //Eigenvalues of correlation matrix';
            for i := 1 to NVar do
              Res[i, 1] := Lambda^[i];
            //Eigenvectors (columns) of correlation matrix';
            for i := NVar + 1 to NVar * 2 do for j := 1 to Nvar do
                Res[i, j] := C^[j]^[i - NVar];
            //Correlations between factors (columns) and variables (lines)';
            for i := NVar * 2 + 1 to NVar * 3 do for j := 1 to Nvar do
                Res[i, j] := Rc^[j]^[i - NVar * 2];
            //Principal factors
            for i := NVar * 3 + 1 to NVar * 3 + N do for j := 1 to Nvar do
                Res[i, j] := F^[j]^[i - NVar * 3];
          end;
        MAT_NON_CONV:
          WriteError(Res, MAT_NON_CONV);
      end;
    except
      on E: Exception do
        WriteError(Res, DLL_MEM_ERROR)
    end;
  finally
    Result := Res;
    //Free up pointers
    DelVector(M, Nvar);
    DelVector(S, Nvar);
    DelMatrix(X, Nvar, N);
    DelMatrix(V, Nvar, Nvar);
    DelMatrix(R, Nvar, Nvar);
    DelVector(Lambda, Nvar);
    DelMatrix(C, Nvar, Nvar);
    DelMatrix(Rc, Nvar, Nvar);
    DelMatrix(Z, Nvar, N);
    DelMatrix(F, Nvar, N);
  end;
end;

function dSVD(var ExcelVariant: Variant): Variant; stdcall;
{ Calls TPMath: SVD factors the matrix A (n x m, with n >= m) as a product
  U * S * V' where U is a (n x m) column-orthogonal matrix, S is a (m x m)
  diagonal matrix with elements >= 0 (the singular values) and V is a (m x m)
  orthogonal matrix. }
var
  A, V                  : PMatrix;
  S                     : PVector;
  Rows, Cols, i, j      : integer;
  R                     : Variant;
begin
  if GetExcelData(ExcelVariant, A, Rows, Cols) then
  try
    if rows < Cols then
    begin
      Result := DIM_ERROR;
      exit;
    end;
    //Get memory for TPMath routines
    //DimMatrix(A, Rows, Cols);
    DimMatrix(V, Rows, Rows);
    DimVector(S, Cols);
    //Create array to hold results
    R := VarArrayCreate([1, Rows, 1, Cols * 3], varDouble);
    try
      //Copy the values of V to a PMatrix
      //for i := 1 to Rows do for j := 1 to Cols do A^[i]^[j] := ExcelVariant[i, j];
      case SV_Decomp(A, 1, Rows, Cols, S, V) of
        MAT_OK:
          begin
            for i := 1 to Rows do
              for j := 1 to Cols do
              begin
                R[i, j] := A^[i]^[j];
                if i <= Cols then if i = j then R[i, j + Cols] := S^[i] else R[i, j + Cols] := 0;
                if i <= Cols then R[i, j + Cols * 2] := V^[i]^[j];
              end;
          end;
        MAT_NON_CONV:
          WriteError(R, MAT_NON_CONV);
      end;
    except
      on E: Exception do
        WriteError(R, DLL_MEM_ERROR);
    end;
  finally
    Result := R;
    //Free up pointers
    DelMatrix(A, Rows, Cols);
    DelMatrix(V, Rows, Cols);
    DelVector(S, Rows);
  end
  else
    result := DLL_MEM_ERROR
end;

function dQR(var ExcelVariant: Variant): Variant; stdcall;
{ Calls TPMath: QR decomposition factors the matrix A (n x m, with n >= m) as a
  product Q * R where Q is a (n x m) column-orthogonal matrix, and R
  a (m x m) upper triangular matrix. This routine is used in conjunction
  with QR_Solve to solve a system of equations.  }
var
  A, RofQR              : PMatrix;
  S                     : PVector;
  Rows, Cols, i, j      : integer;
  R                     : Variant;
begin
  if GetExcelData(ExcelVariant, A, Rows, Cols) then
  try
    if rows < Cols then
    begin
      Result := DIM_ERROR;
      exit;
    end;
    //Get memory for TPMath routines
    //DimMatrix(A, Rows, Cols);
    DimMatrix(RofQR, Cols, Cols);
    //Create array to hold results
    R := VarArrayCreate([1, Rows, 1, Cols * 2], varDouble);
    try
      //Calculate result and store
      case QR_Decomp(A, 1, Rows, Cols, RofQR) of
        MAT_OK:
          begin
            for i := 1 to Rows do
              for j := 1 to Cols do
                R[i, j] := A^[i]^[j];
            for i := 1 to Cols do
              for j := Cols + 1 to Cols + Cols do
                R[i, j] := RofQR^[i]^[j - Cols];
          end;
        MAT_NON_CONV:
          WriteError(R, MAT_NON_CONV);
      end;
    except
      on E: Exception do
        WriteError(R, DLL_MEM_ERROR);
    end;
  finally
    Result := R;
    //Free up pointers
    DelMatrix(A, Rows, Cols);
    DelMatrix(RofQR, Cols, Cols);
  end
  else
    result := DLL_MEM_ERROR
end;

function dRoots(var ExcelVariant: Variant): Variant; stdcall;
{ Calls TPMath: Real and complex roots of a real polynomial by the method of the
  companion matrix }
var
  A                     : PMatrix;
  V                     : PVector;
  Rows, Cols, i, j      : integer;
  X_Re, X_Im            : PVector;
  R                     : Variant;
begin
  if GetExcelData(ExcelVariant, A, Rows, Cols) then
  try
    if rows = 1 then
    begin
      DimVector(V, Cols);
      DimVector(X_Re, Cols);
      DimVector(X_Im, Cols);
      for i := 1 to Cols do V^[i - 1] := A^[1]^[i];
      R := VarArrayCreate([1, Cols - 1, 1, 2], varDouble);
      case RootPol(V, Cols - 1, X_Re, X_Im) of
        MAT_OK:
          for i := 1 to Cols - 1 do
          begin
            R[i, 1] := X_Re^[i];
            R[i, 2] := X_Im^[i];
          end;
        MAT_NON_CONV:
          WriteError(R, MAT_NON_CONV);
      end;
      Result := R;
      DelVector(V, Cols);
      DelVector(X_Re, Cols);
      DelVector(X_Im, Cols);
    end
    else if Cols = 1 then
    begin
      DimVector(V, Rows);
      DimVector(X_Re, Rows);
      DimVector(X_Im, Rows);
      for i := 1 to Rows do V^[i - 1] := A^[i]^[1];
      R := VarArrayCreate([1, Rows - 1, 1, 2], varDouble);
      case RootPol(V, Rows - 1, X_Re, X_Im) of
        MAT_OK:
          for i := 1 to Rows - 1 do
          begin
            R[i, 1] := X_Re^[i];
            R[i, 2] := X_Im^[i];
          end;
        MAT_NON_CONV:
          WriteError(R, MAT_NON_CONV);
      end;
      Result := R;
      DelVector(V, Rows);
      DelVector(X_Re, Rows);
      DelVector(X_Im, Rows);
    end
    else
    begin
      Result := DIM_ERROR;
      exit;
    end;
  except on E: Exception do
      Result := DLL_MEM_ERROR;
  end                                   {try}
  else
    Result := DLL_MEM_ERROR
end;

function Normal1: double;
var
  s1, s2, test          : double;
begin
  repeat
    s1 := 2 * Random - 1;
    s2 := 2 * Random - 1;
    test := s1 * s1 + s2 * s2;
  until (test <= 1) and (test > 0);
  Normal1 := s1 * sqrt(-2 * ln(test) / test);
end;

function dNormalDev(mn, sd: double): double; stdcall;
begin
  result := mn + sd * normal1;
end;

function dNormalInt(mn, sd: double): Integer; stdcall;
begin
  result := round(dNormalDev(mn, sd));
end;

function dRandReal(x, y: double): double; stdcall;
var
  Range                 : double;
begin
  Range := Y - X;
  dRandReal := X + Random * Range;
end;

function dRandInt(x, y: double): integer; stdcall;
var
  Range                 : integer;
begin
  Range := round(y) - round(x);
  result := round(x) + round(Random * Range);
end;

function gammln(xx: double): double;
const
  stp                   = 2.50662827465;
  half                  = 0.5;
  one                   = 1.0;
  fpf                   = 5.5;
var
  x, tmp, ser           : double;
  j                     : integer;
  cof                   : array[1..6] of double;
begin
  cof[1] := 76.18009173;
  cof[2] := -86.50532033;
  cof[3] := 24.01409822;
  cof[4] := -1.231739516;
  cof[5] := 0.120858003E-2;
  cof[6] := -0.536382E-5;
  x := xx - one;
  tmp := x + fpf;
  tmp := (x + half) * ln(tmp) - tmp;
  ser := one;
  for j := 1 to 6 do
  begin
    x := x + one;
    ser := ser + cof[j] / x
  end;
  gammln := tmp + ln(stp * ser)
end;
{---------------------------------------------------------------------}

function dExpDev(mean: double): double; stdcall;
begin
  result := -ln(Random) * mean
end;
{---------------------------------------------------------------------}

function dGeomDev(mean: double): Integer; stdcall;
begin
  result := round(-ln(Random) * mean)
end;
{---------------------------------------------------------------------}

function dBinomialDev(n: double; pp: double): integer; stdcall;
label
  1, 2;
const
  pi                    = 3.14159265358979;
var
  am, em, en, g, angle  : double;
  oldg, p, pc, bnl      : double;
  pclog, plog, pold, sq, t, y: double;
  j, nold               : integer;
  InternalCounter       : integer;
begin
  if (pp <= 0) or (n <= 0) then         //Altered Preiss et al's routine here
  begin                                 //
    result := 0;                        //
    goto 2;                             //
  end;
  InternalCounter := 0;                 //Reqd for Delphi 32
  nold := -1;
  pold := -1.0;
  if (pp <= 0.5) then
    p := pp
  else
    p := 1.0 - dMinD(pp, 1.0);
  am := n * p;

  if (n < 25) then
  begin
    bnl := 0.0;
    for j := 1 to trunc(n) do
    begin
      if (Random < p) then
        bnl := bnl + 1.0
    end
  end
  else                                  {n >= 25}
    if (am < 1.0) then
    begin
      g := exp(-am);
      t := 1.0;
      for j := 0 to trunc(n) do
      begin
        t := t * Random;                {       QUICK & DIRTY FIX            }
        InternalCounter := j;           {j is not recognised outside this    }
        if (t < g) then                 {loop in Delphi optimised 32 bit code}
          goto 1                        {so set a local variable equal to j  }
      end;                              {for later use                       }
      {j := n;}
      1: bnl := InternalCounter;
    end
    else                                {n >= 25 and am >= 1.0}
    begin
      if (n <> nold) then
      begin
        en := n;
        oldg := gammln(en + 1.0);
        nold := trunc(n)
      end;
      if (p <> pold) then
      begin
        pc := 1.0 - p;
        plog := ln(p);
        pclog := ln(pc);
        pold := p
      end;
      sq := sqrt(2.0 * am * pc);
      repeat
        repeat
          angle := pi * Random;
          y := sin(angle) / cos(angle);
          em := sq * y + am;
        until ((em >= 0.0) and (em < en + 1.0));
        em := trunc(em);
        t := 1.2 * sq * (1.0 + sqr(y)) * exp(oldg - gammln(em + 1.0) -
          gammln(en - em + 1.0) + em * plog + (en - em) * pclog);
      until (Random <= t);
      bnl := em
    end;
  if (p <> pp) then
    bnl := n - bnl;
  result := round(bnl);
  2:
end;

function dLogNormalDev(mn, sd: double): double; stdcall;
{ Generates lognormally distributed deviates with mean=mn and std dev=sd }
var
  delta, epsilon, temp  : double;
begin
  {Determine the std dev for seeding the normal rv generator}
  temp := (mn * mn + sd * sd) / (mn * mn);
  delta := sqrt(ln(temp));
  {Determine the mean for seeding the normal rv generator}
  epsilon := ln(mn) - sd * sd * 0.5;
  result := Exp(dNormalDev(epsilon, delta));
end;

function dGammaDev(ia: double): double; stdcall;
var
  am, e, s, v1, v2, x, y: double;
  j                     : integer;

begin
  ia := round(ia);
  if (ia < 1) then
  begin
    result := 0;
    exit
  end;
  if (ia < 6) then
  begin
    x := 1.0;
    for j := 1 to round(ia) do x := x * Random;
    x := -ln(x);
  end else
  begin
    repeat
      repeat
        repeat
          v1 := 2.0 * Random - 1.0;
          v2 := 2.0 * Random - 1.0;
        until ((sqr(v1) + sqr(v2)) <= 1.0);
        y := v2 / v1;
        am := ia - 1;
        s := sqrt(2.0 * am + 1.0);
        x := s * y + am;
      until (x > 0.0);
      e := (1.0 + sqr(y)) * exp(am * ln(x / am) - s * y);
    until (Random <= e)
  end;
  result := x
end;

function dPoissonDev(mu: double): integer; stdcall;
//Generates Poisson deviates with mean = mu (and variance = mu ... naturally!)
//Algorithm from Burgman et al
var
  p, C                  : double;
  n                     : integer;
begin
  if mu = 0 then
    result := 0
  else
    if mu < 50 then
    begin
      p := 1;
      n := 0;
      C := Exp(-mu);
      repeat
        p := p * Random;
        inc(n)
      until p < C;
      result := n - 1;
    end
    else                                {Normal approximation}
      result := round(dNormalDev(mu, sqrt(mu)));
end;

function dGamma(N: Double): Double; stdcall;
begin
  result := Gamma(N)
end;

function dNegBinomial(N, mu, K: Double): Double; stdcall;
//Probability of n events given mean mu and clumping parameter k
var
  FirstPart             : Double;
begin
  FirstPart := Gamma(K + N) / (Gamma(K) * Fact(trunc(N)));
  result := FirstPart * pow(mu / (K + mu), N) * pow(K / (K + mu), K)
end;

function dNegBinomialDev(mu, K: Double): Integer; stdcall;
var
  SUM, U                : Double;
  i                     : Integer;
begin
  U := random;
  SUM := 0;
  i := 0;
  repeat
    SUM := SUM + dNegBinomial(i, mu, K);
    inc(i);
  until SUM >= U;
  result := i - 1;
end;

exports
  dNormalDev,
  dNormalInt,
  dRandReal,
  dRandInt,
  dExpDev,
  dGeomDev,
  dBinomialDev,
  dLogNormalDev,
  dGammaDev,
  dPoissonDev,
  dNegBinomial,
  dNegBinomialDev,
  dGamma,
  dEigVal,
  dDomEig,
  dLife,
  dEigVect,
  dPCA,
  dSVD,
  dCholesky,
  dLUDecomp,
  dQR,
  dRoots,
  dQuickSort,
  dRowMatrixMult,
  dMtoV,
  dACF;

begin
  Randomize;
end.

