{==========================================================================

    testfft.pas  -  Don Cross <dcross@intersrv.com>

    Modified by Jean Debord <JDebord@compuserve.com> for use with TP Math.

    This program is a test/demo for the file 'fourier.pas'.
    Get the latest version of 'fourier.pas' and 'testfft.pas' at the
    following URL.

       http://www.intersrv.com/~dcross/fft.html#pascal

    NOTE:  You may need to modify the const string 'PathToBGI' to point
           to the correct drive and subdirectory for the BGI drivers on
           your computer, in order for the graphics to work.

    ---------------   What this program does -------------------------

    First, it generates a time signal consisting of a large 200 Hz sine
    wave added to a small 2000 Hz cosine wave, which is graphed on the
    screen.  (Press ENTER after you are done viewing each graph.)

    Next, it performs the FFT and graphs the resulting complex
    frequency samples.

    Then, it filters out all frequency components above 1000 Hz in
    the transformed data.

    Finally, it performs the inverse transform to get a filtered
    time signal back, and graphs the result.

    ------------------------ Revision history ------------------------

1997 March 1 [Jean Debord]
     Modifications for use with the TP Math library:
    1. Added a USES clause for the TP Math units.
    2. Set real type to Float (defined in FMATH.PAS) which means Real,
       Single, Double or Extended, according to the compiler directives.
    3. Changed array types to those defined in TP Math. Modified array
       allocation, deallocation and reference accordingly.
    4. Removed compiler directives, which were no longer necessary.
    5. Modified some typographical and formatting options so that the
       code looks like the other TP Math units.
    No modification was made to the original algorithm.

1996 December 12 [Don Cross]
    Added code to test the new procedure Fourier.CalcFrequency.
    Cleaned up some comments.
    Added code to preserve the original text mode.

1996 November 17 [Don Cross]
    Wrote and debugged first version.

==========================================================================}

program TestFFT;

uses
  { Turbo units }
  Crt, Graph,
  { TP Math units }
  FMath, Matrices, Fourier;

const
  PathToBGI = 'C:\BP\BGI';  { Change as necessary }

  function F(T : Float) : Float;
  begin
    F := Sin(200 * 2 * PI * T) + 0.2 * Cos(2000 * 2 * PI * T);
  end;

const
  NumSamples = 512;      { buffer size must be power of 2 }
  SamplingRate = 22050;  { sampling rate in Hz }

var
  RealIn, ImagIn, RealOut, ImagOut : PVector;
  OutputListingFile : Text;
  I : Integer;
  T, Dt : Float;

  procedure Test_CalcFrequency;
  var
    Yr, Yi : Float;
    I : Integer;
  begin
    { Fill input buffers with random data }
    for I := 0 to NumSamples - 1 do
      begin
        RealIn^[I] := Random(10000);
        ImagIn^[I] := Random(10000);
      end;

    WriteLn(OutputListingFile);
    WriteLn(OutputListingFile, '*** Testing procedure CalcFrequency ***');
    WriteLn(OutputListingFile);

    FFT(NumSamples, RealIn, ImagIn, RealOut, ImagOut);
    for I := 0 to NumSamples - 1 do
      begin
        CalcFrequency(NumSamples, I, RealIn, ImagIn, Yr, Yi);
        WriteLn(OutputListingFile, I:4,
                RealOut^[I]:15:6, Yr:15:6,
                ImagOut^[I]:20:6, Yi:15:6);
      end;
  end;

  procedure ListData(RealData, ImagData : PVector; Comment : String);
  var
    I, Yr, Yi, Prev_Yr, Prev_Yi : Integer;
    Trash : Char;
    MaxAbsValue : Float;
  begin
    WriteLn(OutputListingFile, '*** ', Comment, ' ***');
    WriteLn(OutputListingFile);
    WriteLn(OutputListingFile, 'index':20, 'real':20, 'imag':20);
    for I := 1 to NumSamples do
      begin
        WriteLn(OutputListingFile, I:20,
                RealData^[I]:20:5, ImagData^[I]:20:5);
      end;

    WriteLn(OutputListingFile);
    WriteLn(OutputListingFile, '------------------------------------------------------------------------');
    WriteLn(OutputListingFile);

    MaxAbsValue := 0.0;
    for I := 0 to NumSamples - 1 do
      begin
        if Abs(RealData^[I]) > MaxAbsValue then
          MaxAbsValue := Abs(RealData^[I]);
        if Abs(ImagData^[I]) > MaxAbsValue then
          MaxAbsValue := Abs(ImagData^[I]);
      end;

    for I := 0 to NumSamples - 1 do
      begin
        Yr := Trunc(GetMaxY * (1 - (RealData^[I] / MaxAbsValue + 1) / 2));
        Yi := Trunc(GetMaxY * (1 - (ImagData^[I] / MaxAbsValue + 1) / 2));

        if I > 0 then
          begin
            SetColor(LIGHTRED);
            Line(I - 1, Prev_Yr, I, Yr);
            SetColor(LIGHTGREEN);
            Line(I - 1, Prev_Yi, I, Yi);
          end;

        Prev_Yr := Yr;
        Prev_Yi := Yi;
      end;

    sound(800);
    delay(100);
    nosound;
    Trash := ReadKey;  { Pause }
    ClearDevice;
  end;

var
  GraphDriver, GraphMode, StartupTextMode : Integer;
  FreqIndex : Integer;

begin
  DimVector(RealIn, NumSamples);
  DimVector(ImagIn, NumSamples);
  DimVector(RealOut, NumSamples);
  DimVector(ImagOut, NumSamples);

  StartupTextMode := LastMode;
  Assign(OutputListingFile, 'fftout.txt');
  Rewrite(OutputListingFile);

  GraphDriver := Detect;
  InitGraph(GraphDriver, GraphMode, PathToBGI);

  Dt := 1.0 / SamplingRate;
  T := 0.0;

  for I := 0 to NumSamples - 1 do
    begin
      RealIn^[I] := F(T);
      ImagIn^[I] := 0.0;
      T := T + Dt;
    end;

  ListData(RealIn, ImagIn, 'Time domain data before transform');

  FFT(NumSamples, RealIn, ImagIn, RealOut, ImagOut);

  ListData(RealOut, ImagOut, 'Frequency domain data after transform');

  { Filter out everything above 1000 Hz (low-pass) }
  FreqIndex := Trunc(1000.0 * NumSamples / SamplingRate);
  for I := 0 to NumSamples - 1 do
    begin
      if ((I > FreqIndex) and (I < NumSamples div 2)) or
      ((I >= NumSamples div 2) and (I < NumSamples - FreqIndex)) then
        begin
          RealOut^[I] := 0.0;
          ImagOut^[I] := 0.0;
        end;
    end;

  IFFT(NumSamples, RealOut, ImagOut, RealIn, ImagIn);
  ListData(RealIn, ImagIn, 'Time domain data after inverse transform');

  Test_CalcFrequency;

  Close(OutputListingFile);
  CloseGraph;
  TextMode(StartupTextMode);
end.
