{ **********************************************************************
  *                         Program MINFUNC.PAS                        *
  *                             Version 1.6                            *
  *                    (c) J. Debord, December 1998                    *
  **********************************************************************
  This program demonstrates the 4 methods implemented in OPTIM.PAS for
  minimizing a function of several variables: Marquardt, BFGS, Simplex
  and Simulated Annealing.

  The first 3 methods are local optimization methods which work only at
  the vicinity of a minimum. By contrast, simulated annealing can escape
  from such a local minimum and find the global one. This is obtained by
  accepting that the function increases sometimes during the process.
  The probability of acceptation is controlled by a parameter called
  temperature: a higher temperature means a higher probability of
  acceptation. At each temperature, the algorithm generates a random
  point from the current one (using a gaussian distribution in our
  implementation). The new point is always accepted if the function
  decreases, otherwise the probability of acceptation depends on the
  temperature. The process is repeated for a given number of points,
  then the temperature is decreased. The algorithm stops when the
  temperature has reached a limit defined by the user.

  Simulated annealing is used mainly to determine an approximation to
  the global minimum, which may be refined by one of the other methods.
  ********************************************************************** }

program MinFunc;

uses
  Crt, FMath, Matrices, Random, Optim;

const
  MAXITER = 1000;     { Maximum number of iterations }
  TOL     = 1.0E-08;  { Required precision }

var
  X       : PVector;  { Coordinates of minimum }
  H_inv   : PMatrix;  { Inverse hessian matrix }
  F_min   : Float;    { Function value at minimum }
  ErrCode : Integer;  { Error code }

{ ----------------------------------------------------------------------
  Include here the function to be minimized: MINFUNCx.INC, where x = 0-9
  ---------------------------------------------------------------------- }

  {$I MINFUNC0.INC}

{ ---------------------------------------------------------------------- }

  procedure Pause;
  begin
    GotoXY(1, 25);
    Write('Press <Enter> to continue');
    ReadLn;
    ClrScr;
  end;

  procedure WriteResult(Hessian : Boolean);
  { Output results to screen. Hessian indicates
    if the Hessian matrix has been evaluated }
  var
    I, J : Integer;
  begin
    WriteLn('Coordinates of minimum :', #10);
    for I := 1 to NVAR do
      WriteLn(X^[I]:14:8);
    WriteLn(#10, 'Function value :', F_min:14);

    if Hessian then
      begin
        WriteLn(#10, 'Inverse Hessian matrix :', #10);
        for I := 1 to NVAR do
          begin
            for J := 1 to NVAR do
              Write(H_inv^[I]^[J]:14:8);
            WriteLn;
          end;
      end;
  end;

{ ----------------------------------------------------------------------
  Include here the minimization algorithm MINALGOx.INC, where
  x = 1 (Marquardt), 2 (BFGS), 3 (Simplex), 4 (Simulated Annealing)
  ---------------------------------------------------------------------- }

  {$I MINALGO3.INC}

{ ---------------------------------------------------------------------- }

begin
  ClrScr;

  { Allocate arrays }
  DimVector(X, NVAR);
  DimMatrix(H_inv, NVAR, NVAR);

  { Set approximate coordinates of minimum }
  InitVar;

  { Mimimize function }
  Minimize;

  Pause;

  { Deallocate arrays }
  DelVector(X, NVAR);
  DelMatrix(H_inv, NVAR, NVAR);
end.

