{ **********************************************************************
  *                             MATHFUNC.INC                           *
  **********************************************************************
             Mathematical functions for TPMATH (Pascal version)
  ********************************************************************** }


  {$I COMMON.INC}

  function Exp(X : Float) : Float;
  begin
    MathErr := FN_OK;
    if X < MINLOG then
      Exp := DefaultVal(UNDERFLOW)
    else if X > MAXLOG then
      Exp := DefaultVal(OVERFLOW)
    else
      Exp := System.Exp(X);
  end;

  function Exp2(X : Float) : Float;
  var
    XLn2 : Float;
  begin
    MathErr := FN_OK;
    XLn2 := X * LN2;
    if XLn2 < MINLOG then
      Exp2 := DefaultVal(UNDERFLOW)
    else if XLn2 > MAXLOG then
      Exp2 := DefaultVal(OVERFLOW)
    else
      Exp2 := System.Exp(XLn2);
  end;

  function Exp10(X : Float) : Float;
  var
    XLn10 : Float;
  begin
    MathErr := FN_OK;
    XLn10 := X * LN10;
    if XLn10 < MINLOG then
      Exp10 := DefaultVal(UNDERFLOW)
    else if XLn10 > MAXLOG then
      Exp10 := DefaultVal(OVERFLOW)
    else
      Exp10 := System.Exp(XLn10);
  end;

  function Ln(X : Float) : Float;
  begin
    MathErr := FN_OK;
    if X < 0.0 then
      Ln := DefaultVal(DOMAIN)
    else if X = 0.0 then
      Ln := DefaultVal(SING)
    else
      Ln := System.Ln(X);
  end;

  function Log10(X : Float) : Float;
  begin
    MathErr := FN_OK;
    if X < 0.0 then
      Log10 := DefaultVal(DOMAIN)
    else if X = 0.0 then
      Log10 := DefaultVal(SING)
    else
      Log10 := System.Ln(X) * INVLN10;
  end;

  function Log2(X : Float) : Float;
  begin
    MathErr := FN_OK;
    if X < 0.0 then
      Log2 := DefaultVal(DOMAIN)
    else if X = 0.0 then
      Log2 := DefaultVal(SING)
    else
      Log2 := System.Ln(X) * INVLN2;
  end;

  function LogA(X, A : Float) : Float;
  begin
    MathErr := FN_OK;
    if (X < 0.0) or (A <= 0.0) or (A = 1.0) then
      LogA := DefaultVal(DOMAIN)
    else if X = 0.0 then
      LogA := Sgn(1.0 - A) * DefaultVal(SING)
    else
      LogA := System.Ln(X) / System.Ln(A);
  end;

  function Pow(X, Y : Float) : Float;
  { Computes X^Y = Exp(Y * Ln(X)), for X >= 0 }
  var
    YLnX : Float;
  begin
    MathErr := FN_OK;
    if X < 0.0 then
      begin
        Pow := DefaultVal(DOMAIN);
        Exit;
      end;

    if X = 0.0 then
      begin
        if Y = 0.0 then       { 0^0 = lim  x^x = 1 }
          Pow := 1.0          {       x->0         }
        else if Y > 0.0 then
          Pow := 0.0          { 0^Y = 0 }
        else
          Pow := DefaultVal(SING);
        Exit;
      end;

    if Y = 0.0 then
      begin
        Pow := 1.0;
        Exit;
      end;

    YLnX := Y * System.Ln(X);

    if YLnX < MINLOG then
      Pow := DefaultVal(UNDERFLOW)
    else if YLnX > MAXLOG then
      Pow := DefaultVal(OVERFLOW)
    else
      Pow := System.Exp(YLnX);
  end;

  function Sin(X : Float) : Float;
  begin
    MathErr := FN_OK;
    Sin := System.Sin(X);
  end;

  function Cos(X : Float) : Float;
  begin
    MathErr := FN_OK;
    Cos := System.Cos(X);
  end;

  procedure SinCos(X : Float; var SinX, CosX : Float);
  begin
    MathErr := FN_OK;
    SinX := System.Sin(X);
    CosX := System.Cos(X);
  end;

  function Tan(X : Float) : Float;
  var
    SinX, CosX : Float;
  begin
    MathErr := FN_OK;
    SinCos(X, SinX, CosX);
    if CosX = 0.0 then
      Tan := Sgn(SinX) * DefaultVal(SING)
    else
      Tan := SinX / CosX;
  end;

  function ArcSin(X : Float) : Float;
  begin
    MathErr := FN_OK;
    if (X < - 1.0) or (X > 1.0) then
      ArcSin := DefaultVal(DOMAIN)
    else if X = 1.0 then
      ArcSin := PIDIV2
    else if X = - 1.0 then
      ArcSin := - PIDIV2
    else
      ArcSin := System.ArcTan(X / Sqrt(1.0 - Sqr(X)));
  end;

  function ArcCos(X : Float) : Float;
  begin
    MathErr := FN_OK;
    if (X < - 1.0) or (X > 1.0) then
      ArcCos := DefaultVal(DOMAIN)
    else if X = 1.0 then
      ArcCos := 0.0
    else if X = - 1.0 then
      ArcCos := PI
    else
      ArcCos := PIDIV2 - System.ArcTan(X / Sqrt(1.0 - Sqr(X)));
  end;

  function ArcTan(X : Float) : Float;
  begin
    MathErr := FN_OK;
    ArcTan := System.ArcTan(X);
  end;

  function ArcTan2(Y, X : Float) : Float;
  var
    Theta : Float;
  begin
    MathErr := FN_OK;
    if X = 0.0 then
      if Y = 0.0 then
        ArcTan2 := 0.0
      else if Y > 0.0 then
        ArcTan2 := PIDIV2
      else
        ArcTan2 := - PIDIV2
    else
      begin
        Theta := System.ArcTan(Y / X);  { 4th/1st quadrant -PI/2..PI/2 }
        if X < 0.0 then                 { 2nd/3rd quadrants }
          if Y >= 0.0 then
            Theta := Theta + PI         { 2nd quadrant:  PI/2..PI }
          else
            Theta := Theta - PI;        { 3rd quadrant: -PI..-PI/2 }
        ArcTan2 := Theta;
      end;
  end;

  function Sinh(X : Float) : Float;
  var
    ExpX : Float;
  begin
    MathErr := FN_OK;
    if (X < MINLOG) or (X > MAXLOG) then
      Sinh := Sgn(X) * DefaultVal(OVERFLOW)
    else
      begin
        ExpX := System.Exp(X);
        Sinh := 0.5 * (ExpX - 1.0 / ExpX);
      end;
  end;

  function Cosh(X : Float) : Float;
  var
    ExpX : Float;
  begin
    MathErr := FN_OK;
    if (X < MINLOG) or (X > MAXLOG) then
      Cosh := DefaultVal(OVERFLOW)
    else
      begin
        ExpX := System.Exp(X);
        Cosh := 0.5 * (ExpX + 1.0 / ExpX);
      end;
  end;

  procedure SinhCosh(X : Float; var SinhX, CoshX : Float);
  var
    ExpX, ExpMinusX : Float;
  begin
    MathErr := FN_OK;
    if (X < MINLOG) or (X > MAXLOG) then
      begin
        CoshX := DefaultVal(OVERFLOW);
        SinhX := Sgn(X) * CoshX;
      end
    else
      begin
        ExpX := System.Exp(X);
        ExpMinusX := 1.0 / ExpX;
        SinhX := 0.5 * (ExpX - ExpMinusX);
        CoshX := 0.5 * (ExpX + ExpMinusX);
      end;
  end;

  function Tanh(X : Float) : Float;
  var
    SinhX, CoshX : Float;
  begin
    SinhCosh(X, SinhX, CoshX);
    Tanh := SinhX / CoshX;
  end;

  function ArcSinh(X : Float) : Float;
  begin
    MathErr := FN_OK;
    ArcSinh := System.Ln(X + Sqrt(Sqr(X) + 1.0));
  end;

  function ArcCosh(X : Float) : Float;
  begin
    MathErr := FN_OK;
    if X < 1.0 then
      ArcCosh := DefaultVal(DOMAIN)
    else
      ArcCosh := System.Ln(X + Sqrt(Sqr(X) - 1.0));
  end;

  function ArcTanh(X : Float) : Float;
  begin
    MathErr := FN_OK;
    if (X < - 1.0) or (X > 1.0) then
      ArcTanh := DefaultVal(DOMAIN)
    else if (X = - 1.0) or (X = 1.0) then
      ArcTanh := Sgn(X) * DefaultVal(SING)
    else
      ArcTanh := 0.5 * System.Ln((1.0 + X) / (1.0 - X));
  end;

