{ **********************************************************************
  *                        Program REGFRAC.PAS                         *
  *                            Version 1.0                             *
  *                     (c) J. Debord, July 1999                       *
  **********************************************************************
  This program performs a nonlinear least squares fit of a rational
  fraction :
                           p0 + p1.x + p2.x^2 + ...
                       y = ------------------------
                           1 + q1.x + q2.x^2 + ...

  The following parameters are passed on the command line :

    1st parameter = Name of input file (default extension = .DAT)
                    The structure of the input file is described in
                    REG_IN.INC
    2nd parameter = Degree of numerator (default = 1)
    3rd parameter = Degree of denominator (default = 1)
    4th parameter = 1 if the function includes a constant term (p0)

  The file ENZYME.DAT is an example of enzyme kinetics (unpublished data
  from the author's laboratory). The best fit is obtained with 2nd-degree
  numerator and denominator, with no constant term, so that the program
  parameters are :

                              ENZYME 2 2

  The program may be executed from Turbo Pascal's integrated environment,
  in which case the parameters are entered through the "Parameters" option
  of the menu, or from DOS (after compilation into an executable file),
  in which case the parameters are entered on the command line (e.g.
  FITFRAC ENZYME 2 2).
  ********************************************************************** }

program RegFrac;

uses
  Crt, Graph, FMath, Matrices, Optim, Regress, Models, PaString, Plot;

const
  MAXITER = 1000;      { Maximal number of iterations }
  TOL     = 1.0E-4;    { Required parameter precision }

var
  InFName : String;    { Name of input file }
  Title   : String;    { Title of study }
  XName,
  YName   : String;    { Variable names }
  N       : Integer;   { Number of points }
  X, Y    : PVector;   { Point coordinates }
  U       : PMatrix;   { Matrix of independent variables (not used here) }
  Ycalc   : PVector;   { Expected Y values }
  S       : PVector;   { Standard deviations of Y values }
  CstPar  : PVector;   { Constant parameters }
  B       : PVector;   { Regression parameters }
  V       : PMatrix;   { Variance-covariance matrix of regression parameters }
  Theta   : PVector;   { Variance parameters }
  RegTest : TRegTest;  { Regression tests }
  ErrCode : Integer;   { Error code }


  procedure ReadCmdLine(var InFName : String; var CstPar : PVector);
{ --------------------------------------------------------------------
  Reads command line parameters. Stores constant parameters in CstPar,
  such that :

       CstPar^[0] = Degree of numerator
       CstPar^[1] = Degree of denominator
       CstPar^[2] = 1 to include a constant term (p0)

  The contents of CstPar are defined in the unit FITFRAC.PAS,
  in the subdirectory REG of the TP Math units directory.
  -------------------------------------------------------------------- }
  var
    I, Deg1, Deg2, ErrCode : Integer;
  begin
    DimVector(CstPar, 2);

    { Name of input file }
    InFName := ParamStr(1);
    if Pos('.', InFName) = 0 then InFName := InFName + '.dat';

    { Degree of numerator }
    Deg1 := 0;
    Val(ParamStr(2), Deg1, ErrCode);
    if (ErrCode <> 0) or (Deg1 < 1) then Deg1 := 1;
    CstPar^[0] := Deg1;

    { Degree of denominator }
    Deg2 := 0;
    Val(ParamStr(3), Deg2, ErrCode);
    if (ErrCode <> 0) or (Deg2 < 1) then Deg2 := 1;
    CstPar^[1] := Deg2;

    { Presence of constant term }
    I := 0;
    Val(ParamStr(4), I, ErrCode);
    CstPar^[2] := I;
  end;


  {$I REG_IN.INC}    { Read input file }

  {$I REG_OUT.INC}   { Write output file }

  {$I REG_PLOT.INC}  { Plot function }

{ *************************** Main program ***************************** }

begin
  { Read command line parameters }
  ReadCmdLine(InFName, CstPar);

  { Read input file }
  if ReadInputFile(InFName, Title, XName, YName, N, X, Y) <> 0 then
    begin
      WriteLn('Error reading file ', InFName);
      Halt;
    end;

  { Initialize regression and variance models.
    See MODELS.PAS in the REG subdirectory for a list of available models }
  InitModel(REG_FRAC,
            VAR_CONST,  { Here we use a constant variance }
            CstPar);

  { Dimension arrays.
    Note: the variance parameters Theta^[1]..Theta^[LastVarParam]
    must be supplied if we use a non-constant variance model }
  DimVector(Theta, LastVarParam);
  DimVector(B, LastParam);
  DimMatrix(V, LastParam, LastParam);
  DimVector(Ycalc, N);
  DimVector(S, N);

  { Perform regression }
  ErrCode := WLSFit(X, U, Y, N, True, MAXITER, TOL,
                    Theta, B, V, Ycalc, S, RegTest);

  { Path to the graphic drivers (Default = C:\BP\BGI) }
  { BGIPath := 'C:\BP\BGI'; }

  { Write results & plot curve }
  case ErrCode of
    MAT_OK     : begin
                   PlotGraph(Title, X, Y, N);
                   WriteOutputFile(InFName, Title, XName, YName,
                                   N, Y, Ycalc, S, B, V, RegTest);
                 end;
    MAT_SINGUL : WriteLn('Singular matrix !');
    BIG_LAMBDA : WriteLn('Too high Marquardt''s parameter !');
    NON_CONV   : WriteLn('Non-convergence !');
  end;
end.
