unit UXlsRangeRecords;

interface
uses UXlsBaseRecords, UXlsBaseRecordLists, UXlsOtherRecords,
     XlsMessages, Classes, SysUtils, UFlxMessages;

type
  TExcelRange= packed record
    R1, R2, C1, C2: word;
  end;
  PExcelRange= ^TExcelRange;

  TRangeValuesList= class(TList) //Items are TExcelRange
  private
    FOtherDataLen :word;
    procedure CopyIntersectRange(const R: PExcelRange; const NewFirstRow, NewLastRow, DestRow, aCount: integer; var MinR1, MaxR2: Word);
  protected
      procedure Notify(Ptr: Pointer; Action: TListNotification); override;
  public
      constructor Create(const aOtherDataLen: word);

      procedure Load(const aRecord: TBaseRecord; const aPos: integer);

      //this methods are to split the record using Continue
      procedure SaveToStreamC(const DataStream: TStream);
      function TotalSizeC: int64;
      function FirstRecordSizeC: integer;

      //this methods are to split the record repeating it
      procedure SaveToStreamR(const DataStream: TStream; const Line: integer);
      function TotalSizeR: int64;
      function RepeatCountR: integer;
      function RecordSizeR(const Line: integer): integer;

      procedure CopyFrom( const RVL: TRangeValuesList);

      procedure ArrangeInsert(const aPos, aCount:integer);

     //Formats are copied if the range intersects with the original. (Merged cells need all the range to be inside the original)
      procedure CopyRowsInclusive(const FirstRow, LastRow, DestRow, aCount: integer; var MinR1, MaxR2: Word);
      procedure CopyRowsExclusive(const FirstRow, LastRow, DestRow, aCount: integer);
      procedure DeleteRows(const aRow, aCount: integer; const Allow1Cell: boolean);
  end;

  TRangeEntry = class
  private
  protected
    RangeValuesList: TRangeValuesList;
    function DoCopyTo: TRangeEntry; virtual;
  public
    constructor Create; virtual;
    destructor Destroy; override;

    function CopyTo: TRangeEntry;

    procedure LoadFromStream( const DataStream: TStream; const First: TRangeRecord);virtual;abstract;
    procedure SaveToStream(const DataStream: TStream);virtual;abstract;
    function TotalSize: int64;virtual; abstract;

    procedure ArrangeInsert(const aPos, aCount:integer; const SheetInfo: TSheetInfo);virtual;
    procedure InsertAndCopyRows(const FirstRow, LastRow, DestRow, aCount: integer; const SheetInfo: TSheetInfo); virtual;
    procedure DeleteRows(const aRow, aCount: word; const SheetInfo: TSheetInfo);virtual;
  end;

  //Merged cells can't be continued. We have to write independent records.
  TMergedCells = class (TRangeEntry)
  public
    constructor Create; override;

    procedure Clear;
    procedure LoadFromStream( const DataStream: TStream; const First: TRangeRecord); override;
    procedure SaveToStream(const DataStream: TStream); override;
    function TotalSize: int64; override;

    procedure InsertAndCopyRows(const FirstRow, LastRow, DestRow, aCount: integer; const SheetInfo: TSheetInfo); override;
    procedure DeleteRows(const aRow, aCount: word; const SheetInfo: TSheetInfo);override;

    function CheckCell(const aRow, aCol: integer; var CellBounds: TXlsCellRange): boolean;
  end;

  ClassOfTRangeEntry = class of TRangeEntry;

implementation

{ TRangeValuesList }

procedure TRangeValuesList.CopyFrom( const RVL: TRangeValuesList);
var
  i: integer;
  R: PExcelRange;
begin
  for i:=0 to RVL.Count-1 do
  begin
    New(R);
    try
      R^:=PExcelRange(RVL[i])^;
      Add(R);
    except
      FreeAndNil(R);
      raise;
    end; //except
  end;
end;

constructor TRangeValuesList.Create(const aOtherDataLen: word);
begin
  inherited Create;
  FOtherDataLen:= aOtherDataLen;
end;

function TRangeValuesList.FirstRecordSizeC: integer;
const
  Rl = SizeOf(TExcelRange);
var
  FirstRecCount, aCount: integer;
begin
  FirstRecCount := ((MaxRecordDataSize-2-FOtherDataLen) div Rl);
  if FirstRecCount<Count then aCount:= FirstRecCount else aCount:=Count;
  Result :=  2+ FOtherDataLen                          //Base data
            + Rl*aCount;                               // Registers

end;

procedure TRangeValuesList.ArrangeInsert(const aPos, aCount: integer);
var
  i:integer;
begin
  for i:=Count -1 downto 0 do
  begin
    if PExcelRange(Items[i]).R1>= aPos then IncMaxMin( PExcelRange(Items[i]).R1, aCount, Max_Rows, aPos);
    if PExcelRange(Items[i]).R2>= aPos then IncMaxMin( PExcelRange(Items[i]).R2, aCount, Max_Rows, PExcelRange(Items[i]).R1);
  end;
end;

procedure TRangeValuesList.CopyIntersectRange(const R: PExcelRange; const NewFirstRow, NewLastRow, DestRow, aCount: integer; var MinR1, MaxR2: Word);
var
  NewRange: PExcelRange;
  k, Lc: integer;
begin
  Lc:=(NewLastRow-NewFirstRow+1)* aCount;

  if (R.R1<=NewFirstRow) and (R.R2>=NewLastRow) then // Just copy one big range
  begin
    New(NewRange);
    try
      NewRange^:=R^;
      NewRange.R1:=DestRow;
      NewRange.R2:=DestRow+Lc-1;
      Add(NewRange);
      if NewRange.R1< MinR1 then MinR1:=NewRange.R1;
      if NewRange.R2> MaxR2 then MaxR2:=NewRange.R2;
    except
      Dispose(NewRange);
      raise;
    end; //Except
  end else // We have to copy one small range for each aCount
  begin
    for k:=0 to aCount -1 do
    begin
      New(NewRange);
      try
        NewRange^:=R^;
        NewRange.R1:=DestRow+(NewLastRow-NewFirstRow+1)*k;
        if R.R1>NewFirstRow then inc(NewRange.R1, R.R1-NewFirstRow);
        NewRange.R2:=DestRow+(NewLastRow-NewFirstRow+1)*(k+1)-1;
        if R.R2<NewLastRow then dec(NewRange.R2, NewLastRow-R.R2);

        Add(NewRange);
        if NewRange.R1< MinR1 then MinR1:=NewRange.R1;
        if NewRange.R2> MaxR2 then MaxR2:=NewRange.R2;
      except
        Dispose(NewRange);
        raise;
      end; //Except
    end;
  end;
end;

procedure TRangeValuesList.CopyRowsInclusive(const FirstRow, LastRow,
  DestRow, aCount: integer; var MinR1, MaxR2: word);
var
  i, Lc:integer;
  R: PExcelRange;
  NewFirstRow, NewLastRow: integer;
begin
  Lc:=(LastRow-FirstRow+1)* aCount;

  if FirstRow<DestRow then NewFirstRow:=FirstRow else NewFirstRow:=FirstRow+ Lc;
  if LastRow<DestRow then NewLastRow:=LastRow else NewLastRow:=LastRow+Lc;

  for i:=0 to Count-1 do
  begin
    R:=PExcelRange(Items[i]);
    if (R.R1<= NewLastRow) and
       (R.R2>= NewFirstRow) then
    begin
      //First Case, Block copied is above the original

      if (FirstRow>=DestRow) then
        if (R.R1<DestRow + Lc) then //nothing, range is automatically expanded
        else if (R.R1=DestRow + Lc) and( R.R2 >=NewLastRow) then //expand the range to include inserted rows
        begin
          Dec(R.R1, Lc);
          if R.R1< MinR1 then MinR1:=R.R1;
        end
        else CopyIntersectRange(R, NewFirstRow, NewLastRow, DestRow, aCount, MinR1, MaxR2) //We have to Copy the intersecting range, and clip the results

      //Second Case, Block copied is below the original

      else
        if (R.R2>DestRow-1) then //nothing, range is automatically expanded
        else if (R.R2=DestRow -1) and (R.R1<=NewFirstRow) then //expand the range to include inserted rows
        begin
          Inc(R.R2, Lc);
          if R.R2> MaxR2 then MaxR2:=R.R2;
        end
        else CopyIntersectRange(R, NewFirstRow, NewLastRow, DestRow, aCount, MinR1, MaxR2); //We have to Copy the intersecting range, and clip the results

    end;
  end;
end;

procedure TRangeValuesList.CopyRowsExclusive(const FirstRow,
  LastRow, DestRow, aCount: integer);
var
  i, k, Lc:integer;
  R, NewRange: PExcelRange;
  NewFirstRow, NewLastRow: integer;
begin
  Lc:=(LastRow-FirstRow+1)* aCount;

  if FirstRow<DestRow then NewFirstRow:=FirstRow else NewFirstRow:=FirstRow+ Lc;
  if LastRow<DestRow then NewLastRow:=LastRow else NewLastRow:=LastRow+Lc;

  for i:=0 to Count-1 do
  begin
    R:=PExcelRange(Items[i]);
    if (R.R1>= NewFirstRow) and
       (R.R2<= NewLastRow) then

      for k:=0 to aCount-1 do
      begin
        New(NewRange);
        try
          NewRange^:=R^;
          if (FirstRow>=DestRow) then
          begin
            IncMax(NewRange.R1, DestRow - FirstRow -(LastRow-FirstRow+1)*(k+1), Max_Rows);
            IncMax(NewRange.R2, DestRow - FirstRow -(LastRow-FirstRow+1)*(k+1), Max_Rows);
          end else
          begin
            IncMax(NewRange.R1, DestRow - FirstRow +(LastRow-FirstRow+1)*k, Max_Rows);
            IncMax(NewRange.R2, DestRow - FirstRow +(LastRow-FirstRow+1)*k, Max_Rows);
          end;
          add(NewRange);
        except
          Dispose(NewRange);
          raise;
        end; //Except
      end;
  end;
end;

procedure TRangeValuesList.DeleteRows(const aRow, aCount: integer; const Allow1Cell: boolean);
var
  i:integer;
  R:PExcelRange;
begin
  for i:=Count-1 downto 0 do
  begin
    R:=PExcelRange(Items[i]);
    if (R.R1>= aRow) and
      ((R.R2< aRow+aCount) or (not Allow1Cell and (R.R2=aRow+aCount) and (R.C1=R.C2))) then
        Delete(i);
  end;
end;


procedure TRangeValuesList.Load(const aRecord: TBaseRecord; const aPos: integer);
var
  i: integer;
  n: word;
  MyPos: integer;
  MyRecord: TBaseRecord;
  ExcelRange: PExcelRange;
begin
  MyPos:= aPos;
  MyRecord:= aRecord;
  ReadMem(MyRecord, MyPos, SizeOf(n), @n);
  for i:=0 to n-1 do
  begin
    New(ExcelRange);
    try
      ReadMem(MyRecord, MyPos, SizeOf(TExcelRange), ExcelRange);
      Add(ExcelRange);
      ExcelRange:=nil;
    finally
      Dispose(ExcelRange);
    end; //finally
  end;
end;

procedure TRangeValuesList.Notify(Ptr: Pointer; Action: TListNotification);
begin
  if Action = lnDeleted then Dispose(PExcelRange(Ptr));
  inherited Notify(Ptr, Action);
end;

procedure TRangeValuesList.SaveToStreamC(const DataStream: TStream);
const
  Rl = SizeOf(TExcelRange);
  OneRecCount = (MaxRecordDataSize div Rl);
var
  RecordHeader: TRecordHeader;
  FirstRecCount, i: integer;
  myCount: word;
begin
  MyCount:=Count;
  DataStream.Write(MyCount, SizeOf(MyCount));

  FirstRecCount := ((MaxRecordDataSize-2-FOtherDataLen) div Rl) ;

  for i:= 0 to Count-1 do
  begin
    if (i>=FirstRecCount) and ((i-FirstRecCount) mod OneRecCount = 0) then
    begin
      //Add continue
      RecordHeader.Id:=xlr_CONTINUE;
      if Count-i> OneRecCount then
        RecordHeader.Size:= OneRecCount * Rl else
        RecordHeader.Size:= (Count-i) * Rl;

      DataStream.Write(RecordHeader, SizeOf(RecordHeader));
    end;
    DataStream.Write(PExcelRange(Items[i])^, Rl);
  end;
end;

function TRangeValuesList.TotalSizeC: int64;
const
  Rl = SizeOf(TExcelRange);
  OneRecCount = (MaxRecordDataSize div Rl);
var
  FirstRecCount: integer;
begin
  Result := SizeOf(TRecordHeader)+ 2+ FOtherDataLen    //Base data
            + Rl*Count;                               // Registers

  //Add Continue Headers...
  FirstRecCount := ((MaxRecordDataSize-2-FOtherDataLen) div Rl);
  if Count > FirstRecCount then
    Result:= Result + SizeOf(TRecordHeader)* ((Count-FirstRecCount-1) div OneRecCount +1);
end;

function TRangeValuesList.RepeatCountR: integer;
const
  Rl = SizeOf(TExcelRange);
var
  OneRecCount: integer;
begin
  OneRecCount := ((MaxRecordDataSize-2-FOtherDataLen) div Rl);
  if Count>0 then Result:= (Count-1) div OneRecCount +1 else Result:=1;
end;

procedure TRangeValuesList.SaveToStreamR(const DataStream: TStream; const Line: integer);
const
  Rl = SizeOf(TExcelRange);
var
  OneRecCount, i: integer;
  myCount: word;
begin
  OneRecCount := ((MaxRecordDataSize-2-FOtherDataLen) div Rl) ;
  if (Line+1)*OneRecCount >Count then MyCount:=Count-Line*OneRecCount else MyCount:=OneRecCount;
  DataStream.Write(MyCount, SizeOf(MyCount));
  for i:=Line*OneRecCount to Line*OneRecCount+myCount-1 do DataStream.Write(PExcelRange(Items[i])^, Rl);
end;

function TRangeValuesList.TotalSizeR: int64;
const
  Rl = SizeOf(TExcelRange);
begin
  Result := (SizeOf(TRecordHeader)+ 2+ FOtherDataLen)* RepeatCountR    //Base data
            + Rl*Count;                               // Registers
end;

function TRangeValuesList.RecordSizeR(const Line: integer): integer;
const
  Rl = SizeOf(TExcelRange);
var
  OneRecCount, MyCount: integer;
begin
  OneRecCount := ((MaxRecordDataSize-2-FOtherDataLen) div Rl) ;
  if (Line+1)*OneRecCount >Count then MyCount:=Count-Line*OneRecCount else MyCount:=OneRecCount;
  Result:= 2+ FOtherDataLen+MyCount*Rl;
end;

{ TRangeEntry }

function TRangeEntry.CopyTo: TRangeEntry;
begin
  if Self=nil then Result:= nil   //for this to work, this cant be a virtual method
  else Result:=DoCopyTo;
end;

constructor TRangeEntry.Create;
begin
  inherited;
end;

destructor TRangeEntry.Destroy;
begin
  FreeAndNil(RangeValuesList);
  inherited;
end;

function TRangeEntry.DoCopyTo: TRangeEntry;
begin
  Result:= ClassOfTRangeEntry(ClassType).Create;
  Result.RangeValuesList.CopyFrom(RangeValuesList);
end;

procedure TRangeEntry.DeleteRows(const aRow, aCount: word; const SheetInfo: TSheetInfo);
begin
  ArrangeInsert(aRow, -aCount, SheetInfo);
end;


procedure TRangeEntry.InsertAndCopyRows(const FirstRow, LastRow, DestRow,
  aCount: integer; const SheetInfo: TSheetInfo);
begin
  ArrangeInsert(DestRow, (LastRow-FirstRow+1)* aCount, SheetInfo);
end;


procedure TRangeEntry.ArrangeInsert(const aPos, aCount:integer; const SheetInfo: TSheetInfo);
begin
  RangeValuesList.ArrangeInsert(aPos, aCount);
end;

{ TMergedCells }

function TMergedCells.CheckCell(const aRow, aCol: integer; var CellBounds: TXlsCellRange): boolean;
var
  i: integer;
begin
  Result:=false;
  for i:=0 to RangeValuesList.Count-1 do
    if (PExcelRange(RangeValuesList[i]).R1<=aRow) and
       (PExcelRange(RangeValuesList[i]).R2>=aRow) and
       (PExcelRange(RangeValuesList[i]).C1<=aCol) and
       (PExcelRange(RangeValuesList[i]).C2>=aCol) then
       begin
         CellBounds.Left:= PExcelRange(RangeValuesList[i]).C1;
         CellBounds.Top:= PExcelRange(RangeValuesList[i]).R1;
         CellBounds.Right:= PExcelRange(RangeValuesList[i]).C2;
         CellBounds.Bottom:= PExcelRange(RangeValuesList[i]).R2;
         Result:=true;
         exit;
       end;

end;

procedure TMergedCells.Clear;
begin
  if RangeValuesList<>nil then RangeValuesList.Clear;
end;

constructor TMergedCells.Create;
begin
  inherited;
  RangeValuesList:= TRangeValuesList.Create(0);
end;

procedure TMergedCells.DeleteRows(const aRow, aCount: word; const SheetInfo: TSheetInfo);
begin
  RangeValuesList.DeleteRows(aRow, acount, false);
  inherited;
end;

procedure TMergedCells.InsertAndCopyRows(const FirstRow, LastRow, DestRow,
  aCount: integer; const SheetInfo: TSheetInfo);
begin
  inherited;
  RangeValuesList.CopyRowsExclusive(FirstRow, LastRow, DestRow, aCount);
end;

procedure TMergedCells.LoadFromStream(const DataStream: TStream;
  const First: TRangeRecord);
var
  aPos: integer;
begin
  Clear;
  aPos:=0;
  RangeValuesList.Load(First, aPos);

  First.Free;
end;

procedure TMergedCells.SaveToStream(const DataStream: TStream);
var
  RecordHeader: TRecordHeader;
  i: integer;
begin
  if RangeValuesList.Count=0 then exit; //don't save empty MergedCells
  RecordHeader.Id:= xlr_CELLMERGING;
  for i:=0 to RangeValuesList.RepeatCountR-1 do
  begin
    RecordHeader.Size:=RangeValuesList.RecordSizeR(i);
    DataStream.Write(RecordHeader, SizeOf(RecordHeader));
    RangeValuesList.SaveToStreamR(DataStream, i);
  end;
end;

function TMergedCells.TotalSize: int64;
begin
  if RangeValuesList.Count=0 then TotalSize:=0 else TotalSize:= RangeValuesList.TotalSizeR;
end;

end.
