(****************************************************************************)
(* UCalcStatDesConditionnalDesc.pas - Copyright (c) 2004 Ricco RAKOTOMALALA *)
(****************************************************************************)

{
@abstract(Description conditionnelle d'une variable continue  partir d'une variable discrte)
@author(Ricco)
@created(12/01/2004)

Cette classe servira entre autres de base pour l'anova  un facteur.

05/08/03, nouvel ajout, le test de Kruskal-Wallis, attention, mthode des rangs alatoires, sinon
on est oblig d'utiliser une statistique et surtout une table diffrente

11/07/2005 -- ajouter d'autres mthodes non-paramtriques (White, Mann-Whitney, etc.)
13/07/2005 -- gestion des ex-aequos

Principales rfrences :
------------------------
Avazian (Elements de modlisation et de traitement primaire des donnes) -- chapitre 11
Bulle (Comparaison de populations) -- chapitre 3
Siegel & Castellan (Nonparametric Statistics for the Behavioral Science -- dition de 1988) -- chapitres 6 et 8

Autre source (web) : http://en.wikipedia.org/wiki/Mann-Whitney_U
et http://www.itl.nist.gov/div898/handbook/eda/section3/eda35.htm
}
unit UCalcStatDesConditionnalDesc;

interface

USES
        UCalcStatDes,
        UDatasetDefinition,
        UDatasetImplementation,
        UDatasetExamples,
        UCalcStatDesRankForNonParametricStat;

TYPE

        //dclaration forward
        TCalcSDCondDesc = class;

        //classe de classe stat. conditionnelles
        TClassCalcSDCondDesc = class of TCalcSDCondDesc;

        {description condtionnelle d'un attribut continu}
        TCalcSDCondDesc = class(TCalcStatDes)
                          private
                          {rcuprer une des stats conditionelles}
                          function    GetStatCond(k: integer): TCalcStatDesContinuous;
                          protected
                          {attribut de description conditionnelle}
                          FAttDesc: TAttDiscrete;
                          {la stat sur l'attribut  dcrire}
                          FStatCont: TCalcStatDesContinuous;
                          {la structure conditionnelle de stat descriptive}
                          FLstStatCont: TLstCalcStatDes;
                          {cration de la structure}
                          procedure   CreateStructureCond(prmExamples: TExamples); virtual;
                          public
                          constructor Create(prmC,prmD: TAttribute; prmExamples: TExamples = nil);
                          destructor  Destroy; override;
                          procedure   BeginUpdate(); override;
                          procedure   AddValue(prmExample: integer); override;
                          procedure   EndUpdate(); override;
                          //rcuprer le nombre de groupes
                          function    getNbGroups(): integer;
                          //affichage HTML
                          function    getHTMLResult(prmOption: integer = -1): string; override;
                          //new -- 16/07/2005 -- critre de tri fond sur la stat -- gnrique --  surcharger absolument chez les hritiers
                          function    getStatForSorting(): double; virtual; abstract;
                          //new -- 16/07/2005 -- critre de tri fond sur la p-value -- gnrique --  surcharger absolument chez les hritiers
                          function    getPValueForSorting(): double; virtual; abstract;
                          {attribut de description conditionnelle}
                          property    AttDescription: TAttDiscrete read FAttDesc;
                          {stat sur tous les individus}
                          property    StatGlobal: TCalcStatDesContinuous read FStatCont;
                          {stat conditionnelles}
                          property    StatCond[k: integer]: TCalcStatDesContinuous read GetStatCond;
                          end;

        {test de Student -- Variances gales dans les 2 groupes -cf. http://www.itl.nist.gov/div898/handbook/eda/section3/eda353.htm}
        TCalcSDTTestEqualVariance = class(TCalcSDCondDesc)
                                    private
                                    //T de Student
                                    FT: double;
                                    //ddl -- c'est un entier ici, mais on veut tre compatible avec le descendant
                                    Fddl: double;
                                    //diffrence
                                    FDiff: double;
                                    //variance et ecart-type estims
                                    FVariance,FStdDev: double;
                                    //p-value
                                    FPValue: double;
                                    protected
                                    //calcul de la variance
                                    function    computeTestVariance(s1,s2: TCalcStatDesContinuous): double; virtual;
                                    //calculer le degr de libert
                                    function    calcDDL(s1,s2: TCalcStatDesContinuous): double; virtual;
                                    //calculer la statistique
                                    procedure   calcTTest();
                                    public
                                    procedure   EndUpdate(); override;
                                    function    getHTMLResult(prmOption: integer = -1): string; override;
                                    function    getStatForSorting(): double; override;
                                    function    getPValueForSorting(): double; override;
                                    end;

        {test de Student -- Variances ingales dans les 2 groupes}
        TCalcSDTTestUnequalVariance = class(TCalcSDTTestEqualVariance)
                                      protected
                                      //calcul de la variance
                                      function    computeTestVariance(s1,s2: TCalcStatDesContinuous): double; override;
                                      //calculer le degr de libert
                                      function    calcDDL(s1,s2: TCalcStatDesContinuous): double; override;
                                      end;


        //description conditionnelle pour l'ANOVA
        TCalcSDCondDescANOVA = class(TCalcSDCondDesc)
                               private
                               {indicateurs sum of squares}
                               FTSS, FBSS, FWSS: double;
                               {les degrs de liberts}
                               Fddl1, Fddl2: integer;
                               {F de Fisher et proba critique}
                               FFisher, FPrFisher: double;
                               {calculer les indicateurs 1-anova}
                               procedure   CalcOneWayAnova();
                               public
                               procedure   EndUpdate(); override;
                               function    getHTMLResult(prmOption: integer = -1): string; override;
                               function    getStatForSorting(): double; override;
                               function    getPValueForSorting(): double; override;
                               {les indicateurs stats}
                               property    TSS: double read FTSS;
                               property    BSS: double read FBSS;
                               property    WSS: double read FWSS;
                               property    ddl1: integer read Fddl1;
                               property    ddl2: integer read Fddl2;
                               property    Fisher: double read FFisher;
                               property    ProbaFisher: double read FPrFisher;
                               end;

        //tableau de conservation des rangs calculs pour chaque population (modalit de la variable de description)
        TTypeSumRankPop = array of double;

        {description conditionnelle calculant des indicateurs non-paramtriques,
        le KW de Kruskal-Wallis en l'occurence}
        TCalcSDNonParamCondDesc = class(TCalcSDCondDesc)
                                  private
                                  {KW de Kruskal-Wallis et Proba associe}
                                  FKW, FPrKW: double;
                                  {KW corrig pour les ex-aequos}
                                  FCKW, FProbaCKW: double;
                                  {degr de libert}
                                  FDDL: integer;
                                  protected
                                  {correction pour le traitement des ex-aequos}
                                  FExAequoCorrection: double;
                                  {tableau global des rangs pour chaque observation -- sera utilis pour le calcul des rangs moyens}
                                  FTabGlobalRank: TRankComputed;
                                  {Tableau de rang pour chaque modalit du descripteur}
                                  FRankTab: TTypeSumRankPop;
                                  {Somme des rangs}
                                  FSumRank: double;
                                  {cration de la structure}
                                  procedure   CreateStructureCond(prmExamples: TExamples); override;
                                  {calcul de la statistique de Kruskal-Wallis}
                                  procedure   CalcNonParamStat(); virtual;
                                  {description de la statistique de test}
                                  function    getHTMLStatTest(): string; virtual;
                                  {calculer la somme des rangs pour chaque population}
                                  procedure   computeRankForPopulation(lst: TExamples);
                                  public
                                  {dtruire le tableau interne}
                                  destructor  Destroy; override;
                                  {calcul stat incluant aussi le Kruskall-Wallis}
                                  procedure   RefreshStat(prmExamples: TExamples); override;
                                  {envoyer le rapport}
                                  function    getHTMLResult(prmOption: integer = -1): string; override;
                                  //critre de tri fond sur la stat
                                  function    getStatForSorting(): double; override;
                                  //critre de tri fond sur la p-value
                                  function    getPValueForSorting(): double; override;
                                  {stat de KW}
                                  property    KW: double read FKW;
                                  {proba critique}
                                  property    ProbaKW: double read FPrKW;
                                  {stat corrig pour les ties}
                                  property    CKW: double read FCKW;
                                  {proba critique}
                                  property    ProbaCKW: double read FProbaCKW;
                                  {tableau de rangs}
                                  property    RankTab: TTypeSumRankPop read FRankTab;
                                  {somme des rangs}
                                  property    SumRank: double read FSumRank;
                                  end;

        {calcul du test de White & Mann-Whitney : cas particulier de KW pour la comparaison de 2 populations}
        TCalcSDNonParamCondDescBinary = class(TCalcSDNonParamCondDesc)
                                        private
                                        //statistique de Withe
                                        FWhite, FZWhite, FProbaWhite: double;
                                        //statistique de Mann-Whitney
                                        FMWhitney, FZMWhitney, FProbaMWhitney: double;
                                        //z corrig pour Mann-Whitney
                                        FZCMWhitney, FProbaCMWhitney: double;
                                        protected
                                        procedure   CalcNonParamStat(); override;
                                        function    getHTMLStatTest(): string; override;
                                        public
                                        //critre de tri fond sur la stat
                                        function    getStatForSorting(): double; override;
                                        //critre de tri fond sur la p-value
                                        function    getPValueForSorting(): double; override;
                                        //proprits
                                        property    White: double read FWhite;
                                        property    ZWhite: double read FZWhite;
                                        property    ProbaWhite: double read FProbaWhite;
                                        property    MWhitney: double read FMWhitney;
                                        property    ZMWhitney: double read FZMWhitney;
                                        property    ProbaMWhitney: double read FProbaMWhitney;
                                        property    ZCMWhitney: double read FZCMWhitney;
                                        property    ProbaCMWhitney: double read FProbaCMWhitney;
                                        end;

        {***********************************************************************}
        {***************** Liste de statistiques *******************************}
        {***********************************************************************}

        {liste de gnrique des stats conditionnelles}
        TLstCalcStatDesCond = class(TLstCalcStatDes)
                              end;

        {classe de classe liste gnrique de stat. conditionnelles}
        TClassLstCalcStatDesCond = class of TLstCalcStatDesCond;


        {liste de rsultats de stat des conditionnelles, critres de tri supplmentaires :
        1 -> nom de l'attribut de description,
        2 -> la statistique, tri invers
        3 -> la proba critique, tri non invers}
        TLstCalcStatDesConditionnal = class(TLstCalcStatDesCond)
                                      public
                                      {comparaison pour les stats conditionnelles}
                                      procedure SortStats(); override;
                                      end;

        //liste de stat. pour le T-Test (comparaison de moyennes)
        TLstStatDescCondTTest = class(TLstCalcStatDesConditionnal)
                                public
                                function  getHeaderHTML(): string; override;
                                end;


        //liste pour l'ANOVA
        TLstStatDesCondANOVA = class(TLstCalcStatDesConditionnal)
                               public
                               function    getHeaderHTML(): string; override;
                               end;

        {liste de rsultats de test non-paramtrique Kruskal-Wallis, critres de tri :
        1 -> nom de l'attribut de description,
        2 -> indicateur de KW, tri invers,
        3 -> proba critique de KW}
        TLstCalcStatDesCondKW = class(TLstCalcStatDesCond)
                                public
                                function  getHeaderHTML(): string; override;
                                end;

        //liste de stats. Mann & Whitney
        TLstCalcStatDesCondMWhitney = class(TLstCalcStatDesCond)
                                      end;


//fonction introduisant un code couleur diffrent pour les tests significatifs
function codeCouleur(valueProba: double): string;

implementation

uses
        FMath,
        Math,
        Classes, Sysutils,
        ULogFile, UConstConfiguration, UCalcRndGenerator;

//******************************************
//fonction pour la mise en forme des sorties
//******************************************
function codeCouleur(valueProba: double): string;
begin
 if (valueProba < SIGNIFICANCE_LEVEL_FOR_TESTS)
  then result:= HTML_BGCOLOR_DATA_RED
  else result:= HTML_BGCOLOR_DATA_BLUE;
end;
//*******************************************

{******************* fonction de comparaison pour le tri *********************}

{trier selon les noms d'attributs}
function ListSortCompareNameDescription(item1,item2: pointer): integer;
var st1,st2: TCalcSDCondDesc;
begin
 st1:= TCalcSDCondDesc(item1);
 st2:= TCalcSDCondDesc(item2);
 if (st1.AttDescription.Name<st2.AttDescription.Name)
  then result:= -1
  else
   if (st1.AttDescription.Name>st2.AttDescription.Name)
    then result:= +1
    else result:= 0;
end;

function ListSortCompareStat(item1,item2: pointer): integer;
var st1,st2: TCalcSDCondDesc;
begin
 st1:= TCalcSDCondDesc(item1);
 st2:= TCalcSDCondDesc(item2);
 //tri invers !!!
 if (st1.getStatForSorting()<st2.getStatForSorting())
  then result:= +1
  else
   if (st1.getStatForSorting()>st2.getStatForSorting())
    then result:= -1
    else result:= 0;
end;

function ListSortComparePValue(item1,item2: pointer): integer;
var st1,st2: TCalcSDCondDesc;
begin
 st1:= TCalcSDCondDesc(item1);
 st2:= TCalcSDCondDesc(item2);
 if (st1.getPValueForSorting()<st2.getPValueForSorting())
  then result:= -1
  else
   if (st1.getPValueForSorting()>st2.getPValueForSorting())
    then result:= +1
    else result:= 0;
end;

{*****************************************************************************}

{ TCalcSDCondDesc }

procedure TCalcSDCondDesc.AddValue(prmExample: integer);
var k: TTypeDiscrete;
begin
 inc(FNbExamples);
 FStatCont.AddValue(prmExample);
 //rechercher la stat  incrmenter
 k:= AttDescription.dValue[prmExample];
 Self.StatCond[pred(k)].AddValue(prmExample);
end;

procedure TCalcSDCondDesc.BeginUpdate;
begin
 FNbExamples:= 0;
 FStatCont.BeginUpdate();
 FLstStatCont.BeginUpdate();
end;


constructor TCalcSDCondDesc.Create(prmC, prmD: TAttribute;
  prmExamples: TExamples);
begin
 inherited Create(prmC,nil);
 FAttDesc:= prmD as TAttDiscrete;
 self.CreateStructureCond(prmExamples);
end;

procedure TCalcSDCondDesc.CreateStructureCond(prmExamples: TExamples);
var i: integer;
    FSameAtts: TLstAttributes;
begin
 //stat globale sur l'attribut continu
 FStatCont:= TCalcStatDesContinuous.Create(Attribute,prmExamples);
 //cration d'une liste fictive d'attributs
 //sachant que c'est le mme attribut que l'on ajoute systmatiquement
 //astuce un peu norme mais bon...
 FSameAtts:= TLstAttributes.Create(FALSE,Attribute.Size);
 for i:= 0 to pred(FAttDesc.nbValues) do
  FSameAtts.Add(Attribute);
 FLstStatCont:= TLstCalcStatDesContinuous.Create(FSameAtts,NIL);
 FSameAtts.Free;
 //puis remplir si demand
 if assigned(prmExamples)
  then self.RefreshStat(prmExamples);
end;

destructor TCalcSDCondDesc.Destroy;
begin
 FLstStatCont.Free;
 inherited;
end;

procedure TCalcSDCondDesc.EndUpdate;
begin
 FStatCont.EndUpdate();
 FLstStatCont.EndUpdate();
end;

function TCalcSDCondDesc.getHTMLResult(prmOption: integer): string;
var s: string;
    k: integer;
begin
 //description sur une variable
 s:= format('<TD>%s</TD><TD>%s</TD>',[self.Attribute.Name,self.AttDescription.Name]);
 //les stat descriptives conditionelles
 s:= s+'<TD valign="top">'+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_BLUE+
       '<TH width=120>Value</TH><TH width=70>Examples</TH><TH width=70>Average</TH><TH width=70>Std-dev</TH></TR>';
 //pour chaque modalit de la variable de description
 for k:= 0 to pred(self.AttDescription.nbValues) do
  begin
   s:= s+HTML_TABLE_COLOR_DATA_BLUE+
         format('<TH>%s</TH><TD align=right>%d</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</td>'+'<td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</td>',
                [self.AttDescription.LstValues.GetDescription(succ(k)),StatCond[k].NbExamples,StatCond[k].Average,StatCond[k].StdDev])+'</tr>';
  end;
 //stat globales (sur tous les individus)
 s:= s+HTML_TABLE_COLOR_DATA_GREEN+
       format('<TH>All</TH><TD align=right>%d</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</td>'+'<td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</td>',
              [self.NbExamples,self.StatGlobal.Average,self.StatGlobal.StdDev])+'</tr>';
 s:= s+'</table></td>';
 //renvoyer le tout
 result:= s;
end;

function TCalcSDCondDesc.getNbGroups: integer;
begin
 result:= FLstStatCont.Count;
end;

function TCalcSDCondDesc.GetStatCond(k: integer): TCalcStatDesContinuous;
begin
 result:= TCalcStatDesContinuous(FLstStatCont.Stat(k));
end;

{ TLstCalcStatDesConditionnal }

procedure TLstCalcStatDesConditionnal.SortStats;
var funcCompare: TListSortCompare;
begin
 if (CompareMode>=0)
  then
   begin

    case CompareMode of
     0: funcCompare:= ListSortCompareName;
     1: funcCompare:= ListSortCompareNameDescription;
     2: funcCompare:= ListSortCompareStat;
     3: funcCompare:= ListSortComparePValue;
     else
      funcCompare:= NIL;
    end;

    if assigned(funcCompare)
     then LstStat.Sort(funcCompare);
   end;
end;

{ TCalcSDNonParamCondDesc }

procedure TCalcSDNonParamCondDesc.CalcNonParamStat();
var k: integer;
    statk: TCalcStatDesContinuous;
    correction: double;
begin
 FKW:= 0.0;
 for k:= 0 to pred(FLstStatCont.Count) do
  begin
   statk:= TCalcStatDesContinuous(FLstStatCont.Stat(k));
   //KW
   if (statk.NbExamples>0)
    then FKW:= FKW+FRankTab[k]*FRankTab[k]/(1.0*statk.NbExamples);
  end;
 //degrs de libert
 FDDL:= pred(FLstStatCont.Count);//numrateur
 //stat de KW
 FKW:= 12.0/(1.0*FNbExamples*(1.0+FNbExamples))*FKW-3.0*(1.0+FNbExamples);
 //proba associe, cf. J. Debord
 FPrKW:= PKHI2(FDDL,FKW);
 //introduction de la correction pour le traitement des ex-aequos -- Siegel p.210
 correction:= 1.0 - FExAequoCorrection/(power(1.0*self.NbExamples,3.0)-1.0*self.NbExamples);
 FCKW:= 0;
 FProbaCKW:= 1.0;
 if (correction > 0)
  then
   begin
    FCKW:= FKW/correction;
    FProbaCKW:= PKHI2(FDDL,FCKW);
   end;
end;

function TCalcSDNonParamCondDesc.getPValueForSorting: double;
begin
 result:= self.ProbaCKW;
end;

function TCalcSDNonParamCondDesc.getStatForSorting: double;
begin
 result:= self.CKW;
end;

{ TCalcSDNonParamCondDescBinary }

procedure TCalcSDNonParamCondDescBinary.CalcNonParamStat;
var statA,statB: TCalcStatDesContinuous;
    WAMoy,WAVar: double;
    UCentered,UMoy,UVar,UVarC: double;
    kRef: integer;
    N: double;
begin
 //rechercher la modalit avec l'effectif la plus faible, elle va servir de rf. (cf. Bulle, pp.30-31)
 if (TCalcStatDesContinuous(FLstStatCont.Stat(0)).NbExamples <= TCalcStatDesContinuous(FLstStatCont.Stat(1)).NbExamples)
  then
   begin
    kRef:= 0;
    statA:= TCalcStatDesContinuous(FLstStatCont.Stat(0));
    statB:= TCalcStatDesContinuous(FLstStatCont.Stat(1));
   end
  else
   begin
    kRef:= 1;
    statA:= TCalcStatDesContinuous(FLstStatCont.Stat(1));
    statB:= TCalcStatDesContinuous(FLstStatCont.Stat(0));
   end;
 //effectif total
 N:= statA.NbExamples + statB.NbExamples;
 //z de White et proba (dans le livre de Siegel - pp.135, il appelle cette stat. Wilcoxon)
 FWhite:= FRankTab[kRef];
 WAMoy:= 0.5*statA.NbExamples*(1.0+N);
 WAVar:= (1.0*statA.NbExamples*statB.NbExamples)/(12.0*N*(N-1.0));
 WAVar:= WAVar*((power(N,3.0) - N) - FExAequoCorrection);
 if (WAVar > 0.0)
  then FZWhite:= (FWhite-WAMoy)/SQRT(WAVar)
  else FZWhite:= 0.0;
 //test bilatral
 FProbaWhite:= PNorm(abs(FZWhite));
 //la stat. de Mann-Whitney (cf. Avazian, Bulle et Siegel pp.132--135)
 // comparer avec STATISTICA et Open Stat V.3 et V.4
 FMWhitney:= FWhite - 0.5*statA.NbExamples*(1.0+statA.NbExamples);
 //calcul du Z
 UMoy:= 0.5*statA.NbExamples*statB.NbExamples;
 //calcul de la stat. centre (cf. STATISTICA -- pas besoin d'introduire la correction de continuit)
 UCentered:= FMWhitney - UMoy;
 //Z de stat de Mann & Whitney non corrig
 UVar:= 1.0*statA.NbExamples*statB.NbExamples*(N+1.0)/12.0;
 if (UVar > 0.0)
  then FZMWhitney:= abs(UCentered)/sqrt(UVar)
  else FZMWhitney:= 0.0;
 FProbaMWhitney:= PNorm(abs(FZMWhitney));
 //calcul Z de la formule corrige -- Siegel -- pp.135 (mme variance que Wilcoxon)
 UVarC:= WAVar;
 if (UVarC > 0.0)
  then FZCMWhitney:= abs(UCentered)/sqrt(UVarC)
  else FZCMWhitney:= 0.0;
 FProbaCMWhitney:= PNorm(abs(FZCMWhitney));
end;

procedure TCalcSDNonParamCondDesc.CreateStructureCond(
  prmExamples: TExamples);
begin
 inherited CreateStructureCond(prmExamples);
 //tableau pour les rangs, taille du tableau [0 .. nb de valeurs]
 //mme structure que la liste conditionnelle de stat
 SetLength(FRankTab,FAttDesc.nbValues);
end;

destructor TCalcSDNonParamCondDesc.Destroy;
begin
 setLength(FRankTab,0);
 inherited Destroy;
end;

function TCalcSDNonParamCondDesc.getHTMLResult(prmOption: integer): string;
var s: string;
    k: integer;
begin
 //description sur une variable
 s:= format('<TD>%s</TD><TD>%s</TD>',[self.Attribute.Name,self.AttDescription.Name]);
 //les stat descriptives conditionelles
 s:= s+'<TD>'+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_BLUE+
       '<TH width=120>Value</TH><TH width=80>Examples</TH><TH width=80>Average</TH><TH width=80>Rank sum</TH></TR>';
 //pour chaque modalit de la variable de description
 for k:= 0 to pred(self.AttDescription.nbValues) do
  begin
   s:= s+HTML_TABLE_COLOR_DATA_BLUE+
         format('<TH>%s</TH><TD align=right>%d</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+
         '</td><td align=right>%.1f</td>',
         [self.AttDescription.LstValues.GetDescription(succ(k)),statCond[k].NbExamples,statCond[k].Average,self.RankTab[k]])+'</tr>';
  end;
 //stat globales (sur tous les individus)
 s:= s+HTML_TABLE_COLOR_DATA_GREEN+
       format('<TH>All</TH><TD align=right>%d</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</td><td align=right>%.1f</td>',
              [self.NbExamples,self.StatGlobal.Average,self.SumRank])+'</tr>';
 s:= s+'</table></td>';
 //tests statistiques
 s:= s+'<TD valign=top>';

 s:= s+self.getHTMLStatTest();

 s:= s+'</TD>';

 result:= s;
end;

function TCalcSDNonParamCondDesc.getHTMLStatTest: string;
var s: string;
begin
 s:= HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_BLUE+
       '<TH>Statistics</TH><TH>Value</TH><TH>Proba</TH></TR>';
 //stat de Kruskal-Wallis -- intoroudction du code couleur
 s:= s+format(HTML_TABLE_COLOR_DATA_BLUE+'<TD>%s</TD><TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+
              '</TD><TD align=right %s>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',['Kruskal-Wallis',self.KW,codeCouleur(self.ProbaKW),self.ProbaKW]);
 //stat corrige -- introduction du code couleur
 s:= s+format(HTML_TABLE_COLOR_DATA_BLUE+'<TD>%s</TD><TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+
              '</TD><TD align=right %s>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',['KW corrected',self.CKW,codeCouleur(self.ProbaCKW),self.ProbaCKW]);
 s:= s+'</table>';
 result:= s;
end;

function TCalcSDNonParamCondDescBinary.getHTMLStatTest: string;
var s: string;
begin
 s:= HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_BLUE+
       '<TH>Statistics</TH><TH>Value</TH><TH>Z-Value</TH><TH>Proba</TH></TR>';

 //White (ou Wilcoxon) n'est pas affich finalement -- on se cale sur la prsentation de STATISTICA

 //Mann-Whitney
 s:= s+format(HTML_TABLE_COLOR_DATA_BLUE+'<TD>%s</TD><TD align=right>'+
              STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD><TD align=right>'+
              STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD><TD align=right %s>'+
              STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',['Mann-Whitney',self.MWhitney,self.ZMWhitney,codeCouleur(self.ProbaMWhitney),self.ProbaMWhitney]);
 //Mann-Whitney corrig
 s:= s+format(HTML_TABLE_COLOR_DATA_BLUE+'<TD>%s</TD><TD align=right>'+
              STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD><TD align=right>'+
              STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD><TD align=right %s>'+
              STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',['MW corrected',self.MWhitney,self.ZCMWhitney,codeCouleur(self.ProbaCMWhitney),self.ProbaCMWhitney]);
 //
 s:= s+'</table>';
 result:= s;
end;


procedure TCalcSDNonParamCondDesc.RefreshStat(prmExamples: TExamples);
var lstSorted: TExamples;
begin
 //les calculs paramtriques, cela induit 2 passages sur les donnes mais je ne vois pas
 //comment on pourrait faire autrement  moins de fusionner les classes, mais c'est pas classe justement
 //de toute manire c'est le tri qui ralentit le plus ici, le premier RefreshStat est en O(n)
 inherited RefreshStat(prmExamples);
 //crer une liste trie des individus
 lstSorted:= TExamples.Create(prmExamples.Size);
 lstSorted.Copy(prmExamples);
 //puis trier selon les X, le rang sur Y est alatoire (on s'en fout)
 lstSorted.QuickSortBy(self.Attribute);
 //affecter le rang de chaque observation -- corriger avec le rang moyen
 FTabGlobalRank:= TRankComputed.create(self.Attribute);
 FExAequoCorrection:= FTabGlobalRank.computeRank(lstSorted);
 //calculer alors le rang pour chaque modalit de l'attribut de description
 self.computeRankForPopulation(lstSorted);
 //vider la mmoire
 FTabGlobalRank.Free();
 lstSorted.Free;
 //calculer les indicateurs stat
 self.CalcNonParamStat();
end;

procedure TCalcSDNonParamCondDesc.computeRankForPopulation(lst: TExamples);
var k,i: integer;
begin
 for k:= low(FRankTab) to high(FRankTab) do FRankTab[k]:= 0.0;
 //additionner les rangs pour chaque observation
 for i:= 1 to lst.Size do
  begin
   k:= pred(self.AttDescription.dValue[lst.Number[i]]);
   //c'est ici qu'on prend en compte la nouvelle formule des rangs moyens
   FRankTab[k]:= FRankTab[k] + FTabGlobalRank.getRank(lst.Number[i]);
  end;
 //calculer la somme totale des rangs
 FSumRank:= 0.0;
 for k:= low(FRankTab) to high(FRankTab) do FSumRank:= FSumRank + FRankTab[k];
end;



function TCalcSDNonParamCondDescBinary.getPValueForSorting: double;
begin
 result:= self.ProbaCMWhitney;
end;

function TCalcSDNonParamCondDescBinary.getStatForSorting: double;
begin
 result:= abs(self.ZCMWhitney);
end;

{ TLstCalcStatDesCondKW }

function TLstCalcStatDesCondKW.getHeaderHTML: string;
begin
 result:= HTML_TABLE_COLOR_HEADER_GRAY+
          '<TH>Attribute_Y</TH><TH>Attribute_X</TH><TH>Description</TH><TH>Statistical test</TH></TR>';
end;

{ TCalcSDCondDescANOVA }

procedure TCalcSDCondDescANOVA.CalcOneWayAnova;
var k: integer;
    statk: TCalcStatDesContinuous;
    avg,avgk: double;
begin
 //stat globales
 FTSS:= FStatCont.TSS;
 avg:= FStatCont.Average;
 //stat conditionnelles
 FBSS:= 0.0;
 FWSS:= 0.0;
 for k:= 0 to pred(FLstStatCont.Count) do
  begin
   statk:= TCalcStatDesContinuous(FLstStatCont.Stat(k));
   avgk:= statk.Average;
   //variance des moyennes
   FBSS:= FBSS+(1.0*statk.NbExamples)*(avgk-avg)*(avgk-avg);
   //moyenne des variances (somme des TSS en gros)
   FWSS:= FWSS+statk.TSS;
  end;
 //degrs de libert
 Fddl1:= pred(FLstStatCont.Count);//numrateur
 Fddl2:= FStatCont.NbExamples-FLstStatCont.Count;//dnominateur
 //stat de Fisher
 if (Fddl1>0) and (Fddl2>0)
  then
   begin
     FFisher:= (FBSS/(1.0*Fddl1))/(FWSS/(1.0*Fddl2));
     //proba cf. la biblio de J. Debord
     //function PSnedecor(Nu1, Nu2 : Integer; X : Float) : Float;  { Prob(F >= X) }
     FPrFisher:= PSnedecor(Fddl1,Fddl2,FFisher);
   end
  else
   begin
    FFisher:= 0.0;
    FPrFisher:= 1.0;
   end;
end;

procedure TCalcSDCondDescANOVA.EndUpdate;
begin
 inherited EndUpdate();
 //calcul ANOVA
 self.CalcOneWayAnova();
end;

function TCalcSDCondDescANOVA.getHTMLResult(prmOption: integer): string;
var s: string;
begin
 //description des effectifs et moyennes conditionnelles
 s:= inherited getHTMLResult(prmOption);

 //*****************
 //information ANOVA
 //*****************


 s:= s+'<TD valign="top">'+HTML_HEADER_TABLE_RESULT;

 //dcomposition de la variance
 s:= s+HTML_TABLE_COLOR_HEADER_BLUE+'<TH colspan=3>Variance decomposition</TH></TR>';
 s:= s+HTML_TABLE_COLOR_HEADER_GREEN+'<TH>Source</TH><TH width=100>Sum of square</TH><TH>d.f.</TH>';
 //variance explique
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+
       format('<TD>BSS</TD><TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</TD><TD align=right>%d</TD></TR>',[self.BSS,self.ddl1]);
 //variance rsiduelle
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+
       format('<TD>WSS</TD><TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</TD><TD align=right>%d</TD></TR>',[self.WSS,self.ddl2]);
 //variance totale
 s:= s+HTML_TABLE_COLOR_DATA_GREEN+
       format('<TD>TSS</TD><TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</TD><TD align=right>%d</TD></TR>',[self.TSS,self.ddl1+self.ddl2]);

 //tests statistiques
 s:= s+HTML_TABLE_COLOR_HEADER_BLUE+'<TH colspan=3>Significance level</TH></TR>';
 s:= s+HTML_TABLE_COLOR_HEADER_GREEN+'<TH>Statistics</TH><TH>Value</TH><TH>Proba</TH></TR>';
 //stat de Fisher
 s:= s+format(HTML_TABLE_COLOR_DATA_BLUE+'<TD>%s</TD><TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD><TD align=right %s>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',
       ['Fisher''s F',self.Fisher,codeCouleur(self.ProbaFisher),self.ProbaFisher]);

 s:= s+'</table>';
 s:= s+'</TD>';
 
 //renvoyer le tout...
 result:= s;
end;

function TCalcSDCondDescANOVA.getPValueForSorting: double;
begin
 result:= self.ProbaFisher;
end;

function TCalcSDCondDescANOVA.getStatForSorting: double;
begin
 result:= self.Fisher;
end;

{ TLstStatDesCondANOVA }

function TLstStatDesCondANOVA.getHeaderHTML: string;
begin
 result:= HTML_TABLE_COLOR_HEADER_GRAY+'<TH>Attribute_Y</TH><TH>Attribute_X</TH><TH>Description</TH><TH>Statistical test</TH></TR>';
end;

{ TCalcSDTTestEqualVariance }

function TCalcSDTTestEqualVariance.calcDDL(s1,s2: TCalcStatDesContinuous): double;
begin
 result:= -2.0 + s1.NbExamples + s2.NbExamples;
end;

procedure TCalcSDTTestEqualVariance.calcTTest;
var st1, st2: TCalcStatDesContinuous;
begin
 //il ne peut y avoir que 2 modalits
 st1:= self.StatCond[0];
 st2:= self.StatCond[1];
 //vrifier
 if (st1.NbExamples > 0) and (st2.NbExamples > 0)
  then
   begin
     //la diffrence
     FDiff:= st1.Average - st2.Average;
     //variance
     FVariance:= self.computeTestVariance(st1,st2);
     //le T de Student
     if (FVariance > 0.0)
      then
       begin
        FStdDev:= SQRT(FVariance);
        FT:= FDiff / FStdDev;
       end
      else
       begin
        FStdDev:= 0.0;
        FT:= 0.0;
       end;
     //le degr de libert
     Fddl:= self.calcDDL(st1,st2);
     //la p-value
     FPValue:= PStudent(ROUND(FDDL),abs(FT));
   end
  else
   begin
    FT:= 0.0;
    Fddl:= 0.0;
    FPValue:= 1.0;
   end;
end;

function TCalcSDTTestEqualVariance.computeTestVariance(s1,s2: TCalcStatDesContinuous): double;
var pooledVar: double;
begin
 //variance commune
 pooledVar:= ((-1.0 + s1.NbExamples) * s1.Variance + (-1.0 + s2.NbExamples) * s2.Variance) / (-2.0 + s1.NbExamples + s2.NbExamples);
 //variance de la statistique
 result:= pooledVar * (1.0/(1.0 * s1.NbExamples) + 1.0/(1.0 * s2.NbExamples));
end;

procedure TCalcSDTTestEqualVariance.EndUpdate;
begin
 inherited EndUpdate();
 //appel de la mthode de calcul
 self.calcTTest();
end;

function TCalcSDTTestEqualVariance.getHTMLResult(
  prmOption: integer): string;
var s: string;
begin

 //description des effectifs et moyennes conditionnelles
 s:= inherited getHTMLResult(prmOption);

 //informations T de Student de comparaison de moyennes

 s:= s+'<TD valign=top>'+HTML_HEADER_TABLE_RESULT;
 //T de Student
 s:= s+format(HTML_TABLE_COLOR_DATA_BLUE+'<TD>T</TD><TD align=right>%.4f / %.4f = '+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',
                                         [FDiff,FStdDev,FT]);
 //ddl
 s:= s+format(HTML_TABLE_COLOR_DATA_BLUE+'<TD>d.f.</TD><TD align=right>%.2f</TD></TR>',[Fddl]);
 //p-value
 s:= s+format(HTML_TABLE_COLOR_DATA_BLUE+'<TD>p-value</TD><TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',[FPValue]);
 //end of the table
 s:= s+'</table></TD>';
 //and then...
 result:= s;
end;

function TCalcSDTTestEqualVariance.getPValueForSorting: double;
begin
 result:= ABS(FPValue);
end;

function TCalcSDTTestEqualVariance.getStatForSorting: double;
begin
 result:= ABS(FT);
end;

{ TLstStatDescCondTTest }

function TLstStatDescCondTTest.getHeaderHTML: string;
begin
 result:= HTML_TABLE_COLOR_HEADER_GRAY+
          '<TH>Attribute_Y</TH><TH>Attribute_X</TH><TH>Description</TH><TH>Statistical test</TH></TR>';
end;

{ TCalcSDTTestUnequalVariance }

function TCalcSDTTestUnequalVariance.calcDDL(s1,
  s2: TCalcStatDesContinuous): double;
var v1, v2: double;
begin
 //moins simplement...
 v1:= 1.0 * s1.Variance / (1.0 * s1.NbExamples);
 v2:= 1.0 * s2.Variance / (1.0 * s2.NbExamples);
 //
 result:= power(v1 + v2,2.0)/(power(v1,2.0)/(-1.0+s1.NbExamples)+power(v2,2.0)/(-1.0+s2.NbExamples));
end;

function TCalcSDTTestUnequalVariance.computeTestVariance(s1,
  s2: TCalcStatDesContinuous): double;
begin
 //trs simplement...
 result:= s1.Variance / (1.0 * s1.NbExamples) + s2.Variance / (1.0 * s2.NbExamples);
end;

end.
