(******************************************************************)
(* UCalcStatDesManova.pas - Copyright (c) 2004 Ricco RAKOTOMALALA *)
(******************************************************************)

{
@abstract(Description conditionnelle d'une liste de variables continues  partir d'une variable discrte)
@author(Ricco)
@created(07/08/2005)

Cette classe de base sera utilise pour l'analyse de variance multi-dimensionnelle. Son principal
intrt est qu'il permet de calculer le Lambda de Wilks et les stats associes,  utiliser donc
dans les classes de calcul tel que l'analyse discriminante, etc.

A voir aussi s'il est possible d'intgrer les tests non-paramtriques.

}
unit UCalcStatDesManova;

interface


USES
    UCalcStatDes,
    UDatasetDefinition,
    UDatasetImplementation,
    UDatasetExamples,
    Matrices,
    Contnrs;    

TYPE
    //classe abstraite -- dfinit les paramtres  passer
    TCalcSDCondMultivariate = class(TCalcStatDes)
                              protected
                              //la liste des attributs  dcrire
                              FDescriptors: TLstAttributes;
                              //stats conditionnelles (par rapport  la variable groupe) --> liste de liste de stats
                              FStatGroups: TObjectList;
                              //stat globales sur les descripteurs
                              FStatGlobal: TLstCalcStatDesContinuous;
                              //tableau des effectifs par groupe
                              FTabSizeGroups: array of integer;
                              //calculer les stats descriptives marginales
                              procedure   computeMultivariateStatDes(prmExamples: TExamples); virtual;
                              //calculer les stats descriptives conditionnelles -- le paramtre est une liste de TExamples
                              procedure   computeMultivariateCondStatDes(lstExamples: TObjectList); virtual;
                              //calculer les statistiques de test
                              procedure   computeStatisticalTests(); virtual; abstract;
                              public
                              //dcrire les stats-conditionnelles en HTML
                              function    getHTMLMultivariateStatDes(): string;
                              //dcrire les statistiques de tests associes  la classe
                              function    getHTMLStatisticalTests(): string; virtual; abstract;
                              //passer les paramtres, et initier les calculs ventuellement
                              constructor create(prmD: TAttribute; prmLstAtts: TLstAttributes; prmExamples: TExamples = nil);
                              //vider les structures internes
                              destructor  destroy(); override;
                              //lancer les calculs
                              procedure   RefreshStat(prmExamples: TExamples); override;
                              //dcrire les rsultats
                              function    getHTMLResult(prmOption: integer = -1): string; override;
                              //accs  une stat de groupe
                              function    getStatGroup(k: integer): TLstCalcStatDesContinuous; 
                              end;

    //classe de classe
    TClassCalcSDCondMultivariate = class of TCalcSDCondMultivariate; 

    //Manova
    TCalcSDCondManova = class(TCalcSDCondMultivariate)
                        private
                        //statistique de Wilks
                        FWilks: double;
                        //transformation de Bartlett et la p-value associe
                        FBartlett,FPValueBartlett: double;
                        //ddl. Bartlett
                        FDDLBartlett: integer;
                        //transformation de Rao
                        FRao, FPValueRao: double;
                        //ddl. Rao --> on peut ne pas avoir des valeurs entires ! donc on a ici les valeurs tronques
                        FDDLNum,FDDLDenom: integer;
                        //Stat. de Box pour l'homogneit des matrices de var-covar --> cf. http://www.rrz.uni-hamburg.de/RRZ/Software/SPSS/Algorith.115/app14_boxs_m.pdf (SPSS)
                        //FBoxM: double;
                        protected
                        //matrice de variance-covariance totale
                        FT: PMatrix;
                        //matrice de variance-covariance intra (sans la pondration finale)
                        //a rend l'outil plus gnrique : c'est 1/(n-k) si LDA, 1/n si Wilks, etc.
                        FWBrut: PMatrix;
                        //calculer la mvcv totale
                        procedure   computeMultivariateStatDes(prmExamples: TExamples); override;
                        //calculer la mcvc intra
                        procedure   computeMultivariateCondStatDes(lstExamples: TObjectList); override;
                        //les transformations associes du Wilks
                        procedure   computeStatisticalTests(); override;
                        public
                        //dcrire les statistiques de tests associes  la classe
                        function    getHTMLStatisticalTests(): string; override;
                        destructor destroy(); override;
                        //calculer le Lambda de Wilks  partir d'un couple de matrices (T et W ) -- attention, les matrices sont chamboules
                        function    computeWilks(T,W: PMatrix; size: integer): double;
                        //properties
                        property    MatT: PMatrix read FT;
                        property    MatWBrut: PMatrix read FWBrut;
                        property    Wilks: double read FWilks;
                        end;

    //liste de statistiques MANOVA -- pas d'option de tri
    TLstCalcSDCondManova = class(TLstCalcStatDes)
                           public
                           function    getHeaderHTML(): string; override;
                           end;

    //classe de classe
    TClassLstCalcSDCondManova = class of TLstCalcSDCondManova;

implementation

uses
    Sysutils,
    UConstConfiguration,
    UCalcMatrixToAttributes, FMath,
    Math;

{ TCalcSDCondMultivariate }

procedure TCalcSDCondMultivariate.computeMultivariateCondStatDes(
  lstExamples: TObjectList);
var k: integer;
    stats: TLstCalcStatDesContinuous;
begin
 //pour chaque groupe
 for k:= 0 to pred(lstExamples.Count) do
  begin
   //rcuprer les effectifs par groupes
   FTabSizeGroups[k]:= (lstExamples.Items[k] as TExamples).Size;
   //calculer les stats descriptives par groupes
   stats:= TLstCalcStatDesContinuous.Create(FDescriptors,lstExamples.Items[k] as TExamples);
   FStatGroups.Add(stats);
  end;
end;

procedure TCalcSDCondMultivariate.computeMultivariateStatDes(
  prmExamples: TExamples);
begin
 FStatGlobal.RefreshStat(prmExamples);
end;

constructor TCalcSDCondMultivariate.create(prmD: TAttribute;
  prmLstAtts: TLstAttributes; prmExamples: TExamples);
begin
 inherited Create(prmD);
 //liste des descripteurs
 FDescriptors:= prmLstAtts;
 //quelques initialisations
 FStatGroups:= TObjectList.Create(TRUE);
 FStatGlobal:= TLstCalcStatDesContinuous.Create(prmLstAtts,NIL);
 setLength(FTabSizeGroups,prmD.nbValues);
 //calculs ventuellement
 if assigned(prmExamples)
  then RefreshStat(prmExamples);
end;

destructor TCalcSDCondMultivariate.destroy;
begin
 Finalize(FTabSizeGroups);
 FStatGlobal.Free();
 FStatGroups.Free();
 inherited;
end;

function TCalcSDCondMultivariate.getHTMLMultivariateStatDes: string;
var s: string;
    k,j: integer;
begin
 //dcrire les variables conditionnellement au groupe d'appartenance
 s:= HTML_HEADER_TABLE_RESULT;
 //en-tte avec le nom des modalits (reprsentant les groupes)
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY + '<TH width=70>Group</TH>';
 for k:= 1 to Attribute.nbValues do
  s:= s + format('<TH width=70>%s</TH>',[Attribute.LstValues.getDescription(k)]);
  s:= s + format('<TH width=70 %s>ALL</TH>',[HTML_BGCOLOR_HEADER_GRAY]);
 s:= s + '</TR>';
 //effectifs par groupes
 s:= s + HTML_TABLE_COLOR_DATA_GREEN + '<TH>Group Size</TH>';
 for k:= 1 to Attribute.nbValues do
  s:= s + format('<TD align=right>%d</TD>',[FTabSizeGroups[pred(k)]]);
 s:= s + format('<TD align="right">%d</TD>',[NbExamples]);
 s:= s + '</TR>';
 //description des variables dans chaque groupe
 for j:= 0 to pred(FDescriptors.Count) do
  begin
   //nom de la variable
   s:= s + HTML_TABLE_COLOR_DATA_GRAY + format('<TH>%s</TH>',[FDescriptors.Attribute[j].Name]);
   //stat conditionnelles
   for k:= 1 to Attribute.nbValues do
    s:= s + format('<TD align="right">%.4f</TD>',[((FStatGroups.Items[pred(k)] as TLstCalcStatDesContinuous).LstStat.Items[j] as TCalcStatDesContinuous).Average]);
   s:= s + format('<TD align="right" %s>%.4f</TD>',[HTML_BGCOLOR_HEADER_GRAY,(FStatGlobal.LstStat.Items[j] as TCalcStatDesContinuous).Average]);
   s:= s + '</TR>';
  end;
 s:= s+'</table>';
 //
 result:= s;
end;

function TCalcSDCondMultivariate.getHTMLResult(prmOption: integer): string;
var s: string;
begin
 s:= format('<TD valign="top" %s>',[HTML_BGCOLOR_DATA_BLUE]) + getHTMLMultivariateStatDes() + '</TD>';
 s:= s + format('<TD valign="top" %s>',[HTML_BGCOLOR_DATA_BLUE]) + getHTMLStatisticalTests() + '</TD>';
 result:= s;
end;

function TCalcSDCondMultivariate.getStatGroup(
  k: integer): TLstCalcStatDesContinuous;
begin
 result:= FStatGroups.Items[k] as TLstCalcStatDesContinuous;
end;

procedure TCalcSDCondMultivariate.RefreshStat(prmExamples: TExamples);
var lst: TObjectList;
begin
 FNbExamples:= prmExamples.Size;
 //caluler les stats marginales
 computeMultivariateStatDes(prmExamples);
 //calculer les stats conditionnelles
 lst:= prmExamples.DispatchExamples(Attribute);
 computeMultivariateCondStatDes(lst);
 lst.Free();
 //calculer les statistiques de test
 computeStatisticalTests();
end;

{ TCalcSDCondManova }

procedure TCalcSDCondManova.computeMultivariateCondStatDes(
  lstExamples: TObjectList);
var mvcv: PMatrix;
    k,i,j: integer;
    ex: TExamples;
    stats: TLstCalcStatDesContinuous;
    vNum: double;
begin
 //stats
 inherited computeMultivariateCondStatDes(lstExamples);
 //matrice de variance covariance intra (sans la pondration finale)
 dimMatrix(FWBrut,FDescriptors.Count,FDescriptors.Count);
 //pour chaque modalit de l'attribut classe
 for k:= 1 to Attribute.nbValues do
  begin
   ex:= lstExamples.Items[pred(k)] as TExamples;
   stats:= FStatGroups.Items[pred(k)] as TLstCalcStatDesContinuous;
   mvcv:= BuildMatVCV(ex,FDescriptors,vcvNormCentered,stats);
   //addition
   vNum:= 1.0*ex.Size;
   for i:= 1 to FDescriptors.Count do
    for j:= 1 to FDescriptors.Count do
     begin
      //dcomposition du calcul car plus tard peut-tre la stat BOX-M pourrait tre programme (galit des mat. var-covar)
      mvcv^[i]^[j]:= vNum * mvcv^[i]^[j];
      FWBrut^[i]^[j]:= FWBrut^[i]^[j] + mvcv^[i]^[j];
     end;
   // optimiser plus tard, c'est dramatique toutes ces allocations/dsallocations inutiles
   delMatrix(mvcv,FDescriptors.Count,FDescriptors.Count);
  end;
end;

procedure TCalcSDCondManova.computeMultivariateStatDes(
  prmExamples: TExamples);
begin
 //stats. descriptives par variables
 inherited computeMultivariateStatDes(prmExamples);
 //matrice
 FT:= BuildMatVCV(prmExamples,FDescriptors,vcvNormCentered,FStatGlobal);
end;

procedure TCalcSDCondManova.computeStatisticalTests;
var matT,matW: PMatrix;
    i,j: integer;
    vPonderation: double;    
    //Tenenhaus, p.251
    n,p,k,r,t,u,num,denom: double;
begin
 //initialisations -- transformation de Bartlett et la p-value associe
 FBartlett:= 0.0;
 FPValueBartlett:= 1.0;
 FDDLBartlett:= 0;
 FRao:= 0.0;
 FPValueRao:= 1.0;
 FDDLNum:= 0;
 FDDLDenom:= 0;
 //copier les matrices et corriger ventuiellement en re-pondrant correctement
 DimMatrix(matT,FDescriptors.Count,FDescriptors.Count);
 CopyMatrix(matT,FT,1,1,FDescriptors.Count,FDescriptors.Count);
 DimMatrix(matW,FDescriptors.Count,FDescriptors.Count);
 CopyMatrix(matW,FWBrut,1,1,FDescriptors.Count,FDescriptors.Count);
 vPonderation:= 1.0 / (1.0 * NbExamples); 
 for i:= 1 to FDescriptors.Count do
  for j:= 1 to FDescriptors.Count do
   matW^[i]^[j]:= vPonderation * matW^[i]^[j];
 //Lambda de Wilks
 FWilks:= self.computeWilks(matT,matW,FDescriptors.Count);
 //la suite est calculable ?
 if (FWilks < 1.0)
  then
   begin
    //****************
    //calcul des stats
    //****************
    n:= 1.0 * NbExamples;
    p:= 1.0 * FDescriptors.Count;
    k:= 1.0 * Attribute.nbValues;
    //transformation de Bartlett --> Chi-2 (Tenenhaus, p.251)
    FBartlett:= -1.0 * (n - 1.0 - 0.5 * (p + k)) * LN(FWilks);
    FDDLBartlett:= FDescriptors.Count * (Attribute.nbValues - 1);
    FPValueBartlett:= PKhi2(FDDLBartlett,FBartlett);
    //transformation de Rao --> Fisher (Tenenhaus, p.251)
    num:= p * (k - 1.0);
    FDDLNum:= TRUNC(num);
    t:= power(p,2.0) + power(k-1.0,2.0) - 5.0;
    if (t > 0.0)
     then t:= SQRT((power(p,2.0) * power(k-1,2.0) - 4.0)/t)
     else t:= 1.0;
    r:= (num - 2.0) /  2.0;
    u:= (2.0 * n - p - k - 2.0) / 2.0; 
    denom:= u * t - r;
    FDDLDenom:= TRUNC(denom);
    FRao:= power(FWilks,1.0/t);
    FRao:= ((1.0 - FRao)/FRao) * (denom / num);
    FPValueRao:= PSnedecor(FDDLNum,FDDLDenom,FRao);
   end;
 delMatrix(matT,FDescriptors.Count,FDescriptors.Count);
 delMatrix(matW,FDescriptors.Count,FDescriptors.Count);
end;

function TCalcSDCondManova.computeWilks(T,W: PMatrix; size: integer): double;
var dT,dW: double;
begin
 result:= 1.0;
 //dterminant de la mvcv totale
 dT:= det(T,1,size);
 if (dT > 0.0)
  then
   begin
    //det de la mvcv intra
    dW:= det(W,1,size);
    //and then...
    result:= dW / dT;
   end;
end;

destructor TCalcSDCondManova.destroy;
begin
 if assigned(FT) then DelMatrix(FT,FDescriptors.Count,FDescriptors.Count);
 if assigned(FWBrut) then DelMatrix(FWBrut,FDescriptors.Count,FDescriptors.Count);
 inherited destroy();
end;

function codeCouleur(alpha: double): string;
begin
 if (alpha < SIGNIFICANCE_LEVEL_FOR_TESTS)
  then result:= HTML_BGCOLOR_DATA_RED
  else result:= HTML_BGCOLOR_DATA_GRAY; 
end;

function TCalcSDCondManova.getHTMLStatisticalTests: string;
var s: string;
begin
 s:= HTML_HEADER_TABLE_RESULT;
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY + '<TH width=80>Stat</TH><TH width=70>Value</TH><TH width=70>p-value</TH></TR>';
 s:= s + HTML_TABLE_COLOR_DATA_GRAY + format('<TD>Wilks'' Lambda</TD><TD align="right">%.5f</TD><TD align="center">-</TD></TR>',[FWilks]);
 s:= s + HTML_TABLE_COLOR_DATA_GRAY + format('<TD>Bartlett -- C(%d)</TD><TD align="right">%.5f</TD><TD align="right" %s>%.5f</TD></TR>',[FDDLBartlett,FBartlett,codeCouleur(FPValueBartlett),FPValueBartlett]);
 s:= s + HTML_TABLE_COLOR_DATA_GRAY + format('<TD>Rao -- F(%d, %d)</TD><TD align="right">%.5f</TD><TD align="right" %s>%.5f</TD></TR>',[FDDLNum,FDDLDenom,FRao,codeCouleur(FPValueRao),FPValueRao]);
 s:= s + '</table>';
 result:= s;   
end;

{ TLstCalcSDCondManova }

function TLstCalcSDCondManova.getHeaderHTML: string;
begin
 result:= HTML_TABLE_COLOR_HEADER_GRAY+'<TH>Descriptive stat. (Mean)</TH><TH>Tests results</TH></TR>';
end;

end.
