(*************************************************************************)
(* UCalcStatDesNormalityTest.pas - Copyright (c) 2005 Ricco RAKOTOMALALA *)
(*************************************************************************)

{
@abstract(Classes de calcul pour les tests de normalit -- Effectue le lien vers le FORTRAN)
@author(Ricco)
@created(29/07/2005)

Les implmentations s'appuient en grande partie sur le code dispo sur le site de STATLIB -- http://lib.stat.cmu.edu/apstat/

Le test de Shapiro-Wilks est un portage du code FORTRAN de la bibliothque de Patrick Royston 
ALGORITHM AS R94 APPL. STATIST. (1995) VOL.44, NO.4

}

unit UCalcStatDesNormalityTest;

interface

USES
        UCalcStatDes,
        UDatasetExamples,
        UDatasetDefinition,
        UCalcStatDesNormalityTestFromFORTRAN;

TYPE
        TCalcSDContNormalityTest = class(TCalcStatDesContinuous)
                                   private
                                   //statistique de Shapiro-Wilk
                                   FSW: TYPE_FLOAT_NOMALITY_TEST;
                                   //proba critique
                                   FSWpvalue: TYPE_FLOAT_NOMALITY_TEST;
                                   //la statistique de Kolmogorov-Smirnov
                                   FKSMoins, FKSPlus, FKS: TYPE_FLOAT_NOMALITY_TEST;
                                   //le rsultats corrig avec Lilliefors
                                   FResLilliefors: TGRID_RESULTS_KS_LILLIEFORS;
                                   //indicateur de Anderson-Darling
                                   FA2 : TYPE_FLOAT_NOMALITY_TEST;
                                   //palier de proba d'Anderson-Darling
                                   FA2Proba: TGRID_RESULTS_KS_ANDERSON_DARLING;
                                   //indicateurs d'Agostino
                                   FzG1, FzG2, FK2, FPK2: TYPE_FLOAT_NOMALITY_TEST;
                                   public
                                   //tout va tre ralis dans cette portion
                                   procedure   RefreshStat(prmExamples: TExamples); override;
                                   //affichage des rultats
                                   function    getHTMLResult(prmOption: integer = -1): string; override;
                                   end;

        TLstCalcSDContNormalityTest = class(TLstCalcStatDesContinuous)
                                      public
                                      function    getHeaderHTML(): string; override;  
                                      end;

implementation

uses
        Sysutils,
        UConstConfiguration;

const
        //taille max.
        NORMALITY_TEST_MAX_SIZE_SHAPIRO_WILK = 5000;        

//**********************************************
//fonctions locales de mise en forme des sorties
//**********************************************

//Shapiro-Wilk --> si p-value < seuil, on rejette l'hypothse de normalit
function codeCouleurSW(pValue: extended): string;
begin
 if (pValue < SIGNIFICANCE_LEVEL_FOR_TESTS)
  then result:= HTML_BGCOLOR_DATA_RED
  else result:= HTML_BGCOLOR_DATA_GREEN;
end;

//KS-Lilliefors --> on ragit par rapport au niveau de p-value atteint (plus il est petit, plus on rejette la normalit)
function codeCouleurKSL(idRes: TGRID_RESULTS_KS_LILLIEFORS): string;
begin
 if (idRes < sup_KSL_05)
  then result:= HTML_BGCOLOR_DATA_GREEN
  else result:= HTML_BGCOLOR_DATA_RED;
end;

//KS-Anderson-Darling --> idem Lillierfors, avec des paliers de probas diffrents dans la table
function codeCouleurKSAD(idRes: TGRID_RESULTS_KS_ANDERSON_DARLING): string;
begin
 if (idRes <= sup_AD_05)
  then result:= HTML_BGCOLOR_DATA_GREEN
  else result:= HTML_BGCOLOR_DATA_RED;
end;

//d'Agostino --> si p-value < alpha, on rejette l'hypothse nulle de normalit de distribution
function codeCouleurAgostino(pValue: extended): string;
begin
 if (pValue < SIGNIFICANCE_LEVEL_FOR_TESTS)
  then result:= HTML_BGCOLOR_DATA_RED
  else result:= HTML_BGCOLOR_DATA_GREEN;
end;

{ TCalcSDContNormalityTest }

function TCalcSDContNormalityTest.getHTMLResult(
  prmOption: integer): string;
var s: string;
begin
 s:= '<TD>'+self.Attribute.name+'</TD>';
 //stats
 s:= s+format('<TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+' ; '+STR_FORMAT_VIEW_STAT_ACCURACY+'</TD>',[Average,StdDev]);
 //Shapiro-Wilks
 s:= s+format('<TD align=right %s>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+' <br>(<i>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</i>)</TD>',[codeCouleurSW(FSWPValue),FSW,FSWPValue]);
 //Lilliefors
 s:= s+format('<TD align=right %s>'+STR_FORMAT_VIEW_STAT_ACCURACY+' = max['+STR_FORMAT_VIEW_STAT_ACCURACY+','+STR_FORMAT_VIEW_STAT_ACCURACY+']'+' <br>(<i>%s</i>)</TD>',
              [codeCouleurKSL(FResLilliefors),FKS,FKSMoins,FKSPlus,TXT_KS_LILLIEFORS[FResLilliefors]]);
 //Anderson-Darling
 s:= s+format('<TD align=right %s>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+' <br>(<i>%s</i>)</TD>',
              [codeCouleurKSAD(FA2Proba),FA2,TXT_ANDERSON_DARLING[FA2Proba]]);
 //d'Agostino
 s:= s+format('<TD align=right %s> %.4f ^ 2 + %.4f ^ 2 = '+STR_FORMAT_VIEW_STAT_ACCURACY+' <br>(<i>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</i>)</TD>',
              [codeCouleurAgostino(FpK2),FzG1,FzG2,FK2,FpK2]);
 //and then...
 result:= s;
end;

procedure TCalcSDContNormalityTest.RefreshStat(prmExamples: TExamples);
var sortedEx: TExamples;
    mean, std: double;
    i: integer;
    X: array of TYPE_FLOAT_NOMALITY_TEST;
    idErr: integer;
begin
 //calculer les stats. standards
 inherited RefreshStat(prmExamples);
 //init.
 FSW:= 0.0;
 FSWPValue:= 0.0;
 FKS:= 0.0;
 FResLilliefors:= sup_KSL_15;
 //calculs possibles
 if (self.StdDev > EPSILON_VALUE)
  then
   begin
     //trier les individus selon l'attribut
     sortedEx:= TExamples.Create(prmExamples.Size);
     sortedEx.Copy(prmExamples);
     sortedEx.QuickSortBy(self.Attribute);
     //copies locales pour plus de rapidit
     mean:= self.Average;
     std:= self.StdDev;
     //prparer le tableaux de calcul -- dans tous les cas, les donnes doivent tre tries, KSLilliefors seul demande des donnes centres et rduites
     setLength(X,sortedEx.Size + 1);
     //remplir en centrant et rduisant -- les donnes sont bien tries
     for i:= 1 to sortedEx.Size do
      X[i]:= (self.Attribute.cValue[sortedEx.Number[i]] - mean) / std;
     //*****************************
     //lancer les calculs successifs
     //*****************************
     //Shapiro-Wilk -- calculable uniquement dans ce cas
     if (sortedEx.Size <= NORMALITY_TEST_MAX_SIZE_SHAPIRO_WILK)
      then SWILK(X,sortedEx.Size,FSW,FSWPValue,idErr)
      else
       //rsultats "non-calcul"
       begin
        FSW:= 0.0;
        FSWPValue:= 1.0;
       end;
     //Kolmogorov-Smirnov & Lilliefors -- ici les donnes tries et centres-rduites sont primordiales
     KSLILLIEFORS(X,sortedEx.Size,FKSMoins,FKSPlus,FKS,FResLilliefors);
     //Kolmogorov-Smirnov & Anderson-Darling -- idem, donnes obligatoirement tries et centres-rduites
     KS_ANDERSON_DARLING(X,sortedEx.Size,FA2,FA2Proba);
     //d'Agostino -- Test omnibus
     D_AGOSTINO(X,sortedEx.Size, FzG1, FzG2, FK2, FpK2);
     //vider tout cela
     Finalize(X);
     sortedEx.Free();
   end;
end;

{ TLstCalcSDContNormalityTest }

function TLstCalcSDContNormalityTest.getHeaderHTML: string;
begin
 result:= HTML_TABLE_COLOR_HEADER_GRAY+
          '<TH>Attribute</TH><TH>Mu ; Sigma</TH><TH>Shapiro-Wilk <br>(p-value)</TH><TH>Lilliefors D = max[D-,D+] <br>(p-value)</TH><TH>Anderson-Darling <br>(p-value)</TH><TH>d''Agostino <br>(p-value)</TH></TR>';
end;

end.
