(***********************************************************************)
(* UCalcClusteringTreeCART.pas - Copyright (c) 2005 Ricco RAKOTOMALALA *)
(***********************************************************************)

{
@abstract(Arbre de Classification -- Extension du schma CART [Regression Tree] au cas endogne-multivarie)
@author(Ricco)
@created(09/09/2005)

Utilisation du cadre CART pour construire un arbre de classification. L'inspiration est la
mthode divisive monothtique de Lechevalier et Chavent (1999), la rsolution thorique est
inspire du chapitre 8 de Breiman et al.(1984).

Classes de calcul et structure d'arbre. Le modle utilis est celui de << UCalcSpvTreeDefinition.pas >>
}

unit UCalcClusteringTreeCART;

interface

USES
    Contnrs,
    UCalcClusteringTree,
    UCalcStatDes, UDatasetdefinition, UDatasetImplementation,
    UOperatorDefinition,
    UDatasetExamples,
    UCalcRulesDefinition,
    UCalcTreeStructureDefinition,
    UCalcDistribution,
    UCompClusteringDefinition;

TYPE
        {type de donnes  valuer pour CTP}
        TEnumCTPDataSet = (ctpGrowingSet,ctpPruningSet);

        {enregistrements pour vrifier les squences d'arbres}
        TRecTreeSequenceCTP = record
                              {numro de sous-arbre}
                              NumTree: integer;
                              {nombre de feuilles}
                              NbLeaves: integer;
                              {%wss pruning}
                              wssPrune: double;
                              {%wss growing}
                              wssGrow: double;
                              end;

        {tableau des enregistrements de rsultats}
        TTabRecTreeSequenceCTP = array of TRecTreeSequenceCTP;

        {feuille pour CTP}
        TSplitLeafClusCTP = class(TSplitLeafClus)
                            end;

        {le split d'un attribut en supervis pour CART}
        TSplitAttributClusCTP = class(TSplitAttributClus)
                                protected
                                function    getClassSplitLeaf(): TClassSplitLeaf; override;
                                end;

        {une liste d'attributs candidats pour un sommet}
        TLstSplitAttClusCTP = class(TLstSplitAttClus)
                              protected
                              function  getClassSplitAttribut(): TClassSplitAttribut; override;
                              end;

        {un noeud de l'arbre CART}
        TMLTreeNodeClusCTP = class(TMLTreeNodeClus)
                             private
                             //stats sur le pruning -- erreur quadratique
                             FSum2Pruning: double;
                             //pour le calcul de l'ecart-type de l'erreur quadratique
                             FSum4Pruning: double;
                             FSumCroisePruning: double;
                             {numro d'appartenance dans les sous-arbres}
                             FCurNsa: integer;
                             protected
                             function  getClassLstSplitAttributes(): TClassLstSplitAttributes; override;
                             public
                             //distance par rapport au centre de classe
                             function  distToMean(example: integer): double;
                             //
                             destructor destroy; override;
                             {test si un frre existe}
                             function  nextBrotherExists(): boolean;
                             {obtenir le frre suivant}
                             function  getNextBrother(): TMLTreeNodeClusCTP;
                             {la liste des sous-arbres corresp.  un numro}
                             procedure getSubTree(prmNsa: integer; lst: TObjectList);
                             {cot de mauvais classement}
                             function  localCost(prmSet: TEnumCTPDataSet): double;
                             {laguer  partir du sommet courant}
                             procedure pruneFromHere();
                             end;

        //cumul des valeurs calcules sur les noeuds
        TRecInfoNode = record
                       growCost: double;
                       pruneS4: double;
                       pruneCroise: double;
                       end;

        {la structure de l'arbre}
        TMLTreeStructureClusCTP = class(TMLTreeStructureClus)
                                  private
                                  {l'arbre optimal}
                                  FOptimalSubTree: integer;
                                  {la solution slectionne}
                                  FSelectedSubTree: integer;
                                  {squences d'arbres utiliss}
                                  FTabTreeSequence: TTabRecTreeSequenceCTP;
                                  {***** recherche des sous-arbres *****}
                                  {liste des squences d'arbres}
                                  FLstSeqTree : TObjectList;
                                  {r-initialisation des numros de squences d'arbres}
                                  procedure  initialiserCurNsa();
                                  {numroter les squences d'arbres}
                                  procedure  numeroterCurNsa(var curNsa: integer);
                                  {marquer les sommets}
                                  function  mark_nodes(prmNsa: integer; prmAlpha: double): boolean;
                                  {calculer l'alpha suivante}
                                  function  computeNextAlpha(prmNsa: integer): double;
                                  {calculer alpha  partir d'un noeud}
                                  function  computeAlphaNode(prmSommet: TMLTreeNodeClusCTP; prmNsa: Integer): Double;
                                  {tester si l'lagage est possible  partir d'un sommet}
                                  function  isPruningPossible(prmNode: TMLTreeNodeClusCTP; prmNsa: Integer; prmAlpha: Double): Boolean;
                                  {marquer un sous-arbre  partir d'un sommet}
                                  procedure mark_subTree(prmNode: TMLTreeNodeClusCTP; prmNsa: integer);
                                  {additionner les cots d'un sous-arbre}
                                  function  computeSubTreeCost(lst: TObjectList; prmSet: TEnumCTPDataSet): double;
                                  {calculer le cot en pruning d'un sous-arbre candidat}
                                  function  computePrunedSubTreeCosts(prmNsa: integer; var nbLeaves: integer; var info: TRecInfoNode): double;
                                  {rechercher le sous-arbre pour un pruning}
                                  procedure explorePrunedSubTreeCosts(node: TMLTreeNodeClusCTP; prmNsa: integer; var costs: double; var nbLeaves: integer; var info: TRecInfoNode);
                                  {numrer les sous-arbres et leurs cots -- et renvoyer l'arbre optimal}
                                  function enumerateSubTrees(nbTrees: integer): integer;
                                  {laguer effectivement  partir du n de sous-arbre}
                                  procedure pruneSubTree(prmNsa: integer);
                                  {laguer effectivement  partir d'un sommet et du n de sous-arbre}
                                  procedure pruneSubTreeFromNode(prmNode: TMLTreeNodeClusCTP; prmNsa: integer);
                                  {laguer  partir d'un sommet, quel que soit son numro}
                                  procedure pruningFromNode(prmNode: TMLTreeNodeClusCTP);
                                  {*************************************}
                                  {insrer les individus en pruning}
                                  procedure applyPruningSet(prmExamples: TExamples);
                                  {isrer 1 individu en test}
                                  procedure applyPruningExample(node: TMLTreeNode; example: integer);
                                  protected
                                  FPruningSet: TExamples;
                                  //dcomposer app-pruning le cas chant, surcharg chez CART par ex.
                                  procedure  defineTrainingSet(prmExamples: TExamples); override;
                                  {type du noeud de l'arbre}
                                  function   getClassMLTreeNode(): TClassMLTreeNode; override;
                                  {ajouter le gestionnaire de liste de squences d'arbres}
                                  procedure  prepareInternalStructures(); override;
                                  {dtruire le gestionnaire de gestionnaire de squences d'arbres}
                                  procedure  destroyInternalStructures(); override;
                                  public
                                  destructor destroy(); override;
                                  {post-lagage  la CART}
                                  procedure   PostPruning(); override;
                                  {taille growing}
                                  function getGrowingSetSize(): integer;
                                  {taille pruning}
                                  function getPruningSetSize(): integer;
                                  {description de la squence d'arbres}
                                  property TabTreeSequence: TTabRecTreeSequenceCTP read FTabTreeSequence;
                                  {numro de l'arbre slection}
                                  property SelectedSubTree: integer read FSelectedSubTree;
                                  {numro de l'arbre optimal}
                                  property OptimalSubTree: integer read FOptimalSubTree; 
                                  end;

        //oprateur de calcul
        TCalcClusTreeCTP = class(TCalcClusTree)
                           protected 
                           function  getClassTreeStructure(): TClassMLTreeStrucClus; override;
                           function  getHTMLSpecificDescription(): string; override; 
                           end; 
                                                           
implementation

uses
    Sysutils, Math,
    UCompClusteringTreeCART, ULogFile, UConstConfiguration;

const
    //epsilon  prendre en compte dans les comparaisons lors du pruning
    CTP_PRUNING_EPSILON = 0.0001;

    //nombre de squences  montrer
    SEUIL_NB_SEQUENCES = 15;

    //prcision dans le calcul de la pente
    ACCURACY_SLOPE_PRUNE_WSS = 3;


{ TSplitAttributClusCTP }

function TSplitAttributClusCTP.getClassSplitLeaf: TClassSplitLeaf;
begin
 result:= TSplitLeafClusCTP;
end;

{ TLstSplitAttClusCTP }

function TLstSplitAttClusCTP.getClassSplitAttribut: TClassSplitAttribut;
begin
 result:= TSplitAttributClusCTP;
end;

{ TMLTreeNodeClusCTP }

destructor TMLTreeNodeClusCTP.destroy;
begin
 inherited;
end;

function TMLTreeNodeClusCTP.distToMean(example: integer): double;
var norm: TTabVarianceNormalisation;
    j: integer;
    value,dist: double;
begin
 norm:= (self.TreeStructure as TMLTreeStructureClusCTP).Normalization;
 dist:= 0.0;
 for j:= 0 to pred(self.FStats.Count) do
  begin
   value:= SQR(self.FStats.Stat(j).Attribute.cValue[example] - (self.FStats.Stat(j) as TCalcStatDesContinuous).Average);
   value:= value / norm[j];
   dist:= dist + value;
  end;
 result:= dist;
end;

function TMLTreeNodeClusCTP.getClassLstSplitAttributes: TClassLstSplitAttributes;
begin
 result:= TLstSplitAttClusCTP;
end;

function TMLTreeNodeClusCTP.getNextBrother: TMLTreeNodeClusCTP;
var p: Integer;
begin
 RESULT:= NIL;
 {$B-}
 if assigned(self.Predecessor) and (self.Predecessor.getCountSuccessors()>0)
  Then
   Begin
    p:= self.Predecessor.getSuccessor(self);
    //il existe, et il reste de la marge pour passer au suivant --> d'o le (-1) !!!
    if (p>=0) and (p<(self.Predecessor.getCountSuccessors()-1))
     then result:= self.Predecessor.getSuccessor(succ(p)) as TMLTreeNodeClusCTP;
   End;
end;

procedure TMLTreeNodeClusCTP.getSubTree(prmNsa: integer; lst: TObjectList);
var fini: Boolean;
    sommet: TMLTreeNodeClusCTP;
begin
 //vider la liste
 lst.Clear();

 //initialiser
 fini:= FALSE;
 sommet:= self;//point de dpart, soi-mme !!!

 //parcourir
 while not(fini) do
  Begin
   //traiter le sommet courant
   if sommet.isLeaf() OR ((sommet.FCurNsa>0) and (sommet.FCurNsa<prmNsa))
    Then
     Begin
      if (sommet<>self)
       Then lst.Add(sommet);
     End;

   //passer au sommet suivant si possible
   if sommet.isLeaf() OR ((sommet.FCurNsa>0) and (sommet.FCurNsa<prmNsa))
    Then
     //passer au frre
     Begin
      //remonte
      while (sommet<>self) and not(sommet.nextBrotherExists()) do
       sommet:= sommet.Predecessor as TMLTreeNodeClusCTP;

      //passage au frre si ce n'est pas la racine
      if (sommet<>self)
       Then sommet:= sommet.getNextBrother()
       Else FINI:= TRUE;//la recherche en arbre est termine
       
     End
    Else
     Begin
      sommet:= sommet.getSuccessor(0) as TMLTreeNodeClusCTP;
     End;
  End;
end;

function TMLTreeNodeClusCTP.localCost(prmSet: TEnumCTPDataSet): double;
begin
 case (prmSet) of
  ctpGrowingSet: result:= FStats.inertie((self.TreeStructure as TMLTreeStructureClus).Normalization)
  //pruning
  else result:= FSum2Pruning;
 end;
end;

function TMLTreeNodeClusCTP.nextBrotherExists: boolean;
var p: Integer;
begin
 RESULT:= FALSE;
 {$B-}
 if assigned(self.Predecessor) and (self.Predecessor.getCountSuccessors()>0)
  Then
   Begin
    p:= self.Predecessor.getSuccessor(self);
    //il existe, et il reste de la marge pour passer au suivant --> d'o le (-1) !!!
    RESULT:= (p>=0) and (p<(self.Predecessor.getCountSuccessors()-1));
   End;
end;

procedure TMLTreeNodeClusCTP.pruneFromHere;
var curNode: TMLTreeNode;
    i: integer;
begin
 if not(self.isLeaf())
  then
   begin
    for i:= 0 to pred(self.getCountSuccessors()) do
     begin
      curNode:= self.getSuccessor(i);
      //en rcursif, se vider de son sous-arbre
      (curNode as TMLTreeNodeClusCTP).pruneFromHere();
      //si on est l c'est que le boulot a t fait
      //la suppression entrane la destruction de l'objet !!!
      self.TreeStructure.deleteNode(curNode);
     end;
    //vider la liste des successeurs -- rien  librer explicitement
    self.clearSuccessors();
   end;
end;

procedure TMLTreeStructureClusCTP.applyPruningExample(node: TMLTreeNode;
  example: integer);
var curNode: TMLTreeNodeClusCTP;
    nextNode: TMLTreeNode; 
    i: integer;
    dist: double;
begin
 curNode:= node as TMLTreeNodeClusCTP;
 //ajouter l'exemple dans le comptage -- distance par rapport au centre de classe
 dist:= curNode.distToMean(example);
 curNode.FSum2Pruning:= curNode.FSum2Pruning + dist;
 curNode.FSum4Pruning:= curNode.FSum4Pruning + dist * dist;
 curNode.FSumCroisePruning:= curNode.FSumCroisePruning + dist * (curNode.TreeStructure.RootNode as TMLTreeNodeClusCTP).distToMean(example);
 //passer au suivant en dtectant la feuille concerne
 if not(curNode.isLeaf)
  then
   begin
    //chercher le sommet  appeler
    nextNode:= NIL;
    for i:= 0 to pred(curNode.getCountSuccessors()) do
     begin
      if curNode.getSuccessor(i).Condition.TestExample(example)
       then
        begin
         nextNode:= curNode.getSuccessor(i);
         BREAK;
        end;
     end;
    //alors, alors, s'il n'y a rien, on est mal parce que c'est une valeur non-rfrence !!!
    if assigned(nextNode)
     then self.applyPruningExample(nextNode,example)
     else TraceLog.WriteToLogFile(format('[CTP] !!!! ERREUR !!!, valeur non rfrence dans PRUNING, example n %d',[example]));
   end;
end;

procedure TMLTreeStructureClusCTP.applyPruningSet(prmExamples: TExamples);
var i,iNode: integer;
begin
 //initaliser tous les cots sur les noeuds
 for iNode:= 0 to pred(self.CountNodes) do
  begin
   (self.Node[iNode] as TMLTreeNodeClusCTP).FSum2Pruning:= 0.0;
   (self.Node[iNode] as TMLTreeNodeClusCTP).FSum4Pruning:= 0.0;
   (self.Node[iNode] as TMLTreeNodeClusCTP).FSumCroisePruning:= 0.0;
  end;
 //puis lancer les exemples pour le comptage des erreurs
 for i:= 1 to prmExamples.Size do
  applyPruningExample(self.RootNode,prmExamples.Number[i]);
end;

function TMLTreeStructureClusCTP.computeAlphaNode(
  prmSommet: TMLTreeNodeClusCTP; prmNsa: Integer): Double;
var cSommet,cFeuilles,tmp: Double;
begin
 cSommet:= prmSommet.localCost(ctpGrowingSet);
 //rcuprer les feuilles
 prmSommet.getSubTree(prmNsa,self.FLstSeqTree);
 //le cot des feuilles
 cFeuilles:= self.computeSubTreeCost(self.FLstSeqTree,ctpGrowingSet);
 //l'opration cabalistique
 tmp:= (cSommet-cFeuilles)/(1.0*self.FLstSeqTree.Count-1.0);
 //renvoyer le tout
 RESULT:= tmp;
end;

function TMLTreeStructureClusCTP.computeNextAlpha(prmNsa: integer): double;
var sommet: TMLTreeNodeClusCTP;
    alpha,minAlpha: double;
    i: Integer;    
begin
 minAlpha:= 1.0e308;
 //pour tous les sommets non-marqus
 For i:= 0 To pred(self.CountNodes) do
  Begin
   sommet:= self.Node[i] as TMLTreeNodeClusCTP;
   //le sommet n'a pas encore t explor ?
   //ce ne doit pas tre non plus une feuille
   If (sommet.FCurNsa=0) and NOT(sommet.isLeaf)
    Then
     Begin
      alpha:= computeAlphaNode(sommet,prmNsa);
      If (alpha<minAlpha)
       Then minAlpha:= alpha;
     End;
  End;
 //TraceLog.WriteToLogFile(format('[CTP] alpha = %.4f',[minAlpha]));
 RESULT:= minAlpha;
end;

function TMLTreeStructureClusCTP.computePrunedSubTreeCosts(prmNsa: integer;
  var nbLeaves: integer; var info: TRecInfoNode): double;
var cost: double;
begin
 //calculer le cout de l'arbre prun de niveau prmNsa
 cost:= 0;
 info.growCost:= 0.0;
 info.pruneS4:= 0.0;
 info.pruneCroise:= 0.0;
 nbLeaves:= 0;
 self.explorePrunedSubTreeCosts(self.RootNode as TMLTreeNodeClusCTP,prmNsa,cost,nbLeaves,info);
 RESULT:= cost;
end;

function TMLTreeStructureClusCTP.computeSubTreeCost(lst: TObjectList;
  prmSet: TEnumCTPDataSet): double;
var i: integer;
    cost: double;
begin
 cost:= 0;
 for i:= 0 to pred(lst.Count) do
  cost:= cost+(lst.Items[i] as TMLTreeNodeClusCTP).localCost(prmSet);
 result:= cost;
end;

procedure TMLTreeStructureClusCTP.defineTrainingSet(prmExamples: TExamples);
var proportion: single;
begin
 FTrainingSet:= TExamples.Create(prmExamples.Size);
 FPruningSet:= TExamples.Create(prmExamples.Size);
 //subdiviser : proportion est la taille relative du train ici et "PruningSetSize" est en pourcentage
 proportion:= 1.0-1.0*(self.PrmMethod as TOpPrmClusTreeCART).PruningSetSize/100.0;
 prmExamples.SamplingSplitting(proportion,FTrainingSet,FPruningSet,(self.PrmMethod as TOpPrmClusTreeCART).ModeRndGenerator);
end;

destructor TMLTreeStructureClusCTP.destroy();
begin
 FreeAndNil(FTrainingSet);
 FreeAndNil(FPruningSet);
 inherited;
end;

procedure TMLTreeStructureClusCTP.destroyInternalStructures;
begin
 inherited;
 FLstSeqTree.Free();
 SetLength(FTabTreeSequence,0);
end;

//new -- 11/09/2005 -- utilisation du "coude" pour dtecter le point de coupure
//on utilise la rgression sur 3 points pour calculer la pente de la courbe
//le point "optimal" correspond au point o la pente est proche de zro ( un epsilon prs)
function TMLTreeStructureClusCTP.enumerateSubTrees(
  nbTrees: integer): integer;
var subTree,nbLeaves,optSubTree: integer;
    pruneCost,growCost,minCost: double;
    info,minInfo: TRecInfoNode;
    root: TMLTreeNodeClusCTP;
    nPrim,growRoot,pruneRoot: double;
    slopeCut,slope: double;
begin
 nPrim:= 1.0 * FPruningSet.Size;
 //root
 root:= self.RootNode as TMLTreeNodeClusCTP;
 growRoot:= root.FStats.inertie(self.Normalization) / (1.0*self.FTrainingSet.Size);
 pruneRoot:= root.FSum2Pruning / nPrim;
 //
 SetLength(FTabTreeSequence,nbTrees);
 //lister les squences avec leurs perfs. respectives
 //en profiter pour dtecter l'arbre optimal
 FOptimalSubTree:= -1;
 minCost:= MATH.MaxDouble;
 for subTree:= 1 to nbTrees do
  begin
   pruneCost:= self.computePrunedSubTreeCosts(subTree,nbLeaves,info);
   pruneCost:= pruneCost/nPrim;
   growCost:= info.growCost/(1.0*self.FTrainingSet.Size);
   //new -- 09/09/2005 -- normaliser avec la racine tre dans [0 ; 1] --> Breiman, pp.224
   pruneCost:= pruneCost / pruneRoot;
   growCost:= growCost / growRoot;
   //affecter les valeurs
   FTabTreeSequence[pred(subTree)].NumTree:= subTree;
   FTabTreeSequence[pred(subTree)].NbLeaves:= nbLeaves;
   FTabTreeSequence[pred(subTree)].wssPrune:= pruneCost;
   FTabTreeSequence[pred(subTree)].wssGrow:= growCost;
   //TraceLog.WriteToLogFile(format('Tree n%4d, leaves = %4d, growing/pruned cost = %8.4f / %8.4f',[subTree,nbLeaves,growCost,pruneCost]));
   //tester
   if (pruneCost <= minCost)
    then
     begin
      minCost:= pruneCost;
      minInfo:= info;
      FOptimalSubTree:= subTree;
     end;
  end;
 //seuil de la pente  slectionner
 slopeCut:= (self.PrmMethod as TOpPrmClusTreeCART).ThresholdSlope;
 //rechercher le "bon"  partir du moins complexe vers le plus complexe
 //le point de dpart est l'arbre optimal
 optSubTree:= -1;
 for subTree:= nbTrees downto MAX(FOptimalSubTree,3) do
  begin
   //calculer la pente
   slope:= 0.5*(FTabTreeSequence[pred(subTree) - 2].wssPrune - FTabTreeSequence[pred(subTree)].wssPrune);
   //arrondir  la prcision que l'on veut
   slope:= ROUNDTO(slope,-ACCURACY_SLOPE_PRUNE_WSS);
   //TraceLog.WriteToLogFile(format('Tree n%4d --> slope = %.4f',[subTree,slope]));
   if (slope >= slopeCut)
    then
     begin
      optSubTree:= subTree;
      BREAK;
     end;
  end;
 TraceLog.WriteToLogFile(format('[CTP] the selected tree is n%2d, with err = %.4f',[optSubTree,FTabTreeSequence[pred(optSubTree)].wssPrune]));
 //renvoyer la bonne rponse
 result:= optSubTree;
end;

(* --> devant la difficult d'valuer correctement l'cart-type du rapport d'inertie....
function TMLTreeStructureClusCTP.enumerateSubTrees(
  nbTrees: integer): integer;
var subTree,nbLeaves,optSubTree: integer;
    pruneCost,growCost,minCost,thresold: double;
    info,minInfo: TRecInfoNode;
    seMinCost: double;
    root: TMLTreeNodeClusCTP;
    growRoot,pruneRoot: double;
    //Breiman,pp.306
    s,s1Square,s2Square,s12,nPrim: double;
begin
 nPrim:= 1.0 * FPruningSet.Size;
 //root
 root:= self.RootNode as TMLTreeNodeClusCTP;
 growRoot:= root.FStats.inertie(self.Normalization) / (1.0*self.FTrainingSet.Size);
 pruneRoot:= root.FSum2Pruning / nPrim;
 //
 SetLength(FTabTreeSequence,nbTrees);
 //lister les squences avec leurs perfs. respectives
 //en profiter pour dtecter l'arbre optimal
 FOptimalSubTree:= -1;
 minCost:= MATH.MaxDouble;
 for subTree:= 1 to nbTrees do
  begin
   pruneCost:= self.computePrunedSubTreeCosts(subTree,nbLeaves,info);
   pruneCost:= pruneCost/nPrim;
   growCost:= info.growCost/(1.0*self.FTrainingSet.Size);
   //new -- 09/09/2005 -- normaliser avec la racine tre dans [0 ; 1] --> Breiman, pp.224
   pruneCost:= pruneCost / pruneRoot;
   growCost:= growCost / growRoot;
   //affecter les valeurs
   FTabTreeSequence[pred(subTree)].NumTree:= subTree;
   FTabTreeSequence[pred(subTree)].NbLeaves:= nbLeaves;
   FTabTreeSequence[pred(subTree)].wssPrune:= pruneCost;
   FTabTreeSequence[pred(subTree)].wssGrow:= growCost;
   TraceLog.WriteToLogFile(format('Tree n%4d, leaves = %4d, growing/pruned cost = %8.4f / %8.4f',[subTree,nbLeaves,growCost,pruneCost]));
   //tester
   if (pruneCost <= minCost)
    then
     begin
      minCost:= pruneCost;
      minInfo:= info;
      FOptimalSubTree:= subTree;
     end;
  end;
 //cart-type de l'erreur min --> Breiman, pp.306
 s:= SQRT(pruneRoot);
 s1Square:= (1.0 / nPrim) * minInfo.pruneS4 - power(minCost,2.0);
 s2Square:= (1.0 / nPrim) * root.FSum4Pruning - power(s,4.0);
 s12:= (1.0 / nPrim) * minInfo.pruneCroise - minCost * power(s,2.0);
 seMinCost:= s1Square/power(minCost,2.0)-2.0*s12/(minCost*power(s,2.0))+s2Square/power(s,4.0);
 seMinCost:= (1.0 / nPrim) * seMinCost;
 seMinCost:= minCost * SQRT(seMinCost);
 //appliquer la xxx-SE RULE pour trouver le bon seuil --> a reste un souci quand mme cette histoire !!!
 //#ToDo1
 thresold:= minCost+(self.PrmMethod as TOpPrmClusTreeCART).SERule * seMinCost;
 //infos. sur l'erreur min et son ecart-type
 TraceLog.WriteToLogFile(format('[CTP] min err pruning (se) = %.4f (%.4f) --> threshold = %.4f',[minCost,seMinCost,thresold]));
 //rechercher le "bon"  partir des moins complexes vers les plus complexes
 optSubTree:= -1;
 for subTree:= nbTrees downto 1 do
  begin
   if (FTabTreeSequence[pred(subTree)].wssPrune <= thresold)
    then
     begin
      optSubTree:= subTree;
      BREAK;
     end;
  end;
 TraceLog.WriteToLogFile(format('[CTP] the selected tree is n%2d, with err = %.4f',[optSubTree,FTabTreeSequence[pred(optSubTree)].wssPrune]));
 //renvoyer la bonne rponse
 result:= optSubTree;
end;
*)

procedure TMLTreeStructureClusCTP.explorePrunedSubTreeCosts(
  node: TMLTreeNodeClusCTP; prmNsa: integer; var costs: double;
  var nbLeaves: integer; var info: TRecInfoNode);
var i: Integer;
begin
 if (node.FCurNsa > prmNsa) and NOT(node.isLeaf())
  Then
   Begin
    For i:= 0 To pred(node.getCountSuccessors) Do
     self.explorePrunedSubTreeCosts(node.getSuccessor(i) as TMLTreeNodeClusCTP,prmNsa,costs,nbLeaves,info);
   End
  //ds le premier trouv, on calcule et on arrte les frais
  Else
   begin
    costs:= costs + node.localCost(ctpPruningSet);
    //passer les autres infos
    info.growCost:= info.growCost + node.localCost(ctpGrowingSet);
    info.pruneS4:= info.pruneS4 + node.FSum4Pruning;
    info.pruneCroise:= info.pruneCroise + node.FSumCroisePruning;
    //
    inc(nbLeaves);
    (*
    //-- affichage de vrification --
    sTmp:= node.DistClass.TabFreq.getStringResult();
    sTmp:= sTmp+format(' -- growErr = %.2f, pruneErr (on %d examples) = %.2f',[node.localCost(cartGrowingSet),node.FPruningCount,node.localCost(cartPruningSet)]);
    TraceLog.WriteToLogFile(sTmp);
    //--
    *)
   end;
end;

function TMLTreeStructureClusCTP.getClassMLTreeNode: TClassMLTreeNode;
begin
 result:= TMLTreeNodeClusCTP; 
end;

function TMLTreeStructureClusCTP.getGrowingSetSize: integer;
begin
 result:= self.FTrainingSet.Size;
end;

function TMLTreeStructureClusCTP.getPruningSetSize: integer;
begin
 result:= self.FPruningSet.Size;
end;

procedure TMLTreeStructureClusCTP.initialiserCurNsa;
var i: integer;
begin
 self.FLstSeqTree.Clear();
 for i:= 0 to pred(self.CountNodes) do
  (self.Node[i] as TMLTreeNodeClusCTP).FCurNsa:= 0;
end;

function TMLTreeStructureClusCTP.isPruningPossible(
  prmNode: TMLTreeNodeClusCTP; prmNsa: Integer; prmAlpha: Double): Boolean;
var cSommet,cFeuille: Double;
begin
 cSommet:= prmNode.localCost(ctpGrowingSet)+prmAlpha;
 //rcuprer les feuilles
 prmNode.getSubTree(prmNsa,self.FLstSeqTree);
 //le cot des feuilles
 cFeuille:= self.computeSubTreeCost(self.FLstSeqTree,ctpGrowingSet);
 //ajouter la complexit
 cFeuille:= cFeuille+prmAlpha*self.FLstSeqTree.Count;
 //alors...
 RESULT:= (cSommet <= cFeuille + CTP_PRUNING_EPSILON);
end;

function TMLTreeStructureClusCTP.mark_nodes(prmNsa: integer;
  prmAlpha: double): boolean;
var the_end: Boolean;
    node: TMLTreeNodeClusCTP;
    okPruning: Boolean;
begin
 RESULT:= FALSE;

 //initialiser
 the_end:= FALSE;
 node:= self.RootNode as TMLTreeNodeClusCTP;

 //la boucle de recherche
 //elle mule une fonction rcursive
 While NOT(the_end) Do
  Begin
   //doit-on traiter le sommet ?
   //c'est le cas si le sommet n'est pas une feuille et qu'elle est non numrote
   if Not(node.isLeaf) and (node.FCurNsa=0)
    Then
     Begin
      //tester si'l est possible d'laguer  partir du sommet courant
      //le test repose sur la recherche des feuilles du sous-arbre
      okPruning:= isPruningPossible(node,prmNsa,prmAlpha);

      //si ok, alors on insre le numro de squence d'arbre sous le sommet, y compris lui mme
      if okPruning
       Then self.mark_subTree(node,prmNsa);

      //si l'on est  la racine et que l'lagage a t accept
      //il n'y a plus rien  faire
      if (node = self.RootNode) and okPruning
       Then
        Begin
         RESULT:= TRUE;
         BREAK;
        End;
      
     End;

   //on passe au sommet suivant

   //tester si c'est une feuille OU si les sommets en dessous sont inintrssants
   if node.isLeaf OR ((node.FCurNsa>0) and (Node.FCurNsa<prmNsa))
    Then
     //on passe au frre
     Begin
      //remonte
      while (node<>self.RootNode) and not(node.nextBrotherExists()) do
       node:= node.Predecessor as TMLTreeNodeClusCTP;

      //passage au frre si l'on n'est pas sur la racine
      if (node<>self.RootNode)
       Then node:= node.getNextBrother()
       Else the_end:= TRUE;
       
     End
    Else
     //on descend sur l'enfant
     Begin
      node:= node.getSuccessor(0) as TMLTreeNodeClusCTP;
     End;
  End;

end;

procedure TMLTreeStructureClusCTP.mark_subTree(prmNode: TMLTreeNodeClusCTP;
  prmNsa: integer);
Var i: Integer;
begin
 If (prmNode.FCurNsa=0)
  Then
   //le sommet n'a pas t pralablement numrot
   Begin
    prmNode.FCurNsa:= prmNsa;
    If NOT(prmNode.isLeaf())
     Then
      Begin
       //appel rcursif vers les feuilles
       For i:= 0 To pred(prmNode.getCountSuccessors()) Do
        self.mark_subTree(prmNode.getSuccessor(i) as TMLTreeNodeClusCTP,prmNsa);
      End;
   End;
end;

procedure TMLTreeStructureClusCTP.numeroterCurNsa(var curNsa: integer);
var curAlpha: double;
begin
 //tout remettre  zro
 self.initialiserCurNsa();
 //dfinir la squence
 curNsa:= 1;
 curAlpha:= 0.0;
 if (self.CountNodes=1)
  //un seul sommet : la racine
  then (self.RootNode as TMLTreeNodeClusCTP).FCurNsa:= curNsa
  else
   //chercher les squences
   begin
    while not(self.mark_nodes(curNsa,curAlpha)) do
     begin
      inc(curNsa);
      curAlpha:= computeNextAlpha(curNsa);
     end;
   end;
end;

procedure TMLTreeStructureClusCTP.PostPruning;
var nbSubTrees: integer;
begin
 //-- appliquer le pruning-set
 if (self.FPruningSet.Size>0)
  then
   begin
     self.applyPruningSet(self.FPruningSet);
     //-- numroter les squences de sous-arbres --
     self.numeroterCurNsa(nbSubTrees);
     //dtecter celle correspondant  l'optimum
     FSelectedSubTree:= self.EnumerateSubTrees(nbSubTrees);
     //effectuer le post-lagage physique
     if (FSelectedSubTree>0)
      then self.pruneSubTree(FSelectedSubTree);
   end
  else TraceLog.WriteToLogFile('[CTP] mode SANS PRUNING asked ???');
end;

procedure TMLTreeStructureClusCTP.prepareInternalStructures;
begin
 inherited;
 //liste non propritaire...
 FLstSeqTree:= TObjectList.Create(FALSE);
end;

procedure TMLTreeStructureClusCTP.pruneSubTree(prmNsa: integer);
begin
 self.pruneSubTreeFromNode(self.RootNode as TMLTreeNodeClusCTP,prmNsa);
end;

procedure TMLTreeStructureClusCTP.pruneSubTreeFromNode(
  prmNode: TMLTreeNodeClusCTP; prmNsa: integer);
Var i: Integer;
begin
 if (prmNode.FCurNsa > prmNsa)
  Then
   Begin
    if NOT(prmNode.isLeaf())
     Then
      Begin
       For i:= 0 To pred(prmNode.getCountSuccessors()) do
        self.pruneSubTreeFromNode(prmNode.getSuccessor(i) as TMLTreeNodeClusCTP,prmNsa);
      End;
   End
  //et ses enfants sont limins par la mme occasion
  Else self.pruningFromNode(prmNode);
end;

procedure TMLTreeStructureClusCTP.pruningFromNode(
  prmNode: TMLTreeNodeClusCTP);
begin
 if not(prmNode.isLeaf())
  then
   begin
    prmNode.pruneFromHere;
    //et ne pas oublier de s'ajouter dans la liste des feuilles
    self.addLeaveReference(prmNode);
   end;
end;

{ TCalcClusTreeCTP }

function TCalcClusTreeCTP.getClassTreeStructure: TClassMLTreeStrucClus;
begin
 result:= TMLTreeStructureClusCTP;
end;

function TCalcClusTreeCTP.getHTMLSpecificDescription: string;
var i: integer;
    arbre: TMLTreeStructureClusCTP;
    tmp,colorLine: string;
    ok_NB_SEQ,curOk: boolean;
begin
 arbre:= FTree as TMLTreeStructureClusCTP;
 ok_Nb_SEQ:= (Length(arbre.TabTreeSequence)<=SEUIL_NB_SEQUENCES);
 tmp:= format('<H3>Trees sequence (# %d) -- Inertia Within-Groups</H3>',[Length(arbre.TabTreeSequence)]);;
 tmp:= tmp+HTML_HEADER_TABLE_RESULT;
 tmp:= tmp+HTML_TABLE_COLOR_HEADER_GRAY+'<TH>N</TH><TH># Leaves</TH><TH>Inertia (growing set)</TH><TH>Inertia (pruning set)</TH></TR>';
 for i:= high(arbre.TabTreeSequence) downto low(arbre.TabTreeSequence) do
  begin
   curOk:= ok_Nb_SEQ OR (self.PrmCalc as TOpPrmClusTreeCART).ShowAllTreeSeq;
   //si a passe pas, un des cas suivant peut faire passer
   curOk:= curOk OR
           ((i=low(arbre.TabTreeSequence))
             or (i=high(arbre.TabTreeSequence))
             or (arbre.TabTreeSequence[i].NumTree = arbre.OptimalSubTree)
             or (arbre.TabTreeSequence[i].NumTree = arbre.SelectedSubTree)
           );
   //suite normale... si ok...
   if curOk
    then
     begin
       colorLine:= HTML_TABLE_COLOR_DATA_GRAY;
       if (arbre.TabTreeSequence[i].NumTree = arbre.OptimalSubTree)
        then colorLine:= HTML_TABLE_COLOR_DATA_GREEN;
       if (arbre.TabTreeSequence[i].NumTree = arbre.SelectedSubTree)
        then colorLine:= HTML_TABLE_COLOR_DATA_RED;
       tmp:= tmp+colorLine+format('<TD align="right">%d</TD><TD align="right">%d</TD><TD align="right">%.4f</TD><TD align="right">%.4f</TD></TR>',
                                 [arbre.TabTreeSequence[i].NumTree,arbre.TabTreeSequence[i].NbLeaves,arbre.TabTreeSequence[i].wssGrow,arbre.TabTreeSequence[i].wssPrune]);
     end;
  end;
 tmp:= tmp+'</table>';
 result:= tmp;
end;

end.
