(*****************************************************************************)
(* UCalcFactDiscriminantAnalysis.pas - Copyright (c) 2005 Ricco RAKOTOMALALA *)
(*****************************************************************************)

{
@abstract(Analyse Factorielle Discriminante -- classes de calcul)
@author(Ricco)
@created(10/08/2005)

Plus connue sous l'appelation "Canonical Discriminant Analysis" chez les anglo-saxons.
Implmentation de l'algo dcrit dans le bouquin de Lebart et al., p.257.

Excellent l'astuce du passage  une matrice symtrique pour le calcul des valeurs/vecteurs propres !!!

}

unit UCalcFactDiscriminantAnalysis;

interface

USES
   Contnrs,
   UDatasetDefinition,
   UDatasetExamples,
   UCalcStatDes,
   Matrices,
   UCalcStatDesConditionnalDesc;

TYPE
    //type de corrlation
    TEnumCDACorrType = (cda_corr_T,cda_corr_W,cda_corr_B);

    //matrice de structure, recensant les corrlations brutes, within et between
    TCDAStrucMatrix = class
                      private
                      //nombre de lignes des tableaux
                      FRowCount: integer;
                      //nombre de colonnes
                      FColCount: integer;
                      //corrlation totale
                      FTabTCorr: array of array of double;
                      //corrlation within
                      FTabWCorr: array of array of double;
                      //corrlation between
                      FTabBCorr: array of array of double;
                      public
                      constructor create(prmNbAtts, prmNbRoots: integer);
                      destructor  destroy(); override;
                      function    getValue(corr: TEnumCDACorrType; j,k: integer): double;
                      procedure   setValue(corr: TEnumCDACorrType; j,k: integer; value: double);
                      property    RowCount: integer read FRowCount;
                      property    ColCount: integer read FColCount;
                      end;

    //classe de calcul FACT-DISC
    TCalcFactDiscriminant = class
                            private
                            //nombre de groupes
                            FNbGroups: integer;
                            //dimension de reprsentation
                            FDimension: integer;
                            //nombre de racines extraites
                            FNbRacines: integer;
                            //nombre d'exemples
                            FNbExamples: integer;
                            //pointeur sur les exemples
                            FExamples: TExamples;
                            //liste d'exemples par groupe
                            FGroupExamples: TObjectList;
                            //attribut cible
                            FTarget: TAttribute;
                            //les descripteurs
                            FInputs: TLstAttributes;
                            //stats conditionnelles (par rapport  la variable groupe) --> liste de liste de stats
                            FStatGroups: TObjectList;
                            //stat globales sur les descripteurs
                            FStatGlobal: TLstCalcStatDesContinuous;
                            //tableau des effectifs par groupe
                            FTabSizeGroups: array of integer;
                            {vecteur des valeurs propres}
                            FEgVal: PVector;
                            {matrice des vecteurs propres}
                            FEgVec: PMatrix;
                            {vecteur des constantes des coefficients des fonctions discriminantes}
                            FVecConstCoef: PVector;
                            //tableau des rsultats sur les valeurs propres
                            FTabEgVal: array of array of double;
                            //tableau des corrlations T,W,B
                            FStrucMatrix: TCDAStrucMatrix;
                            //tableau des cart-types WITHIN pour chaque descripteur
                            FTabPooledWStdDev: array of double; 
                            //calculer les stats descriptives
                            procedure   computeDescStats(prmExamples: TExamples);
                            //calculer les statistiques
                            procedure   computeStatistics();
                            //calculer la corrlation within
                            function    computeWithinCorr(statAxis: TCalcSDCondDescANOVA; numDescriptor: integer): double;
                            //caluler la corrlation between
                            function    computeBetweenCorr(statAxis: TCalcSDCondDescANOVA; numDescriptor: integer): double;
                            public
                            //passer les paramtres
                            constructor create(prmTarget: TAttribute; prmInputs: TLstAttributes);
                            //destructor
                            destructor  destroy(); override;
                            //envoyer les rsultats en HTML
                            function    getHTMLResults(): string;
                            //lancer les calculs
                            function    compute(prmExamples: TExamples): boolean;
                            //effectuer la projection sur les axes factoriels
                            procedure   setProjection(lstAxis: TLstAttributes);
                            //calculer les corrlations T, W et B -- Tenenhaus, p.248 
                            procedure   computeCorrelations(lstAxis: TLstAttributes);
                            //proprits
                            property    NbRacines: integer read FNbRacines;
                            end;

implementation

USES
    Sysutils,Math,
    UCalcMatrixToAttributes, UCalcMatrixAdditionalFunctions, ULogFile,
  FMath, UConstConfiguration, UCalcStatDesCorrelation;

TYPE
    //types d'affichage pour les valeurs propres
    TEnumCDAEigenValues = (cda_EgVal,cda_EgProportion,cda_CanonicalR,cda_Wilks,cda_Chi2,cda_df,cda_pValue);

//produit matriciel -- (nbCol1 = nbLig2) obligatoirement
function matProd(M1,M2: PMatrix; nbLig1, nbCol1, nbCol2: integer): PMatrix;
var mat: PMatrix;
    v: double;
    i,j,k: integer;
begin
 dimMatrix(mat,nbLig1,nbCol2);
 For i:=1 to nbLig1 Do
  For j:=1 to nbCol2 Do
   Begin
    v:=0;
    For k:=1 to nbCol1 Do
     v:= v + M1^[i]^[k] * M2^[k]^[j];
    mat^[i]^[j]:= v;
   End;
 result:= mat;
end;

function matTranspose(M: PMatrix; nbLig, nbCol: integer): PMatrix;
var mat: PMatrix;
    i,j: integer;
begin
 dimMatrix(mat,nbCol,nbLig);
 for i:= 1 to nbLig do
  for j:= 1 to nbCol do
   mat^[j]^[i]:= M^[i]^[j];
 result:= mat;
end;

{ TCalcFactDiscriminant }

function TCalcFactDiscriminant.compute(prmExamples: TExamples): boolean;
var matT: PMatrix;
    j,k: integer;
    value: double;
    matCPrim,matSym,tmpEgVec: PMatrix;
    matC,matTInv,matProjection: PMatrix;
begin
 //rcuprer le pointeur sur les exemples
 FExamples:= prmExamples;
 //calculs...
 result:= FALSE;
 TRY
 //calculer les stats descriptives
 self.computeDescStats(prmExamples);
 //calculer la matrice T
 matT:= BuildMatVCV(prmExamples,FInputs,vcvNormCentered,FStatGlobal);
 //calculer son inverse
 dimMatrix(matTInv,FDimension,FDimension);
 if (InvMat(matT,1,FDimension,matTInv) = MAT_OK)
  then
   begin
    TRY
    //calculer la matrice C
    dimMatrix(matC,FDimension,FNbGroups);
    for j:= 1 to FDimension do
     for k:= 1 to FNbGroups do
      begin
       //x_barre_k_j - x_barre_j
       value:=  ((FStatGroups.Items[pred(k)] as TLstCalcStatDesContinuous).Stat(pred(j)) as TCalcStatDesContinuous).Average -
                (FStatGlobal.Stat(pred(j)) as TCalcStatDesContinuous).Average;
       //racine(n_k/n) * value [3.3 - 4, Lebart, p.257]
       value:= SQRT((1.0 * FTabSizeGroups[pred(k)])/(1.0 * FNbExamples)) * value;
       //affectation
       matC^[j]^[k]:= value;
       //TraceLog.WriteToLogFile(format('att = %d, group = %d --> value = %.4f',[j,k,value]));
      end;
    //calculer la matrice de projection
    matProjection:= matProd(matTInv,matC,FDimension,FDimension,FNbGroups);
    //calculer sa transpose de C
    matCPrim:= matTranspose(matC,FDimension,FNbGroups);
    //calculer la matrice symtrique
    matSym:= matProd(matCPrim,matProjection,FNbGroups,FDimension,FNbGroups);
    //calculer les valeurs et vecteur propres de cette matrice symtrique
    GetEigensFromSymetricMatrix(matSym,FNbGroups,FEgVal,tmpEgVec);
    //**************************************************************************
    //********* corrections pour obtenir les sorties standard des logiciels*****
    //**************************************************************************
    //recalculer la matrice a (les vraies vecteurs propres de T(-1)B) --> cf. [3.3-5]
    FEgVec:= matProd(matProjection,tmpEgVec,FDimension,FNbGroups,FNbGroups);
    //corriger pour avoir l'expression des valeurs pour W(-1)B --> cf. les deux dernires eq. p.256
    //vecteurs propres
    for j:= 1 to FDimension do
     for k:= 1 to FNbGroups do
      if (FEgVal^[k] < 1.0)
       then FEgVec^[j]^[k]:= FEgVec^[j]^[k] / SQRT(1.0 - FEgVal^[k]);
    //valeur propres
    for k:= 1 to FNbGroups do
     if (FEgVal^[k] < 1.0)
      then FEgVal^[k]:= FEgVal^[k] / (1.0 - FEgVal^[k]);
    //**************************************************************************
    //calculer les statistiques  afficher
    computeStatistics();
    //ok !
    result:= true;
    FINALLY
    //vider les matrices temporaires
    delMatrix(matC,FDimension,FNBGroups);
    //delMatrix(matTInv,FDimension,FNbGroups); --> new -- 30/12/2005 -- c'est de trop cette histoire, pourquoi cette destruction ici ?
    delMatrix(matProjection,FDimension,FNbGroups);
    delMatrix(tmpEgVec,FNbGroups,FNbGroups);
    delMatrix(matSym,FNbGroups,FNbGroups);
    delMatrix(matCPrim,FNbGroups,FDimension);
    END;
   end;
 //vider
 delMatrix(matTInv,FDimension,FDimension);
 delMatrix(matT,FDimension,FDimension);
 EXCEPT
 result:= FALSE;
 END;
end;

function TCalcFactDiscriminant.computeBetweenCorr(statAxis: TCalcSDCondDescANOVA;
  numDescriptor: integer): double;
var A,B,C,denom: double;
    k: integer;
    nk,vAxis,vDescriptor: double;
begin
 //calcul -- corrlation := A / SQRT(B * C) -- Tenenhaus, p.248
 A:= 0.0;
 B:= 0.0;
 C:= 0.0;
 for k:= 1 to FNbGroups do
  begin
   nk:= (1.0 * statAxis.StatCond[pred(k)].NbExamples);
   vAxis:= (statAxis.StatCond[pred(k)].Average - statAxis.StatGlobal.Average);
   vDescriptor:= (((FStatGroups.Items[pred(k)] as TLstCalcStatDesContinuous).Stat(numDescriptor) as TCalcStatDesContinuous).Average - (FStatGlobal.LstStat.Items[numDescriptor] as TCalcStatDesContinuous).Average);
   A:= A + nk * vAxis * vDescriptor;
   B:= B + nk * vAxis * vAxis;
   C:= C + nk * vDescriptor * vDescriptor;
  end;
 denom:= B * C;
 if (denom > 0.0)
  then result:= A / SQRT(denom)
  else result:= 0.0;
end;

procedure TCalcFactDiscriminant.computeCorrelations(
  lstAxis: TLstAttributes);
var k,j: integer;
    axis,descriptor: TAttribute;
    stat: TCalcSDCorrelation;
    statCond: TCalcSDCondDescANOVA;
begin
 //structure
 if (FStrucMatrix <> nil) then FStrucMatrix.Free();
 FStrucMatrix:= TCDAStrucMatrix.create(FDimension,lstAxis.Count); 
 //pour chaque axe factoriel
 for k:= 1 to lstAxis.Count do
  begin
   axis:= lstAxis.Attribute[pred(k)];
   for j:= 1 to FDimension do
    begin
     descriptor:= FInputs.Attribute[pred(j)];
     //corrlation totale
     stat:= TCalcSDCorrelation.Create(axis,descriptor,FExamples);
     FStrucMatrix.setValue(cda_corr_T,pred(j),pred(k),stat.R);
     stat.Free();
     //calculer les stats condtionnelles de l'axe
     statCond:= TCalcSDCondDescANOVA.Create(axis,FTarget,FExamples);
     //corrlation partielle -- W
     FStrucMatrix.setValue(cda_corr_W,pred(j),pred(k),computeWithinCorr(statCond,pred(j)));
     //corrlation between -- B
     FStrucMatrix.setValue(cda_corr_B,pred(j),pred(k),computeBetweenCorr(statCond,pred(j)));
    end;
  end;
end;

procedure TCalcFactDiscriminant.computeDescStats(prmExamples: TExamples);
var k,j: integer;
begin
 FNbExamples:= prmExamples.Size;
 //calculer les stats.globales
 FStatGlobal.RefreshStat(prmExamples);
 //calculer les stats conditionnelles
 if (FGroupExamples <> nil) then FGroupExamples.Free();
 FGroupExamples:= prmExamples.DispatchExamples(FTarget);
 for k:= 1 to FNbGroups do
  begin
   FStatGroups.Add(TLstCalcStatDesContinuous.Create(FInputs,FGroupExamples.Items[pred(k)] as TExamples));
   FTabSizeGroups[pred(k)]:= (FGroupExamples.Items[pred(k)] as TExamples).Size;
  end;
 //groupExamples.Free();
 //calculer l'cart-type WITHIN pour chaque variable (util pour le calcul des coefficients standardizs)
 setLength(FTabPooledWStdDev,FDimension);
 FillChar(FTabPooledWStdDev[0],sizeof(double)*FDimension,0);
 for j:= 0 to pred(FDimension) do
  begin
   For k:= 1 to FNbGroups do
    FTabPooledWStdDev[j]:= FTabPooledWStdDev[j] + ((FStatGroups.Items[pred(k)] as TLstCalcStatDesContinuous).LstStat.Items[j] as TCalcStatDesContinuous).TSS;
   //et donc
   FTabPooledWStdDev[j]:= SQRT(FTabPooledWStdDev[j] / (1.0 * FNbExamples - 1.0 * FNbGroups));
   //TraceLog.WriteToLogFile(format('POOLED-STDDEV(%d) = %.8f',[succ(j),FTabPooledWStdDev[j]]));
  end;
end;

procedure TCalcFactDiscriminant.computeStatistics;
var k,j: integer;
    lambda: double;
    correction: double;
    value: double;
    sumProportion: double;
begin
 //nombre de racines extraites
 FNbRacines:= MATH.MIN(FDimension,FNbGroups - 1);
 //qqs calculs cumulatifs
 lambda:= 1.0;
 sumProportion:= 0.0;
 for k:= 1 to FNbRacines do
  begin
   lambda:= lambda * (1.0 / (1.0 + FEgVal^[k]));
   sumProportion:= sumProportion + FEgVal^[k];
  end;
 //prparer le tableau d'affichage des rsultats
 setLength(FTabEgVal,FNbRacines,succ(ord(high(TEnumCDAEigenValues))));
 value:= 0.0;
 for k:= 1 to FNbRacines do
  begin
   FTabEgVal[pred(k),ord(cda_EgVal)]:= FEgVal^[k];
   //cumul proportion de variance explique
   value:= value + FEgVal^[k];
   if (sumProportion > 0.0)
    then FTabEgVal[pred(k),ord(cda_EgProportion)]:=  value / sumProportion;
   //
   FTabEgVal[pred(k),ord(cda_CanonicalR)]:= SQRT(FEgVal^[k] / (1.0 + FEgVal^[k]));
   FTabEgVal[pred(k),ord(cda_Wilks)]:= lambda;
   //Tenenhaus, p.251 --> Transformation de Bartlett
   FTabEgVal[pred(k),ord(cda_Chi2)] := -1.0 * (1.0 * FNbExamples - 1.0 - (FDimension + FNbGroups) / 2.0) * LN(lambda);
   FTabEgVal[pred(k),ord(cda_df)]   := (FDimension + 1 - k) * (FNbGroups - k);
   FTabEgVal[pred(k),ord(cda_pValue)]:= PKhi2(TRUNC(FTabEgVal[pred(k),ord(cda_df)]),FTabEgVal[pred(k),ord(cda_Chi2)]);
   //correction pour la racine suivante
   lambda:= lambda * (1.0 + FEgVal^[k]);
  end;
 //recalcul des vecteurs propres --> functions discriminantes brutes (objectif les rsultats de STATISTICA)
 correction:= 1.0 * (- 0.5 * FNbGroups + FNbExamples) / (1.0 * FNbExamples);
 for j:= 1 to FDimension do
  for k:= 1 to FNbRacines do
   FEgVec^[j]^[k]:= FEgVec^[j]^[k] * correction / FTabEgVal[pred(k),ord(cda_CanonicalR)];
 //***********************************************************
 //les constantes de la fonction discriminantes brutes
 dimVector(FVecConstCoef,FNbRacines);
 for k:= 1  to FNbRacines do
  begin
   value:= 0.0;
   for j:= 1 to FDimension do
    value:= value + FEgVec^[j]^[k] * (FStatGlobal.LstStat[pred(j)] as TCalcStatDesContinuous).Average;
   FVecConstCoef^[k]:= -1.0 * value;
  end;
end;

function TCalcFactDiscriminant.computeWithinCorr(statAxis: TCalcSDCondDescANOVA;
  numDescriptor: integer): double;
var A,B,C,denom: double;
    i,k: integer;
    examples: TExamples;
    vAxis, vDescriptor: double;
begin
 //Tenenhaus, p.248 -- A / SQRT(B*C)
 A:= 0.0;
 B:= 0.0;
 C:= 0.0;
 //pour chaque groupe
 for k:= 1 to FNbGroups do
  begin
   examples:= FGroupExamples.Items [pred(k)] as TExamples;
   for i:= 1 to examples.Size do
    begin
     vAxis:= statAxis.Attribute.cValue[examples.Number[i]] - statAxis.StatCond[pred(k)].Average;
     vDescriptor:= FInputs.Attribute[numDescriptor].cValue[examples.Number[i]] - ((FStatGroups.Items[pred(k)] as TLstCalcStatDesContinuous).Stat(numDescriptor) as TCalcStatDesContinuous).Average; 
     A:= A + vAxis * vDescriptor;
     B:= B + vAxis * vAxis;
     C:= C + vDescriptor * vDescriptor; 
    end;
  end;
 //
 denom:= B * C;
 if (denom > 0.0)
  then result:= A / SQRT(denom)
  else result:= 0.0;
end;

constructor TCalcFactDiscriminant.create(prmTarget: TAttribute;
  prmInputs: TLstAttributes);
begin
 inherited Create();
 //rcuprer les variables
 FTarget:= prmTarget;
 FInputs:= prmInputs;
 //qqs initialisation
 FNbGroups:= FTarget.nbValues;
 FDimension:= FInputs.Count;
 setLength(FTabSizeGroups,FNbGroups);
 FStatGroups:= TObjectList.Create(TRUE);
 FStatGlobal:= TLstCalcStatDesContinuous.Create(FInputs,NIL);
end;

destructor TCalcFactDiscriminant.destroy;
begin
 if (FTabPooledWStdDev <> nil) then Finalize(FTabPooledWStdDev);
 if (FStrucMatrix <> nil) then FreeAndNil(FStrucMatrix);
 if (FGroupExamples <> nil) then FreeAndNil(FGroupExamples);
 if (FVecConstCoef <> nil) then delVector(FVecConstCoef,FNbRacines);
 if (FTabEgVal <> nil) then Finalize(FTabEgVal);
 if (FEgVal <> nil) then delVector(FEgVal,FNbGroups);
 if (FEgVec <> nil) then delMatrix(FEgVec,FDimension,FNbGroups);
 if (FStatGroups <> nil) then FreeAndNil(FStatGroups);
 if (FStatGlobal <> nil) then FreeAndNil(FStatGlobal);
 if (FTabSizeGroups <> nil) then Finalize(FTabSizeGroups);
 inherited destroy();
end;

function TCalcFactDiscriminant.getHTMLResults: string;
var k,j: integer;
    s: string;
begin
 //***********************************************
 //synthse des rsultats -- cf. modle STATISTICA
 //***********************************************
 s:= '<H3>Roots and Wilks'' Lambda</H3>';
 s:= s + HTML_HEADER_TABLE_RESULT;
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY +
         '<TH width=70>Root</TH><TH width=70>Eigenvalue</TH><TH width=70>Proportion</TH><TH width=70>Canonical R</TH><TH width=70>Wilks Lambda</TH><TH width=70>CHI-2</TH><TH width=70>d.f.</TH><TH width=70>p-value</TH></TR>';
 for k:= 1 to FNbRacines do
  begin
   s:= s + HTML_TABLE_COLOR_DATA_GRAY + format('<TH>%d</TH>',[k]);
   s:= s + format('<TD align="right">%.5f</TD>',[FTabEgVal[pred(k),ord(cda_EgVal)]]);
   s:= s + format('<TD align="right">%.5f</TD>',[FTabEgVal[pred(k),ord(cda_EgProportion)]]);
   s:= s + format('<TD align="right">%.6f</TD>',[FTabEgVal[pred(k),ord(cda_CanonicalR)]]);
   s:= s + format('<TD align="right">%.6f</TD>',[FTabEgVal[pred(k),ord(cda_Wilks)]]);
   s:= s + format('<TD align="right">%.4f</TD>',[FTabEgVal[pred(k),ord(cda_Chi2)]]);
   s:= s + format('<TD align="right">%.0f</TD>',[FTabEgVal[pred(k),ord(cda_df)]]);
   s:= s + format('<TD align="right">%.6f</TD>',[FTabEgVal[pred(k),ord(cda_pValue)]]);
  end;
 s:= s + '</table>';

 //*********************************************
 //les coefficients de la fonction discriminante
 //*********************************************
 s:= s + '<H3>Canonical Discriminant Function</H3>';
 s:= s + HTML_HEADER_TABLE_RESULT;
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY +
         format('<TH>Coefficients</TH><TH colspan=%d %s>Unstandardized</TH><TH colspan=%d %s>Standardized</TH></TR>',
                [FNbRacines,HTML_BGCOLOR_HEADER_BLUE,FNbRacines,HTML_BGCOLOR_HEADER_GREEN]);
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY + '<TH>Attribute</TH>';
 for k:= 1 to FNbRacines do s:= s + format('<TH width=70>Root n%d</TH>',[k]);
 for k:= 1 to FNbRacines do s:= s + format('<TH width=70>Root n%d</TH>',[k]);
 s:= s + '</TR>';
 //coeficients bruts sur les variables
 for j:= 1 to FDimension do
  begin
   s:= s + HTML_TABLE_COLOR_DATA_GRAY +format('<TD>%s</TD>',[FInputs.Attribute[pred(j)].Name]);
   //coefficients bruts
   for k:= 1 to FNbRacines do
    s:= s + format('<TD align=right %s>%.4f</TD>',[HTML_BGCOLOR_DATA_BLUE,FEgVec^[j]^[k]]);
   //coefficients standardizs
   for k:= 1 to FNbRacines do
    s:= s + format('<TD align=right %s>%.4f</TD>',[HTML_BGCOLOR_DATA_GREEN,FEgVec^[j]^[k] * FTabPooledWStdDev[pred(j)]]);
   //fin de ligne
   s:= s + '</TR>';
  end;
 //et la constante
 s:= s + HTML_TABLE_COLOR_DATA_GRAY + '<TD>constant</TD>';
 for k:= 1 to FNbRacines do
  s:= s + format('<TD align=right %s>%.4f</TD>',[HTML_BGCOLOR_DATA_BLUE,FVecConstCoef^[k]]);
 s:= s + format('<TD colspan=%d align=center %s>-</TD>',[FNbRacines,HTML_BGCOLOR_DATA_GREEN]);
 s:= s + '</TR>';
 s:= s + '</table>';

 //******************************************************
 //tableau de structure -- Les corrlations avec les axes
 //******************************************************
 s:= s + '<H3>Factor Structure Matrix - Correlations</H3>';
 s:= s + HTML_HEADER_TABLE_RESULT;
 //description de l'en-tte du tableau
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY + '<TH>Root</TH>';
 for k:= 1 to FStrucMatrix.ColCount do
  s:= s + format('<TH colspan=3>Root n%d</TH>',[k]);
 s:= s + '</TR>';
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY + '<TH>Descriptors</TH>';
 for k:= 1 to FStrucMatrix.ColCount do
  s:= s + '<TH width=50>Total</TH><TH  width=50>Within</TH><TH  width=50>Between</TH>';
 s:= s + '</TR>';
 //pour chaque variable
 for j:= 1 to FStrucMatrix.RowCount do
  begin
   s:= s + HTML_TABLE_COLOR_DATA_GRAY + format('<TD>%s</TD>',[FInputs.Attribute[pred(j)].Name]);
   for k:= 1 to FStrucMatrix.ColCount do
    begin
     s:= s + format('<TD align="right">%.3f</TD>',[FStrucMatrix.getValue(cda_corr_T,pred(j),pred(k))]);
     s:= s + format('<TD align="right">%.3f</TD>',[FStrucMatrix.getValue(cda_corr_W,pred(j),pred(k))]);
     s:= s + format('<TD align="right">%.3f</TD>',[FStrucMatrix.getValue(cda_corr_B,pred(j),pred(k))]);
    end;
   s:= s + '</TR>';
  end;
 s:= s + '</table>';

 //
 result:= s;
end;

procedure TCalcFactDiscriminant.setProjection(lstAxis: TLstAttributes);
var i,j,k: integer;
    axis: TAttribute;
    value: single;
begin
 //pour chaque axe
 for k:= 1 to lstAxis.Count do
  begin
   axis:= lstAxis.Attribute[pred(k)];
   //prdiction pour l'individu i
   for i:= 1 to axis.Size do
    begin
     //passer en revue les coefs bruts (produit scalaire)
     value:= FVecConstCoef^[k];
     for j:= 1 to FDimension do
      value:= value + FEgVec^[j]^[k] * FInputs.Attribute[pred(j)].cValue[i];
     //attribuer sa valeur
     axis.cValue[i]:= value;
    end;
  end;
 //et puisqu'on est l, autant calculer les corrlations avec les axes
 computeCorrelations(lstAxis);
end;

{ TCDAStrucMatrix }

constructor TCDAStrucMatrix.create(prmNbAtts, prmNbRoots: integer);
begin
 inherited Create();
 FRowCount:= prmNbAtts;
 FColCount:= prmNbRoots;
 setLength(FTabTCorr,prmNbAtts,prmNbRoots);
 setLength(FTabWCorr,prmNbAtts,prmNbRoots);
 setLength(FTabBCorr,prmNbAtts,prmNbRoots);
end;

destructor TCDAStrucMatrix.destroy;
begin
 Finalize(FTabBCorr);
 Finalize(FTabWCorr);
 Finalize(FTabTCorr);
 inherited destroy();
end;

//TEnumCDACorrType = (cda_corr_T,cda_corr_W,cda_corr_B);

function TCDAStrucMatrix.getValue(corr: TEnumCDACorrType; j,
  k: integer): double;
begin
 case corr of
  cda_corr_W: result:= FTabWCorr[j,k];
  cda_corr_B: result:= FTabBCorr[j,k]
  else result:= FTabTCorr[j,k];
 end;
end;

procedure TCDAStrucMatrix.setValue(corr: TEnumCDACorrType; j, k: integer;
  value: double);
begin
 case corr of
  cda_corr_W: FTabWCorr[j,k]:= value;
  cda_corr_B: FTabBCorr[j,k]:= value
  else FTabTCorr[j,k]:= value;
 end;
end;

end.
