(***************************************************************)
(* UCalcFactNIPALS.pas - Copyright (c) 2005 Ricco RAKOTOMALALA *)
(***************************************************************)

{
@abstract(NIPALS -- classes de calcul)
@author(Ricco)
@created(24/03/2005)

Implmentation de l'algo dcrit dans le bouquin de TENEHAUS, pp. 62
}

unit UCalcFactNIPALS;

interface

USES
   UDatasetDefinition,
   UDatasetExamples;

TYPE

   {type de donnes utilises -- penser  la mmoire}
   TDataNIPALS = single;

   {vecteur de valeurs}
   TVectorNIPALS = array of TDataNIPALS;

   {matrice de valeurs}
   TMatrixNIPALS = array of array of TDataNIPALS;

   {donnes stats  partir d'un ensemble de variables}
   TStatNIPALS = class
                 private
                 //taille d'chantillon
                 FSize: integer;
                 //nombre de variables
                 FCount: integer;
                 //moyennes
                 FAvg: TVectorNIPALS;
                 //ecart-types
                 FStd: TVectorNIPALS;
                 //run
                 procedure runComputation(source: TLstAttributes; examples: TExamples);
                 public
                 constructor create(source: TLstAttributes; examples: TExamples);
                 destructor  destroy(); override;
                 property    Size: integer read FSize;
                 property    Count: integer read FCount;
                 end;

   {classe de calcul}
   TCalcNIPALS = class
                 private
                 //savoir si on divise par l'cart-type ou pas...
                 FStandardize: boolean;
                 //vecteur des cart-types  utiliser -- dpend du paramtre FStandardize
                 FUsedStdDev: TVectorNIPALS;
                 //Valeur de la variance totale -- somme du carr des cart-types
                 FVarianceTotale: TDataNIPALS;
                 //nombre d'observations -- (n)
                 FSize: integer;
                 //nombre de variables -- (p)
                 FCount: integer;
                 //nombre d'axes  produire -- (a)
                 FNbAxis: integer;
                 //source de donnes -- sera utilis pour la projection
                 FDataSource: TLstAttributes;
                 //les statistiques sur chaque variable
                 FStatData: TStatNIPALS;
                 //vecteur des "valeurs propres" -- (a)
                 FLh: TVectorNIPALS;
                 //matrice des "vecteurs propres" -- notation livre Tenenhaus (p x a)
                 FPh: TMatrixNIPALS;
                 //matrices des donnes -- (n x p)
                 FXh: TMatrixNIPALS;
                 //matrices des projections -- (n x a)
                 FTh: TMatrixNIPALS;
                 //initialiser les matrices
                 procedure initMatrix();
                 //remplir la premire version de X -- (Xo)
                 procedure setFirstXMatrix(examples: TExamples);
                 //dtruire les matrices
                 procedure disposeMatrix();
                 //tester si on converge pour l'axe nh
                 function isConvergent(curPh: TVectorNIPALS; h: integer): boolean;
                 //calculer qqs stats rcapitulatives -- les valeurs propres par exemple
                 procedure computeSomeStats();
                 public
                 //rcuprer les donnes
                 constructor create(source: TLstAttributes; examples: TExamples; prmStandardize: boolean; prmNbAxis: integer);
                 //vider
                 destructor  destroy(); override;
                 //lancer les calculs
                 procedure   runAnalysis(examples: TExamples);
                 //renvoyer les rsultats sous forme de tableaux HTML
                 function getHTMLResults(): string;
                 //effectuer une projection
                 procedure   setProjection(prmAtts: TLstAttributes);
                 end;



implementation

USES
       SysUtils, ULogFile, UConstConfiguration;

//constantes de calcul
CONST
       //tolerance pour la convergence
       NIPALS_TOLERANCE = 1.0e-4;

       //nombre d'itration maximum
       NIPALS_MAX_ITERATION = 50;

{ TStatNIPALS }

constructor TStatNIPALS.create(source: TLstAttributes;
  examples: TExamples);
begin
 inherited Create();
 //rcuprer les paramtres
 FSize:= examples.Size;
 FCount:= source.Count;
 //prparer les tableaux
 setLength(FAvg,FCount);
 setLength(FStd,FCount);
end;

destructor TStatNIPALS.destroy;
begin
 setLength(FAvg,0);
 setLength(FStd,0);
 inherited;
end;

procedure TStatNIPALS.runComputation(source: TLstAttributes;
  examples: TExamples);
var sum,sum2: TVectorNIPALS;
    i,j: integer;
    value: TDataNIPALS;
begin
 setLength(sum,FCount);
 setLength(sum2,FCount);
 //zoo...
 for j:= 0 to pred(FCount) do
  begin
   //sommer
   sum[j]:= 0;
   sum2[j]:= 0;
   for i:= 1 to examples.Size do
    begin
     value:= source.Attribute[j].cValue[examples.Number[i]];
     sum[j]:= sum[j]+value;
     sum2[j]:= sum2[j]+value*value;
    end;
   //et rcapituler
   FAvg[j]:= sum[j]/FSize;
   FStd[j]:= sum2[j]-FSize*FAvg[j]*FAvg[j];
   //!\ et on prend l'cart-type sur chantillon (n-1) au dnominateur !!!
   FStd[j]:= SQRT(FStd[j]/(-1.0+FSize));
   //TraceLog.WriteToLogFile(format('[NIPALS] avg-std[%d] = %.4f -- %.4f',[j,FAvg[j],FStd[j]]));
  end;
 //vider
 setLength(sum,0);
 setLength(sum2,0);
end;

{ TCalcNIPALS }

procedure TCalcNIPALS.computeSomeStats;
var h,i: integer;
    prod: TDataNIPALS;
begin
 //appliquer le (1/(n-1))*Th'Th
 for h:= 0 to pred(FNbAxis) do
  begin
   prod:= 0.0;
   for i:= 1 to FSize do
    prod:= prod+FTh[i,h]*FTh[i,h];
   //don't forget le 1/(n-1)
   FLh[h]:= prod/(-1.0+FSize);
  end;
end;

constructor TCalcNIPALS.create(source: TLstAttributes; examples: TExamples;
  prmStandardize: boolean; prmNbAxis: integer);
begin
 inherited Create();
 //rcuprer les paramtres
 FDataSource:= source;
 FStandardize:= prmStandardize;
 FNbAxis:= prmNbAxis;
 FSize:= examples.Size;
 FCount:= source.Count;
 //initialiser les matrices
 self.initMatrix(); 
end;

destructor TCalcNIPALS.destroy;
begin
 FStatData.Free();
 self.disposeMatrix();
 inherited;
end;

procedure TCalcNIPALS.disposeMatrix;
begin
 setLength(FLh,0);
 setLength(FPh,0,0);
 setLength(FXh,0,0);
 setLength(FTh,0,0);
 setLength(FUsedStdDev,0);
end;

function TCalcNIPALS.getHTMLResults(): string;
var h: integer;
    s: string;
    value,cumul: TDataNIPALS;
begin
 s:= '<H3>Eigen values</H3>';
 s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY;
 s:= s+'<TH>N factor</TH><TH>Eigen value</TH><TH>explained (%)</TH><TH>cimulated (%)</TH></TR>';
 cumul:= 0.0;
 for h:= 0 to pred(FNbAxis) do
  begin
   value:= 100.0*FLh[h]/FVarianceTotale;
   cumul:= cumul+value;
   s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>%d</TD><TD align=right>%.6f</TD><TD align=right>%.2f</TD><TD align=right>%.2f</TD></TR>',
                                           [succ(h),FLh[h],value,cumul]);
  end;
 s:= s+'</table>';
 //and then...
 result:= s;
end;

procedure TCalcNIPALS.initMatrix;
begin
 //vecteur des "valeurs propres" -- (a)
 setLength(FLh,FNbAxis);
 //matrice des "vecteurs propres" -- notation livre Tenenhaus (p x a)
 setLength(FPh,FCount,FNbAxis);
 //matrices des donnes -- (n x p)
 setLength(FXh,succ(FSize),FCount);
 //matrices des projections -- (n x a)
 setLength(FTh,succ(FSize),FNbAxis);
 //vecteur des cartypes  utiliser
 setLength(FUsedStdDev,FCount);
end;

function TCalcNIPALS.isConvergent(curPh: TVectorNIPALS;
  h: integer): boolean;
var sum: TDataNIPALS;
    j: integer;
begin
 sum:= 0;
 for j:= 0 to pred(FCount) do
  sum:= sum+ABS(curPh[j]-FPh[j,h]);
 //pour mettre toutes les approches sur un mme pied d'galit
 sum:= sum/(1.0*FCount);
 //ok, si la somme des valeurs absolues est infrieure au seuil
 result:= (sum <= NIPALS_TOLERANCE);
end;

procedure TCalcNIPALS.runAnalysis(examples: TExamples);
var h,i,j,iteration: integer;
    curPh: TVectorNIPALS;
    prodTh,sumPh: TDataNIPALS;
    okConvergence: boolean;
begin

 //calculer les stats
 FStatData:= TStatNIPALS.create(FDataSource,examples);
 FStatData.runComputation(FDataSource,examples);
 
 //remplir la premire version de Xh i.e. Xo
 self.setFirstXMatrix(examples);

 //Rserver la mmoire de Ph courant
 setLength(curPh,FCount);

 //pour chaque axe  produire -- h = 1...a
 for h:= 0 to pred(FNbAxis) do
  begin

   //copier la premire colonne de Xh-1 dans Th
   for i:= 1 to FSize do
    FTh[i,h]:= FXh[i,0];

   //initialiser le vecteur Ph courant
   for j:= 0 to pred(FCount) do curPh[j]:= 0.0;

   //*** rpeter jusqu' convergence ***
   iteration:= 1;
   REPEAT
    //calculer Th'Th
    prodTH:= 0.0;
    for i:= 1 to FSize do prodTh:= prodTh+FTh[i,h]*FTh[i,h];
  
    //calculer Ph
    for j:= 0 to pred(FCount) do
     begin
      sumPh:= 0.0;
      for i:= 1 to FSize do
       sumPh:= sumPh+FXh[i,j]*FTh[i,h];
      //normaliser par Th'Th
      if (prodTh>0)
       then sumPh:= sumPh/prodTh;
      //acffecter  Ph
      curPh[j]:= sumPh;
     end;

    //normer Ph  1 -- i.e. la somme des carrs des valeurs doit tre gal  1 !!!
    sumPh:= 0.0;
    for j:= 0 to pred(FCount) do sumPh:= sumPh+curPh[j]*curPh[j];
    if (sumPh>0)
     then for j:= 0 to pred(FCount) do curPh[j]:= curPh[j]/SQRT(sumPh);

    //recalculer la projection Th
    for i:= 1 to FSize do
     begin
      prodTh:= 0.0;
      for j:= 0 to pred(FCount) do
       prodTh:= prodTh+FXh[i,j]*curPh[j];
      //affecter
      FTh[i,h]:= prodTh;
     end;

   //tester la convergence
   okConvergence:= isConvergent(curPh,h);

   //rcuprer le Ph
   for j:= 0 to pred(FCount) do
    FPh[j,h]:= curPh[j];

   inc(iteration);

   UNTIL (iteration>NIPALS_MAX_ITERATION) or okConvergence;
   //*** ok pour la convergence ***

   //contrle, la convergence a-t-elle t force ?
   if (iteration>NIPALS_MAX_ITERATION)
    then TraceLog.WriteToLogFile(Format('[NIPALS] *** warning *** force convergence for axis n%d',[succ(h)]));

   //nouvelle version de Xh -- tape 2.3
   for i:= 1 to FSize do
    begin
     for j:= 0 to pred(FCount) do
      FXh[i,j]:= FXh[i,j]-FTh[i,h]*curPh[j];
    end;

  end;

 //vider la mmoire de curPh
 setLength(curPh,0);

 //lancer les calculs rcapitulatifs -- valeur propre  partir des projections
 self.computeSomeStats();

 //les matrices de donnes (rsidus de la rgression) et de projections sont maintenant inutiles
 //autant les virer ds maintenant... d'autant plus qu'ils peuvent tre volumineux !!!
 setLength(FXh,0,0);
 setLength(FTh,0,0);
end;

procedure TCalcNIPALS.setFirstXMatrix(examples: TExamples);
var i,j: integer;
    example: integer;
begin
 //initialiser le vecteur des cart-types
 if not(FStandardize)
  then
   begin
    for j:= 0 to pred(FCount) do FUsedStdDev[j]:= 1.0;
    //trace de la matrice de variance co-variance (et non plus de corrlation puisque analyse non-norme)
    FVarianceTotale:= 0.0;
    for j:= 0 to pred(FCount) do
     FVarianceTotale:= FVarianceTotale+SQR(FUsedStdDev[j]);
   end
  else
   begin
    //moche, moche, mais bon...
    for j:= 0 to pred(FCount) do FUsedStdDev[j]:= FStatData.FStd[j];
    //et donc la trace de la matrice de corrlation...
    FVarianceTotale:= 1.0*FCount;
   end;
 //une fois pour toutes la pondration -- on utilise bien 1/(n-1) -- cf. Tenenhaus, pp. 63 -- non, non, inutile !!!
 //nM1:= 1.0/(-1.0+FSize);
 //calculer la variance totale, pour le pourcentage expliqu -- ncessairement suprieur  zro
 //TraceLog.WriteToLogFile(Format('[NIPALS] total variance = %.4f',[FVarianceTotale]));
 //zoo...
 for i:= 1 to FSize do
  begin
   example:= examples.Number[i];
   //pondre, centre et ventuellement rduite...
   for j:= 0 to pred(FCount) do
    begin
     if (FUsedStdDev[j]>0)
      then
       begin
        //TRY
        FXh[i,j]:= (FDataSource.Attribute[j].cValue[example]-FStatData.FAvg[j])/FUsedStdDev[j];
        //EXCEPT
        //TraceLog.WriteToLogFile(Format('[NIPALS] *** error *** Xh[%d,%d] for example n%d',[i,j,example]));
        //END;
       end
      else FXh[i,j]:= 0.0;
    end;
  end;
end;

procedure TCalcNIPALS.setProjection(prmAtts: TLstAttributes);
var i,j,h: integer;
    axis,att: TAttribute;
    s: TTypeContinue;
begin
 //pour chaque axe  manipuler
 for h:= 0 to pred(FNbAxis) do
  begin
   axis:= prmAtts.Attribute[h];
   //pour chaque individu
   for i:= 1 to axis.Size do
    begin
     s:= 0.0;
     //pour chaque variable dans l'ensemble de donnes
     for j:= 0 to pred(FCount) do
      begin
       att:= self.FDataSource.Attribute[j];
       //appliquer la formule de projection en multipliant avec les vecteurs propres
       if (FUsedStdDev[j]>0)
        then s:= s+FPh[j,h]*(att.cValue[i]-FStatData.FAvg[j])/FUsedStdDev[j];
      end;
     //affecter la coordonne de l'individu sur l'axe
     axis.cValue[i]:= s;
    end;
  end;
end;

end.
