(***************************************************************)
(* JFGCalcPLS.pas - Copyright (c) 2005 Jean-Franois GRANGE *)
(***************************************************************)

{
@abstract(PLS classes de calcul)
@author(Jean-Franois)
@created(t 2005)

Implmentation de l'algorithme de regression PLS 1 et 2  dcrit dans le livre
"La regression PLS" de TENENHAUS

R.R. -- modif. 23/10/2005
(1) modification de l'affichage pour coller  l'affichage d'autres logiciels connus
(2) calcul des coefficients de la rgression pour la projection
(3) les donnes sont systmatiquement standardises

R.R. -- modif. 06/03/2006
ajout d'une fonctionnalit de projection  partir des coefficients calculs
//!\ il doit y avoir autant de projections que de variables Y
 
}

unit JFGCalcPLS;

interface

USES
   UDatasetDefinition,
   UDatasetExamples;

TYPE

   {donnes utilises}
   TDataPLS = single;

   {vecteur de valeurs}
   TVectorPLS = array of TDataPLS;

   {matrice de valeurs}
   TMatrixPLS = array of array of TDataPLS;

   {Classe statistique}
   TStatPLS = class

                 private

                        //taille d'chantillon
                        FSize: integer;

                        //nombre de variables X
                        FCountX: integer;

                        //nombre de variables Y
                        FCountY: integer;

                        //moyennes des X
                        FAvgX: TVectorPLS;

                        //ecart-types des X
                        FStdX: TVectorPLS;

                        //moyennes des Y
                        FAvgY: TVectorPLS;

                        //ecart-types des Y
                        FStdY: TVectorPLS;

                        //calcul des stats
                        procedure runComputation(source: TLstAttributes; examples: TExamples; FCount:integer; var FAvg:TVectorPLS; var FStd: TVectorPLS);

                 public


                        constructor create(sourceInput: TLstAttributes; sourceTarget: TLstAttributes;examples: TExamples);
                        destructor  destroy(); override;
                        property    Size: integer read FSize;
                        property    CountX: integer read FCountX;
                        property    CountY: integer read FCountY;
                        property    AvgY: TVectorPLS read FAvgY;                       
   end;


   {classe de calcul}
   TCalcPLS = class

                 private

                 //new -- 06/03/2006 -- rsultat dtaill
                 FDetailedResult: boolean;

                 //standardisation ou pas ( division par l'cart-type ou non)
                 FStandardize: boolean;

                 //vecteur des cart-types  utiliser
                 FUsedStdDevX: TVectorPLS;
                 FUsedStdDevY: TVectorPLS;

                 //Valeur de la variance totale
                 FVarianceTotaleX: TDataPLS;
                 FVarianceTotaleY: TDataPLS;

                 //nombre d'observations
                 FSize: integer;

                 //nombre de variables X
                 FCountX: integer;

                 //nombre de variables Y
                 FCountY: integer;

                 //nombre d'axes  produire
                 FNbAxis: integer;



                 // les sources de donnes
                 FDataSourceInput: TLstAttributes;
                 FDataSourceTarget: TLstAttributes;

                 //les statistiques sur chaque variables
                 FStatData: TStatPLS;


                 //matrice des coefficients de regression de Th dans la regression de
                 //Xh-1 sur Th
                 FPh: TMatrixPLS;

                 //matrices des donnes des variables explicatives
                 FXh: TMatrixPLS;

                 //matrices des donnes des variables  expliquer
                 FYh:TMatrixPLS;

                 //matrices des prdictions
                 FYprev:TMatrixPLS;
                 Fresidus:TMatrixPLS;

                 //R.R. -- matrice de coefficients de la rgression
                 FBCoefReg: TMatrixPLS;

                 //matrices des composantes
                 FTh: TMatrixPLS;

                 //matrices de composantes
                 FUh: TMatrixPLS;

                  //matrice des coefficients de regression de Yh-1 dans la regression de
                 //Xh-1 sur Yh-1
                 FWh: TMatrixPLS;
                 FWhseconde: TMatrixPLS;


                 //matrice des coefficients de rgression de th dans la rgression de la variable
                 //yh-1 sur la variable th
                 FCh: TMatrixPLS;

                 //matrices des correlation
                 FCorXT:TMatrixPLS;
                 FCorYT:TMatrixPLS;

                 FRdXT:TVectorPLS;
                 FRdYT:TVectorPLS;

                 //coefficients de regression
                 FCoefReg:TMatrixPLS;

                 //Matrice des quantits RES et PRESS ( permettent de calculer le coefficient Q indicateur de qualit )
                 FRES:  TMatrixPLS;
                 FPRESS:  TMatrixPLS;
                // FQ2: TMatrixPLS;

                 FVIP:TMatrixPLS;

                 //initialiser les matrices
                 procedure initMatrix();

                 //remplir la premire version d'une matrice de donnes
                 procedure setFirstMatrix(examples: TExamples; FDataSource:TLstAttributes; Count: Integer; var FVarianceTotale:TDataPLS;var FUsedStdDev:TVectorPLS; FAvg:TVectorPLS; FStd:TVectorPLS; var Fh:TMatrixPLS);

                 //dtruire les matrices
                 procedure disposeMatrix();

                 //tester si on converge pour l'axe nh
                 function isConvergent(curWh: TVectorPLS; h: integer): boolean;



                 public

                 //rcuprer les donnes
                 //new -- r.r. -- 06/03/2006 -- passage de "full affichage"
                 constructor create(sourceInput: TLstAttributes; sourceTarget: TLstAttributes; examples: TExamples; prmStandardize: boolean; prmNbAxis: integer; detailedResult: boolean);

                 //vider la memoire
                 destructor  destroy(); override;

                 //lancer les calculs
                 procedure   runAnalysis(examples: TExamples);

                 //tester la qualit des modles par le calcul des quelques coefficients clefs
                 procedure globalGoodness(h:integer;FXhpred:TMatrixPLS;FYhpred:TMatrixPLS;FWhi:TMatrixPLS;FChi:TMatrixPLS; FXhinit:TMatrixPLS; FYhinit:TMatrixPLS);

                 //sorties des reusltats HTML
                 function getHTMLResults(): string;

                 //new -- R.R. -- 06/03/2006 -- projection  partir des valeurs de X, renvoie Y (le X est modifi)
                 procedure projection(var X: TVectorPLS; var Y: TVectorPLS);
                 //new -- R.R. -- 10/03/2006 -- projection directe  partir des listes de X et des listes de Y
                 procedure projectionOnLstX(example: integer; lstX: TLstAttributes; var Y: TVectorPLS);
                 

                 //projection sur toutes les observations
                 procedure setProjection(atts: TLstAttributes);

                 //les stats
                 property  StatPLS: TStatPLS read FStatData;

                 end;



implementation

USES
       SysUtils, Windows, ULogFile, UConstConfiguration, UStringAddBuffered;


CONST
       //tolerance pour la convergence
       PLS_TOLERANCE = 1.0e-8;

       //nombre d'itration maximum
       PLS_MAX_ITERATION = 300;

{ TStatPLS }


constructor TStatPLS.create(sourceInput: TLstAttributes; sourceTarget:TLstAttributes; examples: TExamples);

        begin
                inherited Create();

                //rcuprer les paramtres
                FSize:= examples.Size;
                FCountX:= sourceInput.Count;
                FCountY:= sourceTarget.Count;

                //prparer les tableaux
                setLength(FAvgX,FCountX);
                setLength(FStdX,FCountX);
                setLength(FAvgY,FCountY);
                setLength(FStdY,FCountY);
        end;


destructor TStatPLS.destroy;

        begin
                setLength(FAvgX,0);
                setLength(FStdX,0);
                setLength(FAvgY,0);
                setLength(FStdY,0);
                inherited;
        end;


procedure TStatPLS.runComputation(source: TLstAttributes; examples: TExamples;   FCount:integer;  var FAvg:TVectorPLS;  var FStd: TVectorPLS);

        var sum,sum2: TVectorPLS;
                 i,j: integer;
               value: TDataPLS;

        begin
                //initialisation des vecteurs des calculs
                setLength(sum,FCount);
                setLength(sum2,FCount);

                //Calcul des moyennes et des ecart types des variables
                for j:= 0 to pred(FCount) do

                        begin


                        sum[j]:= 0;
                        sum2[j]:= 0;

                                for i:= 1 to examples.Size do

                                        begin
                                                value:= source.Attribute[j].cValue[examples.Number[i]];
                                                sum[j]:= sum[j]+value;
                                                sum2[j]:= sum2[j]+value*value;
                                        end;

                        FAvg[j]:= sum[j]/FSize;
                        FStd[j]:= sum2[j] - FSize*FAvg[j]*FAvg[j];
                        FStd[j]:= SQRT(FStd[j]/(-1.0+FSize));

                         end;
                //On vide la mmoire
                setLength(sum,0);
                setLength(sum2,0);

        end;



constructor TCalcPLS.create(sourceInput: TLstAttributes; sourceTarget: TLstAttributes; examples: TExamples; prmStandardize: boolean; prmNbAxis: integer; detailedResult: boolean);

        begin
                inherited Create();

                //rcupration des paramtres
                FDataSourceInput:= sourceInput;
                FDataSourceTarget:= sourceTarget;
                FStandardize:= prmStandardize;
                FNbAxis:= prmNbAxis;
                FSize:= examples.Size;
                FCountX:= sourceInput.Count;
                FCountY:=sourceTarget.Count;
                FDetailedResult:= detailedResult;

                //initialiser les matrices
                self.initMatrix();
        end;


destructor TCalcPLS.destroy;

        begin
                FStatData.Free();
                self.disposeMatrix();
                inherited;
        end;


procedure TCalcPLS.disposeMatrix;

        begin
                setLength(Fresidus,0,0);
                setLength(FPh,0,0);
                setLength(FXh,0,0);
                setLength(FYh,0,0);
                setLength(FYprev,0,0);
                setLength(FWh,0,0);
                setLength(FCoefReg,0,0);
                setLength(FWhseconde,0,0);
                setLength(FCh,0,0);
                setLength(FUh,0,0);
                setLength(FTh,0,0);
                setLength(FUsedStdDevX,0);
                setLength(FUsedStdDevY,0);
                setLength(FRES,0,0);
                setLength(FPRESS,0,0);
                //setLength(FQ2,0,0);
                setLength(FCorXT,0,0);
                setLength(FCorYT,0,0);
                setLength(FRdXT,0);
                setLength(FRdYT,0);
                setLength(FVIP,0,0);

                //R.R. -- 23/10/2005
                setlength(FBCoefReg,0,0);

        end;


function TCalcPLS.getHTMLResults(): string;

        var h,k,i,j: integer;

            s: string;
            value,sumValue: TDataPLS;
            buf: TBufString;

                begin
                       { Normalement affichage du coefficient Q2 dont le calcul ne fonctionne hlas pas
                        s:= '<H3>Quality Coefficient</H3>';
                        s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY;
                        s:= s+'<TH>N factor</TH><TH>Q2</TH><TH>Q2 cumul</TH></TR>';
                        cumul:= 0.0;
                        coef := 0.0;
                      for h:= 0 to pred(FNbAxis) do
                               begin
                                       value:=0.0;
                                      for k:= 0 to pred(FCountY) do
                                       begin
                                       value:=FQ2[k,h];
                                       end;
                                       value:=value/FCountY;
                                        s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>%d</TD><TD align=right>%.4f</TD><TD align=right>%.4f</TD></TR>',[succ(h),value,cumul]);
                                  end;
                        s:= s+'</table>';
                       }

                       //R.R. -- new -- 23/10/2005 -- coefficients de la rgression sur variables centres et rduites
                       buf:= TBufString.Create();
                       buf.BeginUpdate();

                       s:= '<H3>Regression coefficients</H3>';
                       s:= s + HTML_HEADER_TABLE_RESULT;
                       //en-tte les variables Y
                       s:= s + HTML_TABLE_COLOR_HEADER_GRAY+'<TH>X/Y</TH>';
                       for i:= 0 to pred(FCountY) do
                        s:= s + format('<TH>%s</TH>',[FDataSourceTarget.Attribute[i].Name]);
                       s:= s + '</TR>';

                       buf.AddStr(s);
                       //les coeffs. pour chaque X
                       for j:= 0 to pred(FCountX) do
                        begin
                         s:= HTML_TABLE_COLOR_DATA_GRAY + format('<TH>%s</TH>',[FDataSourceInput.Attribute[j].Name]);
                         for i:= 0 to pred(FCountY) do
                          s:= s + format('<TD align="right">%.4f</TD>',[FBCoefReg[j,i]]);
                         s:= s + '</TR>';
                         buf.AddStr(s);
                        end;

                       //new -- 10/03/2006 -- dernire ligne pour la constante
                       s:= HTML_TABLE_COLOR_DATA_GRAY + format('<TH>%s</TH>',['constant']);
                       for i:= 0 to pred(FCountY) do
                        s:= s + format('<TD align="right">%.4f</TD>',[FBCoefReg[FCountX,i]]);
                       s:= s + '</TR>';
                       buf.AddStr(s);

                       buf.AddStr('</table>');
                       buf.EndUpdate();
                       //************************************************************************************************

                       s:= buf.BufS;
                       buf.Free();

                       //si rsultat dtaill demand... (attention au temps de formation de la chane !!!)
                       if (FDetailedResult)
                       then
                       begin


                        //****************************************************
                        //aide  l'interprtation -- les R et les redondances
                        //****************************************************
                        s:= s+'<H3>R coefficients and redundancy on inputs (X)</H3>';
                        s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY;
                        s:= s+'<TH>Attribute</TH>';
                        for h:=0 to pred(FNbAxis) do s:= s+format('<TH width=60>Axis_%d</TH>',[h+1]);
                        s:= s+'</TR>';
                        for j:=0 to pred(FCountX) do
                         begin
                          s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TH>%s</TH>',[FDataSourceInput.Attribute[j].Name]);
                          sumValue:= 0.0;
                          for h:=0 to pred(FNbAxis) do
                           begin
                            value:=FCorXT[j,h]*FCorXT[j,h];
                            sumValue:= sumValue + value;
                            s:= s+format('<TD align=right>%.4f <br>(%.4f)</TD>',[value,sumValue]);
                           end;
                          s:= s+'</TR>';
                         end;
                        s:= s + HTML_TABLE_COLOR_DATA_GREEN + '<TH>Redundancy</TH>';
                        sumValue:= 0.0;
                        for h:=0 to pred(FNbAxis) do
                         begin
                          sumValue:= sumValue + FRdXT[h];
                          s:= s+format('<TD align=right>%.4f<br>(%.4f)</TD>',[FRdXT[h],sumValue]);
                         end;
                        s:= s + '</table>';

                        s:= s+'<H3>R coefficients and redundancy on targets (Y)</H3>';
                        s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY;
                        s:= s+'<TH>Attribute</TH>';
                        for h:=0 to pred(FNbAxis) do s:= s+format('<TH width=60>Axis_%d</TH>',[h+1]);
                        s:= s+'</TR>';
                        for k:=0 to pred(FCountY) do
                         begin
                          s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TH>%s</TH>',[FDataSourceTarget.Attribute[k].Name]);
                          sumValue:= 0.0;
                          for h:=0 to pred(FNbAxis) do
                           begin
                            value:= FCorYT[k,h] * FCorYT[k,h];
                            sumValue:= sumValue + value;
                            s:= s+format('<TD align=right>%.4f <br>(%.4f)</TD>',[value,sumValue]);
                           end;
                          s:= s+'</TR>';
                         end;
                        s:= s + HTML_TABLE_COLOR_DATA_GREEN + '<TH>Redundancy</TH>';
                        sumValue:= 0.0;
                        for h:=0 to pred(FNbAxis) do
                         begin
                          sumValue:= sumValue + FRdYT[h];
                          s:= s+format('<TD align=right>%.4f<br>(%.4f)</TD>',[FRdYT[h],sumValue]);
                         end;
                         s:= s+'</table>';

                       //****************************************************
                       //aide  l'interprtation -- VIP
                       //****************************************************
                       s:= s+'<H3>Variable Importance in the Projection</H3>';

                       s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY;
                       s:= s+'<TH>Attribute</TH>';
                       for h:=0 to pred(FNbAxis) do s:= s+format('<TH width=60>Axis_%d</TH>',[h+1]);
                       s:= s+'</TR>';

                       for j:=0 to pred(FCountX) do

                                begin

                                     s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TH>%s</TH>',[FDataSourceInput.Attribute[j].Name]);

                                 for h:=0 to pred(FNbAxis) do

                                                 begin
                                                          value:=FVIP[j,h];
                                                          s:= s+format('<TD align=right>%.4f</TD>',[value]);
                                                 end;

                                         s:= s+'</TR>';
                                 end;

                         s:= s+'</table>';


                        s:= s+'<H3>Vectors Wh </H3>';

                        s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY;

                        s:= s+'<TH>Attribute</TH>';

                        for h:=0 to pred(FNbAxis) do s:= s+format('<TH width=60>Axis_%d</TH>',[h+1]);
                        s:= s+'</TR>';

                        for j:=0 to pred(FCountX) do

                                begin

                                     s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TH>%s</TH>',[FDataSourceInput.Attribute[j].Name]);

                                 for h:=0 to pred(FNbAxis) do

                                                 begin
                                                          value:=FWh[j,h];
                                                          s:= s+format('<TD align=right>%.4f</TD>',[value]);
                                                 end;

                                         s:= s+'</TR>';
                                 end;

                         s:= s+'</table>';

                         {
                         s:= s+'<H3>Vectors Wh* </H3>';
                        s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY;
                        s:= s+'<TH>Attribute</TH>';
                        for h:=0 to pred(FNbAxis) do s:= s+format('<TH>Axis_%d</TH>',[h+1]);
                        s:= s+'</TR>';
                        for j:=0 to pred(FCountX) do
                                begin
                                 s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TH>%s</TH>',[FDataSourceInput.Attribute[j].Name]);
                                 for h:=0 to pred(FNbAxis) do
                                                 begin
                                                          value:=FWhseconde[j,h];
                                                          s:= s+format('<TD align=right>%.4f</TD>',[value]);
                                                 end;
                                         s:= s+'</TR>';
                                 end;
                         s:= s+'</table>'; }


                        s:= s+'<H3>Vectors Ch </H3>';

                        s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY;

                        s:= s+'<TH>Attribute</TH>';

                        for h:=0 to pred(FNbAxis) do s:= s+format('<TH width=60>Axis_%d</TH>',[h+1]);
                        s:= s+'</TR>';

                        for k:=0 to pred(FCountY) do

                                begin

                                     s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TH>%s</TH>',[FDataSourceTarget.Attribute[k].Name]);

                                 for h:=0 to pred(FNbAxis) do

                                                 begin
                                                          value:=FCh[k,h];
                                                          s:= s+format('<TD align=right>%.4f</TD>',[value]);
                                                 end;

                                         s:= s+'</TR>';
                                 end;

                         s:= s+'</table>';



                         s:= s+'<H3>Vectors Ph </H3>';

                        s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY;

                        s:= s+'<TH>Attribute</TH>';

                        for h:=0 to pred(FNbAxis) do s:= s+format('<TH width=60>Axis_%d</TH>',[h+1]);
                        s:= s+'</TR>';

                        for j:=0 to pred(FCountX) do

                                begin

                                     s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TH>%s</TH>',[FDataSourceInput.Attribute[j].Name]);

                                 for h:=0 to pred(FNbAxis) do

                                                 begin
                                                          value:=FPh[j,h];
                                                          s:= s+format('<TD align=right>%.4f</TD>',[value]);
                                                 end;

                                         s:= s+'</TR>';
                                 end;

                         s:= s+'</table>';

                       end;
                       
                       //and then...
                       result:= s;
                end;


procedure TCalcPLS.initMatrix;

        begin


               // Allocation de la mmoire des vecteurs et matrices

               setLength(FPh,FCountX,FNbAxis);
               setLength(FXh,succ(FSize),FCountX);
               setLength(FYh,succ(FSize),FCountY);
               setLength(FYprev,succ(FSize),FCountY);
               setLength(FTh,succ(FSize),FNbAxis);
               setLength(FPh,FCountX,FNbAxis);
               setLength(FUh,succ(FSize),FNbAxis);
               setLength(FCh,FCountY,FNbAxis);
               setLength(FWh,FCountX,FNbAxis);
               setLength(FWhseconde,FCountX,FNbAxis);
               setLength(FCoefReg,FCountX,FCountY);
               setLength(FUsedStdDevX,FCountX);
               setLength(FUsedStdDevY,FCountY);
               setLength(FVIP,FCountX,FNbAxis);
               setLength(FPRESS,FCountY,FNbAxis);
               setLength(FRES,FCountY,FNbAxis);
               setLength(FCorXT,FCountX,FNbAxis);
               setLength(FCorYT,FCountY,FNbAxis);
               setLength(FRdXT,FNbAxis);
               setLength(FRdYT,FNbAxis);
               setLength(Fresidus,succ(FSize),FCountY);
               //setLength(FQ2,FCountY,FNbAxis);

               //R.R. -- coef. B de la rgression -- p.137
               //R.R. -- 10/03/2006 -- ajouter une case pour mettre la constante
               //setlength(FBCoefReg,FCountX,FCountY);
               setlength(FBCoefReg,FCountX+1,FCountY);
        end;


function TCalcPLS.isConvergent(curWh:TVectorPLS; h:integer): boolean;

        var sum: TDataPLS;
              j: integer;

        begin

                sum:= 0;

                for j:= 0 to pred(FCountX) do sum:= sum+ABS(curWh[j]-FWh[j,h]);

                sum:= sum/(1.0*FCountX);

                //On accepte  la condition que la somme des valeurs absolues soient infrieure au seuil de tolrance
                result:= (sum <= PLS_TOLERANCE);

        end;

procedure TCalcPLS.globalGoodness(h:integer; FXhpred:TMatrixPLS; FYhpred:TMatrixPLS; FWhi:TMatrixPLS; FChi:TMatrixPLS; FXhinit:TMatrixPLS; FYhinit:TMatrixPLS);



       var sum1,sum2,sum3,sum4,AvgTh,Avg: TDataPLS;
       i,j,k  :integer;

       // Calcul des coefficients de correlation
        begin

          //AvgTh:=0.0;
          //Avg:=0.0;
          //sum2:=0.0;
          //sum3:=0.0;
          //sum4:=0.0;


                                     
         for j:=0 to pred(FCountX) do

         begin

          AvgTh:=0.0;
          Avg:=0.0;
         sum2:=0.0;
          sum3:=0.0;
          sum4:=0.0;
                                 for i:=1 to FSize do

                                 begin
                                        Avg:=Avg+FXhinit[i,j];


                                  end;
                                  Avg:=Avg/FSize;

                                 for i:=1 to FSize do

                                 begin
                                       
                                       sum2:=sum2+((FXhinit[i,j]-Avg)*(FTh[i,h]-AvgTh));
                                       sum3:=sum3+((FXhinit[i,j]-Avg)*(FXhinit[i,j]-Avg));
                                       sum4:=sum4+((FTh[i,h]-AvgTh)*(FTh[i,h]-AvgTh));



                                 end;

                               sum4:=sqrt(sum4*sum3);

                              if sum4 < 0.001 then FCorXT[j,h]:=0.0
                              else FCorXT[j,h]:=sum2/sum4;

         end;


        for k:=0 to pred(FCountY) do
        begin
          AvgTh:=0.0;
          Avg:=0.0;
          sum2:=0.0;
          sum3:=0.0;
          sum4:=0.0;
                                 for i:=1 to FSize do

                                 begin
                                        Avg:=Avg+FYhinit[i,k];


                                  end;
                                  Avg:=Avg/FSize;

                                  for i:=1 to FSize do

                                 begin

                                       sum2:=sum2+((FYhinit[i,k]-Avg)*(FTh[i,h]-AvgTh));
                                       sum3:=sum3+((FYhinit[i,k]-Avg)*(FYhinit[i,k]-Avg));
                                       sum4:=sum4+((FTh[i,h]-AvgTh)*(FTh[i,h]-AvgTh));



                                 end;
                                 sum4:=sqrt(sum4*sum3);

                                 if sum4 < 0.001 then FCorYT[k,h]:=0.0
                                else FCorYT[k,h]:=sum2/sum4;


       end;



       // calcul des redondances

       sum1:=0.0  ;
       
       for j:=0 to pred(FCountX) do

       begin

              sum1:=sum1+FCorXT[j,h]*FCorXT[j,h];



        end;

        FRdXT[h]:=sum1/FCountX;

       sum1:=0.0 ;

       for k:=0 to pred(FCountY) do

       begin

              sum1:=sum1+FCorYT[k,h]*FCorYT[k,h];



        end;
        
        FRdYT[h]:=sum1/FCountY;
        




        // Calcul du VIP ( Variable Importance for the Projection)
         for j:=0 to pred(FCountX) do

                begin

                         sum1:=0.0;
                         sum2:=0.0;


                         for k:=0 to h do sum1:=sum1+ FRdYT[k]*(FWh[j,k]*FWh[j,k]);

                         k:=h;
                         while k>=0 do
                         begin
                          sum2:=FRdYT[k]+sum2;
                          k:=k-1;
                          end;
                         FVIP[j,h]:=sqrt((FCountX/sum2)*sum1);


                end;




        {    Calcul suppos du coefficient Q2 qui ne fonctionne pour l'instant pas


       calcul du RES = norme de Y kh = ||Ykh||
        calcul du PRESS =  Ch(-i).X(h-1)i.Wh(-i) (formule utilis dans Simca P)

        for k:=0 to pred(FCountY) do


                begin
                         
                        sum1:=0.0;
                        sum2:=0.0;
                        sum3:=0.0;
                        sum4:=0.0;

                        for i:=1 to FSize do

                         begin

                                
                              Sum1:= Sum1+FYhpred[i,k]*FYhpred[i,k];

                              
                             //for l:=1 to FSize do  for j:=0 to pred(FCountX) do  Sum3:= Sum3 + FWhi[i,j]*FXhpred[l,j];

                               for j:=0 to pred(FCountX) do  Sum3  := Sum3 + FWhi[i,j]*FXhpred[i,j];
                               
                                       // Sum3:= Sum3 + FWhi[i,j];
                                       //Sum3:= Sum3 + FWhi[i,j]*FXhpred[i,j];

                           

                              Sum3:= FChi[i,k]*Sum3;

                               Sum4:=Sum4+((FYhpred[i,k]-Sum3)*(FYhpred[i,k]-Sum3));

                         end;




                                FRES[k,h]:=sum1;
                                FPRESS[k,h]:=sum4;

                         FQ2[k,h]:=  1 - (FPRESS[k,h]/FRES[k,h]);

                     

               end;


        if h=pred(FNbAxis) then

        begin
        
            

             for t:=0 to pred(FNbAxis)do

             begin


               for k:=0 to pred(FCountY) do

               begin

                        sum1:=sum1+FRES[k,t];
                        sum2:=sum2+FPRESS[k,t];


               end;

               //sum3:= sum3*(sum2/sum1);

                  FQ2[0,t]:=  1 - (sum2/sum1);
                  //FQ2[0,t]:=  1 - sum3;


            end;

        end;   }


   end;

procedure TCalcPLS.runAnalysis(examples: TExamples);

        var h,i,j,k,iteration: integer;
            curWh,prodThi,sumWhi,prodUhi,sumChi: TVectorPLS;
            FXhpred,FYhpred,FWhi,FChi,FXhinit,FYhinit: TMatrixPLS;
            prodTh,sumWh,prodUh,sumCh,sumPh,sumUh,sum: TDataPLS;
            okConvergence: boolean;

            somme: double;
            WhEtoile: TMatrixPLS;

            //new -- R.R. -- 08/03/2006 -- contrle de dure
            tps: cardinal;

            constante: TDataPLS;

        begin

                tps:= getTickCount();
                //rcupration des paramtres, calcul des stats ( moyennes, ecarts types)
                FStatData:= TStatPLS.create(FDataSourceInput,FDataSourceTarget,examples);
                FStatData.runComputation(FDataSourceInput,examples,FCountX,FStatData.FAvgX,FStatData.FStdX);
                FStatData.runComputation(FDataSourceTarget,examples,FCountY,FStatData.FAvgY,FStatData.FStdY);
                TraceLog.WriteToLogFile(format('[PLS] calcul des stats = %d ms.',[getTickCount()-tps]));


                //Allocation des divers vecteurs et matrices utiliss
                setLength(FYhpred,succ(FSize),FCountY);
                setLength(FXhpred,succ(FSize),FCountX);
                setLength(FWhi,succ(FSize),FCountX);
                setLength(FChi,succ(FSize),FCountY);
                setLength(sumWhi,succ(FSize));
                setLength(prodUhi,succ(FSize));
                setLength(prodThi,succ(FSize));
                setLength(sumChi,succ(FSize));

                tps:= getTickCount();
                //remplir les premires versions de Xh et Yh
                self.setFirstMatrix(examples,FDataSourceInput,FCountX,FVarianceTotaleX,FUsedStdDevX,FStatData.FAvgX,FStatData.FStdX,FXh);
                self.setFirstMatrix(examples,FDataSourceTarget,FCountY,FVarianceTotaleY,FUsedStdDevY,FStatData.FAvgY,FStatData.FStdY,FYh);

                //sauvegarde des matrices initiales pour pouvoir les utiliser  la fin de l'algorithme
                //inutile si pas besoin des calculs dtaills
                if FDetailedResult
                 then
                  begin
                    setLength(FYhinit,succ(FSize),FCountY);
                    setLength(FXhinit,succ(FSize),FCountX);
                    for i:= 1 to FSize do for j:= 0 to pred(FCountX) do FXhinit[i,j]:=FXh[i,j];
                    for i:= 1 to FSize do for k:= 0 to pred(FCountY) do FYhinit[i,k]:=FYh[i,k];
                  end;
                TraceLog.WriteToLogFile(format('[PLS] remplissage (initialisation) des matrices = %d ms.',[getTickCount()-tps]));

                //vecteur Wh courant
                setLength(curWh,FCountX);

                tps:= getTickCount();
                //*************************
                //***** production des axes
                //*************************

                //***** pour chaque axe  produire
                for h:= 0 to pred(FNbAxis) do

                        begin


                                //copier la premire colonne de Yh-1 dans Uh
                                 for i:= 1 to FSize do FUh[i,h]:= FYh[i,0];


                                //initialiser le vecteur Wh courant
                                for j:= 0 to pred(FCountX) do curWh[j]:= 0.0;

                                //*** on rpete jusqu' convergence ***
                                iteration:= 1;

                                 REPEAT

                                        //calculer Uh'Uh
                                        prodUh:= 0.0;

                                        for i:= 1 to FSize do

                                        begin
                                                prodUh:= prodUh+FUh[i,h]*FUh[i,h];
                                        end;

                                        //calculer Wh
                                         for j:= 0 to pred(FCountX) do

                                                 begin
                                                         sumWh:= 0.0;
                                                        for i:= 1 to FSize do sumWh:= sumWh+FXh[i,j]*FUh[i,h];
                                                         //normaliser par Uh'Uh

                                                        if (prodUh>0)
                                                        then sumWh:= sumWh/prodUh;

                                                        //acffecter  Wh
                                                         curWh[j]:= sumWh;

                                                 end;
                                        
                                        //normer Wh  1
                                        sumWh:= 0.0;

                                        for j:= 0 to pred(FCountX) do sumWh:= sumWh+curWh[j]*curWh[j];

                                        if (sumWh>0)
                                        then for j:= 0 to pred(FCountX) do curWh[j]:= curWh[j]/SQRT(sumWh);



                                        //calculer la projection Th
                                        for i:= 1 to FSize do

                                                begin
                                                         prodTh:= 0.0;
                                                         for j:= 0 to pred(FCountX) do prodTh:= prodTh+FXh[i,j]*curWh[j];


                                                        //affecter
                                                         FTh[i,h]:= prodTh;
                                                 end;

                                        //calculer Th'Th
                                        prodTh:= 0.0;

                                        for i:= 1 to FSize do prodTh:= prodTh+FTh[i,h]*FTh[i,h];

                                        //calculer Ch
                                         for k:= 0 to pred(FCountY) do

                                                 begin
                                                        sumCh:= 0.0;
                                                        for i:= 1 to FSize do sumCh:= sumCh+FYh[i,k]*FTh[i,h];
                                                         //normaliser par Th'Th

                                                        if (prodTh>0)
                                                        then sumCh:= sumCh/prodTh;

                                                        //affecter  Ph
                                                         FCh[k,h]:= sumCh;

                                                 end;

                                          //calculer Uh
                                        for i:= 1 to FSize do

                                                begin
                                                         sumUh:= 0.0;
                                                         for k:= 0 to pred(FCountY) do sumUh:= sumUh+FYh[i,k]*FCh[k,h];

                                                        
                                                         FUh[i,h]:= sumUh;
                                                 end;



                                        //tester la convergence
                                        okConvergence:= isConvergent(curWh,h);


                                        //rcuprer le Wh
                                        for j:= 0 to pred(FCountX) do FWh[j,h]:= curWh[j];

                                        inc(iteration);

                                UNTIL (iteration>PLS_MAX_ITERATION) or okConvergence;

                                //si ok pour la convergence

                                { Calcul des diverses valeurs permettant la determination du Q2
                                  for i := 1 to FSize do
                                        begin
                                               prodUhi[i]:=0;
                                                for compteur := 1 to FSize do
                                                begin
                                                     if (i<>compteur) then prodUhi[i]:= prodUhi[i]+FUh[compteur,h]*FUh[compteur,h];
                                                end;
                                        end;
                                        for j:= 0 to pred(FCountX) do
                                                 begin
                                                        for i:= 1 to FSize do
                                                        begin
                                                                sumWhi[i]:=0.0;

                                                                for compteur := 1 to FSize do

                                                                begin
                                                                        if (i<>compteur) then sumWhi[i] := sumWhi[i]+FXh[compteur,j]*FUh[compteur,h];
                                                                end;

                                                                 if (prodUhi[i]>0) then sumWhi[i]:= sumWhi[i]/prodUhi[i];
                                                        FWhi[i,j]:= sumWhi[i];

                                                        end;
                                                         end;


                                        for i:= 1 to FSize do sumWhi[i]:= 0.0;
                                       
                                        for j:= 0 to pred(FCountX) do
                                        begin
                                                for i:= 1 to FSize do sumWhi[i]:= sumWhi[i]+FWhi[i,j]*FWhi[i,j];
                                        end;
                                        for i:= 1 to FSize do if (sumWhi[i]>0) then for j:= 0 to pred(FCountX) do FWhi[i,j]:= FWhi[i,j]/SQRT(sumWhi[i]);





                                for i := 1 to FSize do
                                        begin
                                                prodThi[i]:=0;

                                                for compteur := 1 to FSize do

                                                begin
                                                     if (i<>compteur) then prodThi[i]:= prodThi[i]+FTh[compteur,h]*FTh[compteur,h];

                                                end;


                                        end;

                                        for k:= 0 to pred(FCountY) do

                                                 begin

                                                        for i:= 1 to FSize do

                                                        begin

                                                                sumChi[i]:=0.0;

                                                                for compteur := 1 to FSize do

                                                                begin
                                                                        if (i<>compteur) then sumChi[i] := sumChi[i]+FYh[compteur,k]*FTh[compteur,h];

                                                                end;

                                                                 if (prodThi[i]>0) then sumChi[i]:= sumChi[i]/prodThi[i];


                                                        FChi[i,k]:= sumChi[i];

                                                        end;

                                                 end;      }
                                              


                                //la convergence a-t-elle t force ?
                                if (iteration>PLS_MAX_ITERATION)
                                then TraceLog.WriteToLogFile(Format('[PLS] *** warning *** force convergence for axis n%d',[succ(h)]));

                                 //calculer Th'Th
                                prodTh:= 0.0;
                                for i:= 1 to FSize do prodTh:= prodTh+FTh[i,h]*FTh[i,h];

                                //Calcul de Ph
                                         for j:= 0 to pred(FCountX) do

                                                 begin
                                                         sumPh:= 0.0;
                                                        for i:= 1 to FSize do sumPh:= sumPh+FXh[i,j]*FTh[i,h];
                                                         //normaliser par Th'Th

                                                        if (prodTh>0) then sumPh:= sumPh/prodTh;

                                                        //affecter  Ph
                                                         FPh[j,h]:= sumPh;

                                                 end;

                         // sauvegarde de l'etat des matrices des rsidus
                                for i:= 1 to FSize do

                                 begin
                                        for j:= 0 to pred(FCountX) do FXhpred[i,j]:=FXh[i,j];
                                        for k:= 0 to pred(FCountY) do FYhpred[i,k]:=FYh[i,k] ;

                                 end;



                                //nouvelle version de Xh -- tape 2.4
                                 for i:= 1 to FSize do

                                        begin
                                                for j:= 0 to pred(FCountX) do FXh[i,j]:= FXh[i,j]-FTh[i,h]*FPh[j,h];

                                         end;


                                 //nouvelle version de Yh -- tape 2.5
                                 for i:= 1 to FSize do

                                        begin
                                                for k:= 0 to pred(FCountY) do FYh[i,k]:= FYh[i,k]-FTh[i,h]*FCh[k,h];

                                         end;




                                for j:= 0 to pred(FCountX) do
                                begin

                                         sum:=0.0;


                                                for k:= 0 to h-1 do
                                                begin

                                                sum:=sum+FWhseconde[j,k]*FPh[j,k]*FWh[j,h];

                                                end;
                                             //if h=0 then  FWhseconde[j,h]:=FWh[j,h];
                                             //if h>0 then  FWhseconde[j,h]:=FWh[j,h]-FWhseconde[j,h-1]*FPh[j,h-1]*FWh[j,h];

                                            FWhseconde[j,h]:=FWh[j,h]-sum;
                                           //FWhseconde[j,h]:=FWh[j,h];
                                           //for k:=0 to h-1 do  FWhseconde[j,h]:=FWhseconde[j,h]-FWhseconde[j,k]*FPh[j,k]*FWh[j,h];


                                end;


                                         if (h=pred(FNbAxis)) then

                                         begin

                                                 for j:= 0 to pred(FCountX) do

                                                 begin

                                                        for k:=0 to pred(FCountY) do FCoefReg[j,k]:=FWhseconde[j,h]*FCh[k,h];

                                                 end;

                                        end;


                            //new -- 10/03/2006 -- optimisation -- pas besoin de calculer les indicateurs si classement
                            //puisque a ne sera jamais affich
                            if FDetailedResult
                             then self.globalGoodness(h,FXhpred,FYhpred,FWhi,FChi,FXhinit,FYhinit);

                        end;

                 //********************************
                 //***** fin de production des axes
                 //********************************
                 TraceLog.WriteToLogFile(format('[PLS] production des (%d) axes = %d ms.',[FNbAxis,getTickCount()-tps]));


                 //************************************************************
                 //R.R. -- new -- 23/10/2005
                 //calcul des coefficients de la rgression -- Tenenhaus, p.137
                 //B est de dimension [FCountX x FCountY]
                 //Nota : j'ai bien l'impression que le code ci-dessus y est ddi
                 //mais une erreur doit s'y glisser, notamment pour le calcul de Wh*
                 //************************************************************

                 //--> attention aux dimensions de FCh et FWh
                 //setLength(FCh,FCountY,FNbAxis);
                 //setLength(FWh,FCountX,FNbAxis);
                 //setLength(FPh,FCountX,FNbAxis);

                 //calculer Wh* avec la formule par rcurrence -- p.114
                 tps:= getTickCount();
                 setLength(WhEtoile,FCountX,FNbAxis);
                 for j:= 0 to pred(FCountX) do WhEtoile[j,0]:= FWh[j,0];
                 for k:= 1 to pred(FNbAxis) do
                  begin
                   somme:= 0.0;
                   for j:= 0 to pred(FCountX) do
                    somme:= somme + FPh[j,k-1]*FWh[j,k];
                   for j:= 0 to pred(FCountX) do
                    WhEtoile[j,k]:= FWh[j,k] - WhEtoile[j,pred(k)] * somme;
                  end;

                 for j:= 0 to pred(FCountX) do
                  for i:= 0 to pred(FCountY) do
                   begin
                    sum:= 0.0;
                    for k:= 0 to pred(FNbAxis) do
                     begin
                      //appliquer
                      sum:= sum + WhEtoile[j,k] * FCh[i,k];
                     end;
                    FBCoefReg[j,i]:= sum;
                   end;

                 //new -- R.R. -- 10/03/2006 -- corriger pour avoir les coefficients sur les variables originelles
                 //cela permettra aussi de rduire le temps d'excution sur la projection
                 //pour chaque Yi
                 for i:= 0 to pred(FCountY) do
                  begin
                   constante:= 0.0;
                   for j:= 0 to pred(FCountX) do
                    begin
                     if (FUsedStdDevX[j] > 0.0)
                      then
                       begin
                        FBCoefReg[j,i]:= FUsedStdDevY[i] * FBCoefReg[j,i] / FUsedStdDevX[j];
                        constante := constante + (-1.0) * FBCoefReg[j,i] * FStatData.FAvgX[j];//FBCoefReg a t corrig  la ligne prcdente !
                       end;
                    end;
                   constante:= constante + FStatData.FAvgY[i];
                   FBCoefReg[FCountX,i]:= constante;
                  end;

                 Finalize(WhEtoile);
                 TraceLog.WriteToLogFile(format('[PLS] calcul des coefs. de projection = %d ms.',[FNbAxis,getTickCount()-tps]));
                 //************************************************************


                //vider la mmoire
                setLength(curWh,0);
                setLength(FXh,0,0);
                setLength(FXhpred,0,0);
                setLength(FYh,0,0);
                setLength(FYhpred,0,0);
                setLength(FWhi,0,0);
                setLength(FChi,0,0);
                setLength(sumWhi,0);
                setLength(prodUhi,0);
                setLength(prodThi,0);
                setLength(sumChi,0);

        end;


procedure TCalcPLS.setFirstMatrix(examples: TExamples; FDataSource:TLstAttributes; Count:Integer; var FVarianceTotale:TDataPLS; var FUsedStdDev:TVectorPLS;FAvg:TVectorPLS;FStd:TVectorPLS; var Fh:TMatrixPLS);

        var i,j: integer;
        example: integer;

        begin

                //On initialise le vecteur des cart-types
                if not(FStandardize) then

                        begin
                                //On bloque le vecteur des ecart types  1 pour ne pas reduire les donnes ( non standardises)
                                for j:= 0 to pred(Count) do FUsedStdDev[j]:= 1.0;
                                //trace de la matrice de variance co-variance (et non plus de corrlation puisque analyse non-norme)
                                FVarianceTotale:= 0.0;
                                for j:= 0 to pred(Count) do FVarianceTotale:= FVarianceTotale+SQR(FUsedStdDev[j]);
                        end
                 else
                        begin
                                //On cre le vecteur des carts types
                                for j:= 0 to pred(Count) do FUsedStdDev[j]:= FStd[j];
                                //trace de la matrice de corrlation...
                                 FVarianceTotale:= 1.0*Count;
                        end;

                //*********************************************************************
                //new -- 10/03/2006 -- selon la tte du fichier, vraiment pour chipoter
                //*********************************************************************
                
                if (FSize > Count)
                 then
                  begin
                    for i:= 1 to FSize do
                     begin
                      //dereferencememt
                      example:= examples.Number[i];
                      //pondre, centre et ventuellement rduite...
                      for j:= 0 to pred(Count) do
                       begin
                        if (FUsedStdDev[j] > 0.0) then
                         Fh[i,j]:= (FDataSource.Attribute[j].cValue[example] - FAvg[j]) / FUsedStdDev[j]
                        else
                         Fh[i,j]:= 0.0;
                       end;
                    end;
                  end
                 else
                  begin
                    //new -- 10/03/2006 -- inversons la chose pour minimiser les tests de comparaison
                    //si on a bcp d'observations et peu de variables, a dgraderait plutt les perfs. cette affaire !!!
                    for j:= 0 to pred(Count) do
                     begin
                      if (FUsedStdDev[j] > 0)
                       then
                        begin
                         for i:= 1 to FSize do
                          begin
                           example:= examples.Number[i];
                           Fh[i,j]:= (FDataSource.Attribute[j].cValue[example] - FAvg[j]) / FUsedStdDev[j];
                          end;
                        end
                       else
                        begin
                         for i:= 1 to FSize do Fh[i,j]:= 0.0;
                        end;
                     end;
                  end;
                  
        end;


(*
 //R.R. -- new -- 23/10/2005 -- coefficients de la rgression sur variables centres et rduites
 s:= s + '<H3>Regression coefficients on standardized variables</H3>';
 s:= s + HTML_HEADER_TABLE_RESULT;
 //en-tte les variables Y
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY+'<TH>X/Y</TH>';
 for i:= 0 to pred(FCountY) do
  s:= s + format('<TH>%s</TH>',[FDataSourceTarget.Attribute[i].Name]);
 s:= s + '</TR>';
 //les coeffs. pour chaque X
 for j:= 0 to pred(FCountX) do
  begin
   s:= s + HTML_TABLE_COLOR_DATA_GRAY + format('<TH>%s</TH>',[FDataSourceInput.Attribute[j].Name]);
   for i:= 0 to pred(FCountY) do
    s:= s + format('<TD align="right">%.4f</TD>',[FBCoefReg[j,i]]);
   s:= s + '</TR>';
  end;
 s:= s + '</table>';
 //************************************************************************************************
*)

procedure TCalcPLS.projection(var X, Y: TVectorPLS);
var i,j: integer;
    value: TDataPLS;
begin
 //centrer et rduire les valeurs de X
 //new -- R.R. -- 10/03/2006 -- inutile maintenant puisqu'on a les coefs. sur les variables initiales
 (*
 for j:= 0 to pred(FCountX) do
  begin
   X[j]:= X[j] - FStatData.FAvgX[j];
   if (FUsedStdDevX[j] > 0.0)
    then X[j]:= X[j]/FUsedStdDevX[j]
    else X[j]:= 0.0;//!\ obligatoire car si c'est une constante dans le fichier apprentissage, elle peut ne pas l'tre dans la partie test
  end;
 *)

 //appliquer les coefs. pour calculer chaque sortie Y
 for i:= 0 to pred(FCountY) do
  begin
   value:= 0.0;
   for j:= 0 to pred(FCountX) do
    value:= value + FBCoefReg[j,i]*X[j];
   //valeur brute (sur coefs. centrs et rduits)
   Y[i]:= value;
   //corriger la sortie Y[i] en d-centrant et en d-rduisant sur les Y
   //>>new -- 10/03/2006 -- pas de correction non plus en sortie maintenant... mais ne pas oublier la constante
   //Y[i]:= Y[i]*FUsedStdDevY[i] + FStatData.FAvgY[i];
   Y[i]:= Y[i] + FBCoefReg[FCountX,i];
  end;
end;

procedure TCalcPLS.setProjection(atts: TLstAttributes);
var example: integer;
    j,i: integer;
    X,Y: TVectorPLS;
begin
 //crer les tableaux intermdiaires
 setLength(X,FCountX);
 setLength(Y,FCountY);
 //pour chaque exemple
 for example:= 1 to self.FDataSourceInput.Size do
  begin
   for j:= 0 to pred(FCountX) do X[j]:= self.FDataSourceInput.Attribute[j].cValue[example];
   self.projection(X,Y);
   for i:= 0 to pred(FCountY) do atts.Attribute[i].cValue[example]:= Y[i];
  end;
 //au final...
 finalize(X);
 finalize(Y);
end;

procedure TCalcPLS.projectionOnLstX(example: integer; lstX: TLstAttributes;
  var Y: TVectorPLS);
var i,j: integer;
    value: TDataPLS;
begin
 //appliquer les coefs. pour calculer chaque sortie Y
 for i:= 0 to pred(FCountY) do
  begin
   value:= 0.0;
   for j:= 0 to pred(FCountX) do
    value:= value + FBCoefReg[j,i]*lstX.Attribute[j].cValue[example];
   //valeur brute (sur coefs. centrs et rduits)
   Y[i]:= value;
   //corriger la sortie Y[i] en d-centrant et en d-rduisant sur les Y
   //>>new -- 10/03/2006 -- pas de correction non plus en sortie maintenant... mais ne pas oublier la constante
   //Y[i]:= Y[i]*FUsedStdDevY[i] + FStatData.FAvgY[i];
   Y[i]:= Y[i] + FBCoefReg[FCountX,i];
  end;
end;

end.
