#!/bin/bash

progdir="$1"
datafile="$2"
lrout="$3"
lrcorrect="$4"
if [ -z "$lrcorrect" ]; then
    echo
    echo "Usage: $0 <progdir> <datafile> <lrout_fname> <correct_lrout_fname>,"
    echo
    echo "where"
    echo "  progdir is the location of the train and predict executables,"
    echo "  datafile is the path to the a-or-d datafile,"
    echo "  lrout_fname is the filename to store the computed LR parameters,"
    echo "  correct_lrout_fname contains the expected LR parameters."
    echo
    exit 1
fi

train="$progdir"/train

if [ ! -x "$train" ]; then
    echo
    echo "Error: Program '$train' does not exist, or is not executable"
    echo "(are the permissions set correctly?)"
    echo
    exit 1
fi

if [ ! -r "$datafile" -a ! -r $(echo $datafile | cut -d: -f1) ]; then
    echo
    echo "Error: Data file '$datafile' does not exist, or is not readable"
    echo
    exit 1
fi


run_train_test() {
    if [ -n "$1" ]; then
        echo "$train" in "$datafile" save "$lrout"
    else
        "$train" in "$datafile" csv save "$lrout"
    fi
}

# Run test.
if ! run_train_test; then
    echo "Train error: unknown."
    echo "Command was:"
    echo -n "    "
    run_train_test display
    exit 1
fi


skipfirst() {
    read line
    cat -
}
sumsquares() {
    local numlines="$1"

    # NOTE: dc uses _ for negation, and - for subtraction.  Our input
    # probably uses - for negation, so we have to transpose a little bit.

    ( echo "20k";
      echo "0.0";
      yes 'M 2 ^ +' 2> /dev/null \
          | head -n "$numlines" \
          | paste "$lrcorrect" "$lrout" - \
          | skipfirst \
          | tr -- '-M' '_-';
      echo "p";
    ) | dc
}
floatlt() {
    # The usual test program does not handle floating point values.
    # This function is a less-than test for floating point values,
    # Returning 0 if left is less-than or equal to right, or 1 otherwise.
    local left="$1"
    local right="$2"
    local result=$(dc -e "20k $left $right - p")
    if echo "$result" | fgrep '-' > /dev/null; then
        return 0;
    else
        return 1;
    fi
}

# Sum of squares of coefficient differences.
numlines=$(wc -l "$lrcorrect" | awk '{ print $1 }')
error=$(sumsquares $numlines)
tol="0.001"
if ! floatlt $error $tol; then
    echo
    echo "Train error: lr parameters appear incorrect for command"
    echo "Sum of squared differences of parameters is $error."
    echo "Allowed error for this weak test is $tol."
    echo
    echo -n "    "
    run_train_test display
    echo "It could be a small difference in compilers, kernels, libraries,"
    echo "cflags, etc.  Please check the result of test_predict.sh before"
    echo "concluding that something is very wrong.  You can also check"
    echo "the difference by hand between the correct output '$lrcorrect' and"
    echo "the computed output '$lrout'."
    echo
    echo "You might also look for errors printed just before this message"
    echo "(just before 'Train error: ...')"
    echo
    echo "Note that the datafile should be a-or-d.csv, or a file with the"
    echo "same contents."
    echo
    exit 1
fi
