(******************************************************************)
(* UCalcSpvLibsvmCsvc.pas - Copyright (c) 2006 Ricco RAKOTOMALALA *)
(******************************************************************)

{
@abstract(Utilisation de la mthode C-SVC de la bibliothque LIBSVM)
@author(Ricco)
@created(08/01/2006)

2 objectifs :
(a) une mthode de SVM supplmentaire dans TANAGRA
(b) montrer qu'il est possible d'importer des mthodes  partir de bibliothques
externes crites dans un autre langage (le C++ en l'occurence).

}

unit UCalcSpvLibsvmCsvc;

interface

USES
   UDatasetExamples,
   UCompSpvLDefinition,
   UDatasetDefinition,
   UCalcDistribution,
   UCalcSpvStructScore,
   //et trs important, l'unit d'import de la LIBSVM
   ULibsvmImport,
   //suite...
   UCalcSpvSMOAttTransformation;

TYPE
  TCalcLibsvmCsvc = class(TCalcSpvLearning)
  private
  //paramtrage
  FParam: p_svm_parameter;
  //problme
  FProblem: p_svm_problem;
  //modle
  FModel: p_svm_model;
  //mthode de transformation des variables -- ici, ce sera toujours [0,1] au dpart
  FAttTransSVM: TAttTrans;
  //tableau interne pour prparer les prdictions futures, viter ainsi les allocations successives
  FTmpNodePrediction: array_svm_node;
  //tableau interne pour le calcul des probabilits
  FTmpProbaPrediction: array_double;
  //mettre les paramtres par dfaut -- extrait "svm_train.c"
  procedure setDefaultParameters();
  //rcuprer les paramtres en provenance de Tanagra
  procedure getUserParameters();
  //lancer l'apprentissage -- private,  prparer au cas o il faut insrer qq chose au milieu
  function    runClassifier(examples: TExamples): boolean;
  protected
  procedure   destroyStructures(); override;
  public
  //surcharge normale dans la hirachie
  function    coreLearning(examples: TExamples): boolean; override;
  //prospective totale -- utiliser une fonction logistique sur la sortie
  procedure   getScore(example: integer; var postProba: TTabScore); override;
  //comparer au seuil 0 (si output <= 0 alors c = 1 sinon c = 2
  procedure   classification(example: integer; var response: TTypeDiscrete); override;
  //description des rsultats
  function    getHTMLResults(): string; override;  
  end;

implementation

uses
    Sysutils, ULogFile, UConstConfiguration, UCompSpvLibsvmCsvc;

{ TCalcLibsvmCsvc }

procedure TCalcLibsvmCsvc.classification(example: integer;
  var response: TTypeDiscrete);
var j: integer;
    prediction: double;
begin
 //remplir le tableau temporaire
 for j:= 0 to pred(self.Descriptors.Count) do
  FTmpNodePrediction[j].value:= 2.0*FAttTransSVM.cValue[j,example]-1.0;//pour faire varier de -1  +1
 //rcuprer la prdiction
 prediction:= svm_predict(FModel,FTmpNodePrediction);
 //envoyer la prdiction
 if (prediction>0) and (prediction<succ(self.ClassAttribute.nbValues))
  then response:= trunc(prediction)
  //sinon, rponse au hasard
  else response:= succ(random(self.ClassAttribute.nbValues));
end;

function TCalcLibsvmCsvc.coreLearning(examples: TExamples): boolean;
begin
 result:= self.runClassifier(examples);
 //if ok then TraceLog.WriteToLogFile('[C-SVC] ok, voir les infos  afficher');
end;

procedure TCalcLibsvmCsvc.destroyStructures;
begin
 //le compteur de rfrence devrait s'occuper tout seul des "Array of..." -- j'imagine.
 if assigned(FParam) then svm_destroy_param(FParam);
 if assigned(FModel) then svm_destroy_model(FModel);
 //ici c'est  moi de le faire...
 if assigned(FProblem) then dispose(FProblem); 
 //
 if assigned(FAttTransSVM) then FreeAndNil(FAttTransSVM);
 //les tableaux temporaires
 Finalize(FTmpNodePrediction);
 Finalize(FTmpProbaPrediction);
end;

function TCalcLibsvmCsvc.getHTMLResults: string;
var s: string;
    k: integer;
begin
 s:= '<H3>SVM characteristics</H3>';
 s:= s + HTML_HEADER_TABLE_RESULT;
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY + '<TH>Characteristic</TH><TH>Value</TH></TR>';
 s:= s + HTML_TABLE_COLOR_DATA_GRAY + format('<TD>%s</TD><TD align=right>%d</TD></TR>',['# classes',FModel^.nr_class]);
 s:= s + HTML_TABLE_COLOR_DATA_GRAY + format('<TD>%s</TD><TD align=right>%d</TD></TR>',['# support vectors',FModel^.l]);
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY + '<TD colspan=2># support vectors for each class</TD></TR>';
 for k:= 1 to self.ClassAttribute.nbValues do
  s:= s + HTML_TABLE_COLOR_DATA_GRAY + format('<TD># sv. for %s</TD><TD align=right>%d</TD></TR>',[self.ClassAttribute.LstValues.getDescription(k),FModel^.nSV[pred(k)]]);
 s:= s + '</table>';
 //and then...
 result:= s;
end;

procedure TCalcLibsvmCsvc.getScore(example: integer;
  var postProba: TTabScore);
var j: integer;

begin
 //remplir le tableau temporaire
 for j:= 0 to pred(self.Descriptors.Count) do
  FTmpNodePrediction[j].value:= FAttTransSVM.cValue[j,example];
 //faire calculer les probas
 svm_predict_probability(FModel,FTmpNodePrediction,FTmpProbaPrediction);
 //recopier pour la sortie
 postProba[0]:= 0.0;
 for j:= 1 to self.ClassAttribute.nbValues do
  begin
   postProba[j]:= FTmpProbaPrediction[pred(j)];
   postProba[0]:= postProba[0] + postProba[j];
  end;
end;

procedure TCalcLibsvmCsvc.getUserParameters;
var prm: TOpPrmCSVC;
begin
  prm:= self.OpPrmSpv as TOpPrmCSVC;
  FParam^.kernel_type := prm._kernel_type; //>>type de noyau
  FParam^.degree := prm.degree; //>>degr du kernel
  FParam^.gamma := prm.gamma;	// 1/k >>gamma
  FParam^.coef0 := prm.coef0; //>>coef. 0 dans le kernel
  FParam^.C := prm.C;//>>penalit
  FParam^.eps := prm.eps;//>>tolerance
  FParam^.shrinking := prm.shrinking;//>>use shrinking heuristics
  FParam^.probability := prm.probability;//>>use probability estimate
end;

function TCalcLibsvmCsvc.runClassifier(examples: TExamples): boolean;
var ok: boolean;
    i,j: integer;
    chaine: PChar;
begin
 //vider les structures existantes
 self.destroyStructures();
 //mettre les paramtres par dfaut
 self.setDefaultParameters();
 //rcuprer les paramtres en provenance de l'interface
 self.getUserParameters();
 //paramtres de normalisation -- [0,1]
 FAttTransSVM:= TAttTransNormalize.create(self.Descriptors,examples);
 //prparer les donnes en respectant le formalisme LIBSVM -- cf. le code SVMTrain (le code Java est autrement plus clair)
 ok:= true;
 TRY
 //prparer l'espace mmoire
 new(FProblem);
 setLength(FProblem^.y,examples.Size);
 setLength(FProblem^.x,examples.Size);
 //copier les donnes
 for i:= 1 to examples.Size do
  begin
   FProblem^.y[pred(i)]:= self.ClassAttribute.cValue[examples.Number[i]];
   //!\ ouah, le +1 est fondamental
   setLength(FProblem^.x[pred(i)],self.Descriptors.Count+1);//+1 pour rserver le dernier au bornage des colonnes !!!
   for j:= 0 to pred(self.Descriptors.Count) do
    begin
    FProblem^.x[pred(i)][j].index:= succ(j);
    //donnes transformes
    FProblem^.x[pred(i)][j].value:= 2.0*FAttTransSVM.cValue[j,examples.Number[i]]-1.0;//pour faire varier de -1  +1
    end;
    //!\\ warning --> parce que sinon, y sait pas  quelle colonne s'arrter, il n'y a pas d'indicateur de nombre de colonnes
    FProblem^.x[pred(i)][self.Descriptors.Count].index:= -1;
  end;
 //qqs paramtres  rgler
 FProblem^.l:= examples.Size;
 if (FParam^.gamma = 0) then FParam^.gamma:= 1.0/(1.0*self.Descriptors.Count);
 EXCEPT
 TraceLog.WriteToLogFile('[C-SVC -- LIBSVM] //!\\ exception during data preparation');
 ok:= false;
 END;
 //tester les paramtres
 chaine:= '_test_';
 if ok
  then
   begin
   TRY
   chaine:= svm_check_parameter(FProblem,FParam);
   EXCEPT
   TraceLog.WriteToLogFile('[C-SVC -- LIBSVM] //!\\ exception during check parameters');
   ok:= false;
   END;
   end;
 if (chaine<>nil) and (strlen(chaine)>0)
  then
   begin
   TraceLog.WriteToLogFile(format('[C-SVC -- LIBSVM] //!\\ not passed check parameters == %s',[chaine]));
   ok:= false;
   end;
 //lancer les oprations en faisant appel  la procdure de la bibliothque
 if ok
  then
   begin
     TRY
     TraceLog.WriteToLogFile('[C-SVC -- LIBSVM] ok check parameters, run training');
     FModel:= svm_train(FProblem,FParam);
     EXCEPT
     on e: exception do
      begin
      TraceLog.WriteToLogFile(format('[C-SVC -- LIBSVM] //!\\ exception during model training == %s',[e.Message]));
      ok:= false;
      end;
     END;
   end;
 //prparer le tableau de noeud intermdiaire pour les prdictions
 if ok
  then
   begin
    //tableau pour les prdictions
    setlength(FTmpNodePrediction,self.Descriptors.Count+1);
    for j:= 0 to pred(self.Descriptors.Count) do
     FTmpNodePrediction[j].index:= succ(j);
    FTmpNodePrediction[self.Descriptors.Count].Index:= -1;//don't forget my dear !!!
    //tableau pour les probas
    setLength(FTmpProbaPrediction,self.ClassAttribute.nbValues);
   end;
 result:= ok;
end;

procedure TCalcLibsvmCsvc.setDefaultParameters;
begin
  new(FParam);
  FParam^.svm_type := ord(C_SVC);//-- non --
  FParam^.kernel_type := ord(LINEAR); //>>type de noyau
  FParam^.degree := 1; //>>degr du kernel
  FParam^.gamma := 0;	// 1/k >>gamma
  FParam^.coef0 := 0; //>>coef. 0 dans le kernel
  FParam^.nu := 0.5; //-- non --
  FParam^.cache_size := 100;//-- non --
  FParam^.C := 1;//>>penalit
  FParam^.eps := 1e-3;//>>tolerance
  FParam^.p := 0.1;//--non--
  FParam^.shrinking := 1;//>>use shrinking heuristics
  FParam^.probability := 0;//>>use probability estimate
  FParam^.nr_weight := 0;//--non--
  FParam^.weight_label := nil;//--non--
  FParam^.weight := nil;//--non--
end;

end.
