{ **********************************************************************
  *                        Program EIGENVEC.PAS                        *
  *                             Version 1.0                            *
  *                     (c) J. Debord, January 1997                    *
  **********************************************************************
  This program computes the eigenvector associated to a given real
  eigenvalue of a general square matrix A by inverse iteration. If
  lambda is an approximation of the eigenvalue, and V an approximation
  of the associated eigenvector, then the vector W which satisfies
  (A - lambda * I) * W = V, where I is the identity matrix, is a better
  approximation of the eigenvector. The method is iterated until the
  difference between the elements of V and W becomes lower than a user-
  specified tolerance. The eigenvector is normalized by dividing by the
  element having the largest absolute value, q. The best estimation of
  the eigenvalue is then lambda + 1/q

  The matrix is stored in a data file with the following structure :

    Line 1             : dimension of the matrix (N)
    Lines 2 to (N + 1) : matrix

  The file MATRIX1.DAT is an example data file with N = 4. Use it with
  the approximate eigenvalues 1 and 5
  ********************************************************************** }

uses
  Crt, FMath, Matrices, Eigen;

const
  MAXITER = 100;     { Maximum number of iterations }
  TOL     = 1.0E-8;  { Tolerance on the eigenvector }

var
  A      : PMatrix;  { Matrix }
  N      : Integer;  { Dimension of matrix }
  Lambda : Float;    { Eigenvalue }
  V      : PVector;  { Eigenvector }
  Ch     : Char;     { Key pressed to exit program }

  procedure ReadMatrix(FileName : String; var A : PMatrix;
                       var N : Integer);
{ ----------------------------------------------------------------------
  Reads matrix from file. Note that A is passed as a VAR parameter
  because it is dimensioned inside the procedure.
  ---------------------------------------------------------------------- }
  var
    F    : Text;     { Data file }
    I, J : Integer;  { Loop variable }
  begin
    Assign(F, FileName);
    Reset(F);
    Read(F, N);
    DimMatrix(A, N, N);
    for I := 1 to N do
      for J := 1 to N do
        Read(F, A^[I]^[J]);
    Close(F);
  end;

  procedure ReadEigenValue(var Lambda : Float);
{ ----------------------------------------------------------------------
  Reads the initial approximation to the eigenvalue.
  ---------------------------------------------------------------------- }
  begin
    Write('Approximate eigenvalue : ');
    ReadLn(Lambda);
  end;

  procedure ReadEigenVector(var V : PVector; N : Integer);
{ ----------------------------------------------------------------------
  Reads the initial approximation to the eigenvector.
  ---------------------------------------------------------------------- }
  var
    S       : String;   { Entry string }
    I       : Integer;  { Loop variable }
    ErrCode : Integer;  { Error code }
  begin
    DimVector(V, N);
    WriteLn(#10, 'Approximate eigenvector (Default = [1, 1,... 1]) : ', #10);
    for I := 1 to N do
      begin
        Write('    V[', I, '] = ');
        ReadLn(S);
        Val(S, V^[I], ErrCode);
        if ErrCode <> 0 then
          V^[I] := 1.0;
      end;
  end;

  procedure WriteEigenValue(Lambda : Float);
{ ----------------------------------------------------------------------
  Writes eigenvalue on screen
  ---------------------------------------------------------------------- }
  begin
    WriteLn(#10, 'Refined eigenvalue : ', Lambda:12:6);
  end;

  procedure WriteEigenVector(V : PVector; N : Integer);
{ ----------------------------------------------------------------------
  Writes eigenvector on screen
  ---------------------------------------------------------------------- }
  var
    I : Integer;
  begin
    WriteLn(#10, 'Eigenvector :', #10);
    for I := 1 to N do
      WriteLn(V^[I]:12:6);
  end;

begin
  ClrScr;

  ReadMatrix('matrix1.dat', A, N);
  ReadEigenValue(Lambda);
  ReadEigenVector(V, N);

  { Compute eigenvector }
  case EigenVect(A, 1, N, MAXITER, TOL, V, Lambda) of
    MAT_OK       : begin
                     WriteEigenVector(V, N);
                     WriteEigenValue(Lambda);
                   end;
    MAT_SINGUL   : Write('Singular matrix!');
    MAT_NON_CONV : Write('Too many iterations!');
  end;

  GotoXY(1, 25);
  Write('Press a key ... ');
  Ch := ReadKey;
end.

