{ **********************************************************************
  *                        Program SYSEQ_QR.PAS                        *
  *                             Version 1.0                            *
  *                     (c) J. Debord, March 1999                      *
  **********************************************************************
  This program solves a system of linear equations (A * X = B) with
  several constant vectors by QR decomposition. The system is stored
  in a data file with the following structure :

    Line  1                   : dimension of the matrix (N)
    Lines 2 to (N + 1)        : matrix
    Line  P + 2               : number of constant vectors (P)
    Lines (N + 3) to (2N + 2) : constant vectors (one by column)

  The file MATRIX2.DAT is an example data file with N = 4 and P = 5

  The QR decomposition factors the matrix A (n x m, with n >= m) as a
  product QR where Q is a (n x m) column-orthogonal matrix and R a (m x m)
  upper triangular matrix. The system becomes QRX = B, then RX = Q'B, which
  is solved by back-substitution.

  If the system is impossible (n > m), the method returns a vector X such
  that A * X is the best approximation to B in the least-squares sense.
  ********************************************************************** }

uses
  Crt, FMath, Matrices;

var
  A : PMatrix;  { System matrix, which will be replaced by Q }
  R : PMatrix;  { Upper triangular matrix from the QR decomposition }
  B : PMatrix;  { Constant vectors }
  X : PMatrix;  { Solutions }
  N : Integer;  { Dimension of matrix }
  P : Integer;  { Number of constant vectors }
  I : Integer;  { Loop variable }
  Ch : Char;    { Key pressed to exit program }

  procedure ReadMatrices(FileName : String; var A, B : PMatrix;
                         var N, P : Integer);
{ ----------------------------------------------------------------------
  Reads data from file. Note that matrices are passed as VAR parameters
  because they are dimensioned inside the procedure.
  ---------------------------------------------------------------------- }
  var
    F : Text;        { Data file }
    I, J : Integer;  { Loop variable }
  begin
    Assign(F, FileName);
    Reset(F);

    { Read matrix }
    Read(F, N);
    DimMatrix(A, N, N);
    for I := 1 to N do
      for J := 1 to N do
        Read(F, A^[I]^[J]);

    { Read constant vectors }
    Read(F, P);
    DimMatrix(B, P, N);
    for J := 1 to N do
      for I := 1 to P do
        Read(F, B^[I]^[J]);  { B^[I] is the I-th constant vector }
    Close(F);
  end;

  procedure WriteMatrix(Title : String; A : PMatrix; N : Integer);
{ ----------------------------------------------------------------------
  Writes system matrix on screen
  ---------------------------------------------------------------------- }
  var
    I, J : Integer;
  begin
    WriteLn(Title, ' :', #10);
    for I := 1 to N do
      begin
        for J := 1 to N do
          Write(A^[I]^[J]:12:6);
        WriteLn;
      end;
    WriteLn;
  end;

  procedure WriteVectors(Title : String; B : PMatrix; N, P : Integer);
{ ----------------------------------------------------------------------
  Writes constant vectors or solution vectors
  so that they appear on columns.
  ---------------------------------------------------------------------- }
  var
    I, J : Integer;
  begin
    WriteLn(Title, ' :', #10);
    for J := 1 to N do
      begin
        for I := 1 to P do
          Write(B^[I]^[J]:12:6);
        WriteLn;
      end;
    WriteLn;
  end;

begin
  ClrScr;

  { Read and display data }
  ReadMatrices('matrix2.dat', A, B, N, P);
  WriteMatrix('System matrix', A, N);
  WriteVectors('Constant vectors', B, N, P);

  { Dimension arrays }
  DimMatrix(R, N, N);
  DimMatrix(X, P, N);

  { Perform QR decomposition of A
    (Note that Q is stored in place of A) }
  case QR_Decomp(A, 1, N, N, R) of
    MAT_OK : begin
               { Solve the system for each constant vector }
               for I := 1 to P do
                 QR_Solve(A, R, B^[I], 1, N, N, X^[I]);
               WriteVectors('Solution vectors', X, N, P);
             end;
    MAT_SINGUL : Write('Singular matrix!');
  end;

  GotoXY(1, 25);
  Write('Press a key ... ');
  Ch := ReadKey;
end.


