{ **********************************************************************
  *                        Program SYSEQSVD.PAS                        *
  *                             Version 1.1                            *
  *                     (c) J. Debord, October 1997                    *
  **********************************************************************
  This program solves a system of linear equations (A * X = B) with
  several constant vectors by singular value decomposition (SVD). The
  system is stored in a data file with the following structure :

    Line  1                   : dimension of the matrix (N)
    Lines 2 to (N + 1)        : matrix
    Line  P + 2               : number of constant vectors (P)
    Lines (N + 3) to (2N + 2) : constant vectors (one by column)

  The file MATRIX2.DAT is an example data file with N = 4 and P = 5

  SVD factors the matrix A (n x m, with n >= m) as a product U * S * V'
  where U is a (n x m) column-orthogonal matrix, S a (m x m) diagonal
  matrix with elements >= 0 (the singular values) and V a (m x m)
  orthogonal matrix. The singular values which are close to zero (within
  a user-specified tolerance) are set to zero, and then the solution of
  the system is computed as X = V * Diag(1/s(i)) * U' * B, for s(i) <> 0
  With this editing of the singular values, the method may be applied to
  singular or quasi-singular matrices.

  If the matrix A is singular, two cases are possible:

  1) If the system is underdetermined, the method returns the vector X
     which has the smallest norm.
  2) If the system is impossible, the method returns a vector X such that
     A * X is the best approximation to B in the least-squares sense.
  ********************************************************************** }

uses
  Crt, FMath, Matrices;

const
  TOL = 1.0E-8;  { A singular value will be set to zero if it is   }
                 { lower than TOL times the highest singular value }
var
  A : PMatrix;  { System matrix }
  S : PVector;  { Singular values }
  V : PMatrix;  { Orthogonal matrix from the SV decomposition }
  B : PMatrix;  { Constant vectors }
  X : PMatrix;  { Solutions }
  N : Integer;  { Dimension of matrix }
  P : Integer;  { Number of constant vectors }
  I : Integer;  { Loop variable }
  Ch : Char;    { Key pressed to exit program }

  procedure ReadMatrices(FileName : String; var A, B : PMatrix;
                         var N, P : Integer);
{ ----------------------------------------------------------------------
  Reads data from file. Note that matrices are passed as VAR parameters
  because they are dimensioned inside the procedure.
  ---------------------------------------------------------------------- }
  var
    F : Text;        { Data file }
    I, J : Integer;  { Loop variable }
  begin
    Assign(F, FileName);
    Reset(F);

    { Read matrix }
    Read(F, N);
    DimMatrix(A, N, N);
    for I := 1 to N do
      for J := 1 to N do
        Read(F, A^[I]^[J]);

    { Read constant vectors }
    Read(F, P);
    DimMatrix(B, P, N);
    for J := 1 to N do
      for I := 1 to P do
        Read(F, B^[I]^[J]);  { B^[I] is the I-th constant vector }
    Close(F);
  end;

  procedure WriteMatrix(Title : String; A : PMatrix; N : Integer);
{ ----------------------------------------------------------------------
  Writes system matrix on screen
  ---------------------------------------------------------------------- }
  var
    I, J : Integer;
  begin
    WriteLn(Title, ' :', #10);
    for I := 1 to N do
      begin
        for J := 1 to N do
          Write(A^[I]^[J]:12:6);
        WriteLn;
      end;
    WriteLn;
  end;

  procedure WriteVectors(Title : String; B : PMatrix; N, P : Integer);
{ ----------------------------------------------------------------------
  Writes constant vectors or solution vectors
  so that they appear on columns.
  ---------------------------------------------------------------------- }
  var
    I, J : Integer;
  begin
    WriteLn(Title, ' :', #10);
    for J := 1 to N do
      begin
        for I := 1 to P do
          Write(B^[I]^[J]:12:6);
        WriteLn;
      end;
    WriteLn;
  end;

begin
  ClrScr;

  { Read and display data }
  ReadMatrices('matrix2.dat', A, B, N, P);
  WriteMatrix('System matrix', A, N);
  WriteVectors('Constant vectors', B, N, P);

  { Dimension arrays }
  DimVector(S, N);
  DimMatrix(V, N, N);
  DimMatrix(X, P, N);

  { Perform singular value decomposition of A
    (Note that U is stored in place of A) }
  case SV_Decomp(A, 1, N, N, S, V) of
    MAT_OK : begin
               { Set the lowest singular values to zero }
               SV_SetZero(S, 1, N, TOL);
               { Solve the system for each constant vector }
               for I := 1 to P do
                 SV_Solve(A, S, V, B^[I], 1, N, N, X^[I]);
               WriteVectors('Solution vectors', X, N, P);
             end;
    MAT_NON_CONV : Write('Too many iterations!');
  end;

  GotoXY(1, 25);
  Write('Press a key ... ');
  Ch := ReadKey;
end.


