{ **********************************************************************
  *                         Program HESSIAN.PAS                        *
  *                             Version 1.4                            *
  *                      (c) J. Debord, June 1999                      *
  **********************************************************************
  This program computes the Gradient (vector of first partial derivatives)
  and Hessian (matrix of second partial derivatives) of a function of
  several variables, using numerical differentiation.

  If f is a function of n variables defined by y = f(x(1), x(2),... x(n))
  the gradient vector G and the hessian matrix H are such that :

                  dy                         dy
          G(i) = -----          H(i,j) = -----------          i = 1..n
                 dx(i)                   dx(i) dx(j)          j = 1..n

  The program assumes that the hessian matrix is symmetrical, i.e.
  H(j,i) = H(i,j). This occurs when the second partial derivatives are
  continuous, which is usually the case in applied mathematics.
  ********************************************************************** }

program Hessian;

uses
  Crt, FMath, Matrices, Optim;

{ Define number of variables }
const
  Nvar : Integer = 3;

{ Global variables }
var
  X : PVector;  { Function variables }
  Y : Float;    { Function value }
  G : PVector;  { Gradient vector }
  H : PMatrix;  { Hessian matrix }

{ ----------------------------------------------------------------------
  Define your function here (cf the definition of type TFuncNVar in
  OPTIM.PAS). With a 16-bit compiler (TP/BP) the function must be
  compiled in FAR mode ($F+)
  ---------------------------------------------------------------------- }

{$F+}
function Func(X : PVector) : Float;
begin
  Func := Exp(X^[1] * X^[2] * X^[3]);
end;
{$F-}

procedure WriteResult;
{ Output results to screen }
var
  I, J : Integer;
  Ch   : Char;
begin
  ClrScr;
  Writeln('Variables :', #10);
  for I := 1 to Nvar do
    Writeln(X^[I]:10:4);
  Writeln(#10, 'Function :', #10);
  Writeln(Y:10:4);
  Writeln(#10, 'Gradient :', #10);
  for I := 1 to Nvar do
    Writeln(G^[I]:10:4);
  Writeln(#10, 'Hessian :', #10);
  for I := 1 to Nvar do
    begin
      for J := 1 to Nvar do
        Write(H^[I]^[J]:10:4);
      Writeln;
    end;
  GotoXY(1, 25);
  Write('Press a key ... ');
  Ch := ReadKey;
end;

begin
  { Allocate arrays }
  DimVector(X, Nvar);
  DimVector(G, Nvar);
  DimMatrix(H, Nvar, Nvar);

  { Set the values of the variables for which
    gradient and hessian must be evaluated }
  X^[1] := 0.5;
  X^[2] := 1.0;
  X^[3] := 2.0;

  { Evaluate function }
  Y := Func(X);

  { Compute Gradient and Hessian }
  {$IFDEF FPK}
    NumHessGrad(@Func, X, 1, Nvar, G, H);
  {$ELSE}
    NumHessGrad(Func, X, 1, Nvar, G, H);
  {$ENDIF}

  { Display results }
  WriteResult;

  { Deallocate arrays }
  DelVector(X, Nvar);
  DelVector(G, Nvar);
  DelMatrix(H, Nvar, Nvar);
end.



