{ **********************************************************************
  *                        Program REGPOLY.PAS                         *
  *                            Version 1.0                             *
  *                     (c) J. Debord, July 1999                       *
  **********************************************************************
  This program performs a weighted least squares fit of a straight line
  or polynomial :

                     y = b0 + b1 * x + b2 * x^2 + ...

  The following parameters are passed on the command line :

    1st parameter = name of input file (default extension = .DAT)
                    The structure of the input file is described in
                    REG_IN.INC
    2nd parameter = degree of polynomial (default = 1, i.e. straight line)

  The files LINE.DAT and POLYNOM.DAT are examples of 2- and 5th-order
  polynomials, respectively. The program parameters are :

                              LINE 2
                          or  POLYNOM 5

  The program may be executed from Turbo Pascal's integrated environment,
  in which case the parameters are entered through the "Parameters" option
  of the menu, or from DOS (after compilation into an executable file),
  in which case the parameters are entered on the command line (e.g.
  FITPOLY LINE 2 or FITPOLY POLYNOM 5).
  ********************************************************************** }

program RegPoly;

uses
  Crt, Graph, FMath, Matrices, Polynom, Regress, Models, PaString, Plot;

var
  InFName : String;    { Name of input file }
  Title   : String;    { Title of study }
  XName,
  YName   : String;    { Variable names }
  N       : Integer;   { Number of points }
  X, Y    : PVector;   { Point coordinates }
  U       : PMatrix;   { Matrix of independent variables (not used here) }
  Ycalc   : PVector;   { Expected Y values }
  S       : PVector;   { Standard deviations of Y values }
  CstPar  : PVector;   { Constant parameters }
  B       : PVector;   { Regression parameters }
  V       : PMatrix;   { Variance-covariance matrix of regression parameters }
  Theta   : PVector;   { Variance parameters }
  RegTest : TRegTest;  { Regression tests }
  ErrCode : Integer;   { Error code }


  procedure ReadCmdLine(var InFName : String; var CstPar : PVector);
{ --------------------------------------------------------------------
  Reads command line parameters. Stores constant parameters in CstPar,
  such that :

       CstPar^[0] = Degree of polynomial

  The contents of CstPar are defined in the unit FITPOLY.PAS,
  in the subdirectory REG of the TP Math units directory.
  -------------------------------------------------------------------- }
  var
    Deg, ErrCode : Integer;
  begin
    DimVector(CstPar, 0);

    { Name of input file }
    InFName := ParamStr(1);
    if Pos('.', InFName) = 0 then InFName := InFName + '.dat';

    { Degree of polynomial }
    Deg := 0;
    Val(ParamStr(2), Deg, ErrCode);
    if (ErrCode <> 0) or (Deg < 1) then Deg := 1;
    CstPar^[0] := Deg;
  end;

  {$I REG_IN.INC}    { Read input file }

  {$I REG_OUT.INC}   { Write output file }

  {$I REG_PLOT.INC}  { Plot fitted function }

{ *************************** Main program ***************************** }

begin
  { Read command line parameters }
  ReadCmdLine(InFName, CstPar);

  { Read input file }
  if ReadInputFile(InFName, Title, XName, YName, N, X, Y) <> 0 then
    begin
      WriteLn('Error reading file ', InFName);
      Halt;
    end;

  { Initialize regression and variance models.
    See MODELS.PAS in the REG subdirectory for a list of available models }
  InitModel(REG_POL,
            VAR_CONST,  { Here we use a constant variance }
            CstPar);

  { Dimension arrays.
    Note: the variance parameters Theta^[1]..Theta^[LastVarParam]
    must be supplied if we use a non-constant variance model }
  DimVector(Theta, LastVarParam);
  DimVector(B, LastParam);
  DimMatrix(V, LastParam, LastParam);
  DimVector(Ycalc, N);
  DimVector(S, N);

  { Perform regression. The numbers 1 and 0.1 denote the maximal number
    of iterations and the tolerance on the parameters. They are purely
    formal values here since the polynomial regression does not use an
    iterative minimization algorithm. }
  ErrCode := WLSFit(X, U, Y, N, True, 1, 0.1,
                    Theta, B, V, Ycalc, S, RegTest);

  { Path to the graphic drivers (Default = C:\BP\BGI) }
  BGIPath := 'C:\PROG\BP\BGI';

  { Write results & plot curve }
  case ErrCode of
    MAT_OK     : begin
                   PlotGraph(Title, X, Y, N);
                   WriteOutputFile(InFName, Title, XName, YName,
                                   N, Y, Ycalc, S, B, V, RegTest);
                 end;
    MAT_SINGUL : WriteLn('Singular matrix !');
  end;
end.
