(**********************************************************************)
(* UDatasetImplementation.pas - Copyright (c) 2004 Ricco RAKOTOMALALA *)
(**********************************************************************)

{
@abstract(Mise en oeuvre des structures de donnes)
@author(Ricco)
@created(12/01/2004)
Les structures relles d'accs aux donnes. On va au plus simple possible ici,
toutes les donnes sont charges en mmoire, c'est plus simple. Si on veut des mthodes plus sophistiques, il faudrait passer par
les interfaces. Il faut voir si vraiment a vaut le coup, l'objectif est l'implmentation de mthodes. L'unit en lien direct
avec celui-ci est bien entendu @link(UDatasetDefinition).
30/07/03 - Par dfaut, aucun attribut n'est slectionn.
}
unit UDatasetImplementation;

interface

USES
        Classes,
        UDatasetDefinition,
        UDatasetExamples;

TYPE
        {Structure de tableau discret - 1 based}
        TTabDataDiscrete = array[1..MAX_NB_EXAMPLES] of TTypeDiscrete;
        PTabDataDiscrete = ^TTabDataDiscrete;
        
        {Attribut de type discret - toutes les mthodes sont overrides, mmes celles
        qui ne semblent pas appropries (ex. cGetValue)}
        TAttDiscrete = class(TAttribute)
                       private
                       FDData: PTabDataDiscrete;
                       protected 
                       function    dGetValue(i: Integer): TTypeDiscrete; override;
                       function    cGetValue(i: Integer): TTypeContinue; override;
                       function    sGetValue(i: Integer): string; override;
                       procedure   dSetValue(i: Integer; const prmValue: TTypeDiscrete); override;
                       procedure   cSetValue(i: Integer; const prmValue: TTypeContinue); override;
                       procedure   sSetValue(i: Integer; const prmValue: string); override;
                       procedure   SetSize(newSize: Integer); override;
                       procedure   SetCategory(); override;
                       public
                       constructor Create(prmName: string; prmSize: Integer);
                       destructor  Destroy; override;
                       {sauver dans le flux}
                       procedure   SaveToStream(prmStream: TStream); override;
                       {construire  partir d'un chargement de flux}
                       constructor CreateFromStream(prmStream: TStream); override;
                       end;

        {Structure de tableau continu - 1 based}
        TTabDataContinue = array[1..MAX_NB_EXAMPLES] of TTypeContinue;
        PTabDataContinue = ^TTabDataContinue;

        {Attribut de type continu - idem, toutes les mthodes sont overrides, mme
        sGetValue, dGetValue, etc.}
        TAttContinue = class(TAttribute)
                       private
                       FCData: PTabDataContinue;  
                       protected
                       function    dGetValue(i: Integer): TTypeDiscrete; override;
                       function    cGetValue(i: Integer): TTypeContinue; override;
                       function    sGetValue(i: Integer): string; override;
                       procedure   dSetValue(i: Integer; const prmValue: TTypeDiscrete); override;
                       procedure   cSetValue(i: Integer; const prmValue: TTypeContinue); override;
                       procedure   sSetValue(i: Integer; const prmValue: string); override;
                       procedure   SetSize(newSize: Integer); override;
                       procedure   SetCategory(); override;
                       public
                       constructor Create(prmName: string; prmSize: Integer);
                       destructor  Destroy; override;
                       {sauver dans le flux}
                       procedure   SaveToStream(prmStream: TStream); override;
                       {construire  partir d'un chargement de flux}
                       constructor CreateFromStream(prmStream: TStream); override;
                       end;

        {le rle d'un attribut dans une analyse}
        TEnumAttStatus = (asAll,asTarget,asInput,asIllus);

CONST
        {description de ces diffrents status}
        STR_ENUM_ATT_STATUS : array[TEnumAttStatus] of string
                              = ('All','Target','Input','Illustrative');  

TYPE
        {Un ensemble de donnes pour une analyse de machine learning}
        TMLDataset   = class(TObject)
                       private
                       {tous les attributs de la base de donnes}
                       FAllAtts: TLstAttributes;
                       {les attributs endognes}
                       FTargetAtts: TLstAttributes;
                       {les attributs exognes}
                       FInputAtts: TLstAttributes;
                       {les attributs illustratifs}
                       FIllusAtts: TLstAttributes;
                       {Vecteur d'individus associs  l'ensemble de donnes}
                       FExamples: TExamples;
                       {cration des pointeurs idoines}
                       procedure   InitializeLst(prmSize: integer);
                       {rcuprer la bonne liste}
                       function    GetLstAtts(prmAttStatus: TEnumAttStatus): TLstAttributes;
                       public
                       {un premier mode de construction possible - on lui passe des attributs
                       il vaut surtout pour le premier composant}
                       constructor CreateFromLstAtt(prmAtts: TLstAttributes);
                       {un deuxime mode de construction - il est connect  un autre dataset}
                       constructor CreateFromMLDataset(prmSource: TMLDataset);
                       {copie les attributs, sans le vecteur des exemples}
                       procedure   CopyWithoutExamples(prmSource: TMLDataset);
                       {copier un ensemble de donnes avec ses proprits}
                       procedure   Copy(prmSource: TMLDataset);
                       {supprime les listes avant de se dtruire}
                       destructor  Destroy; override;
                       {toutes les donnes dispo}
                       property    LstAtts[prmAttStatus: TEnumAttStatus]: TLstAttributes read GetLstAtts;
                       {le vecteur des individus}
                       property    Examples: TExamples read FExamples;
                       end;

implementation

USES
        SYSUTILS, UConstConfiguration;

{ TAttDiscrete }

function TAttDiscrete.cGetValue(i: Integer): TTypeContinue;
begin
 result:= self.dGetValue(i);
end;

constructor TAttDiscrete.Create(prmName: string; prmSize: Integer);
begin
 inherited Create(prmName,prmSize);
 //allocation
 FDData:= AllocMem(prmSize*SIZE_DISCRETE);
end;

constructor TAttDiscrete.CreateFromStream(prmStream: TStream);
begin
 inherited CreateFromStream(prmStream);
 FDData:= AllocMem(FSize*SIZE_DISCRETE);
 prmStream.ReadBuffer(FDData^,FSize*SIZE_DISCRETE);
end;

procedure TAttDiscrete.cSetValue(i: Integer; const prmValue: TTypeContinue);
begin
 self.dSetValue(i,TRUNC(prmValue));
end;

destructor TAttDiscrete.Destroy;
begin
 ReAllocMem(FDData,0);
 inherited Destroy;
end;

function TAttDiscrete.dGetValue(i: Integer): TTypeDiscrete;
begin
 result:= FDData^[i];
end;

procedure TAttDiscrete.dSetValue(i: Integer; const prmValue: TTypeDiscrete);
begin
 FDData^[i]:= prmValue;
end;

procedure TAttDiscrete.SaveToStream(prmStream: TStream);
begin
 inherited SaveToStream(prmStream);
 prmStream.WriteBuffer(FDData^,FSize*SIZE_DISCRETE);
end;

procedure TAttDiscrete.SetCategory;
begin
 FCategory:= caDiscrete;
end;

procedure TAttDiscrete.SetSize(newSize: Integer);
begin
 if (newSize<>FSize)
  then
   begin
    FSize:= newSize;
    ReAllocMem(FDData,newSize*SIZE_DISCRETE);
   end;
end;

function TAttDiscrete.sGetValue(i: Integer): string;
begin
 result:= FLstValues.getDescription(self.dGetValue(i));
end;

procedure TAttDiscrete.sSetValue(i: Integer; const prmValue: string);
begin
 self.dSetValue(i,FLstValues.getValue(prmValue));
end;

{ TAttContinue }

function TAttContinue.cGetValue(i: Integer): TTypeContinue;
begin
 result:= FCData^[i];
end;

constructor TAttContinue.Create(prmName: string; prmSize: Integer);
begin
 inherited Create(prmName,prmSize);
 FCData:= AllocMem(prmSize*SIZE_CONTINUE);
end;

constructor TAttContinue.CreateFromStream(prmStream: TStream);
begin
 inherited CreateFromStream(prmStream);
 FCData:= AllocMem(FSize*SIZE_CONTINUE);
 prmStream.ReadBuffer(FCData^,FSize*SIZE_CONTINUE);
end;

procedure TAttContinue.cSetValue(i: Integer; const prmValue: TTypeContinue);
begin
 FCData^[i]:= prmValue;
end;

destructor TAttContinue.Destroy;
begin
 ReAllocMem(FCData,0);
 inherited Destroy;
end;

function TAttContinue.dGetValue(i: Integer): TTypeDiscrete;
begin
 result:= TRUNC(self.cGetValue(i));//danger d'erreur de troncature si valeur suprieure  255 ou ngative
end;

procedure TAttContinue.dSetValue(i: Integer; const prmValue: TTypeDiscrete);
begin
 self.cSetValue(i,prmValue);
end;

procedure TAttContinue.SaveToStream(prmStream: TStream);
begin
 inherited SaveToStream(prmStream);
 prmStream.WriteBuffer(FCData^,FSize*SIZE_CONTINUE);
end;

procedure TAttContinue.SetCategory;
begin
 FCategory:= caContinue;
end;

procedure TAttContinue.SetSize(newSize: Integer);
begin
 if (newSize<>FSize)
  then
   begin
    FSize:= newSize;
    ReAllocMem(FCData,newSize*SIZE_CONTINUE);
   end;
end;

function TAttContinue.sGetValue(i: Integer): string;
begin
 result:= Format(VIEW_ATTRIBUTE_ACCURACY_CONTINUE,[self.cGetValue(i)]);
end;

procedure TAttContinue.sSetValue(i: Integer; const prmValue: string);
begin
 (*
 //la rapidit prime ici...
 //cette procdure est trs souvent appele lors des imports de fichiers de donnes
 //val n'accepte que le '.' comme sparateur dcimal
 //val(prmValue,FCData^[i],err);
 //strtofloat est un peu lent et effectue trop de tests supplmentaires
 FCData^[i]:= StrToFloat(prmValue);
 //court-circuiter strtofloat en virant le gestionnaire d'exceptions - le gain n'est pas si terrible que a...
 //TextToFloat(PChar(prmValue),v,fvExtended);
 //FCData^[i]:= v;
 *)

 //new -- 22/11/2004 -- nouvelle criture -- pour grer spcifiquement le cas du '.' comme point dcimal
 //les perfs dpendent du temps de traitement du IF -- c'est pas concluant du tout !!!
 (*
 if (CURRENT_DECIMAL_SEPARATOR = POINT_DECIMAL_SEPARATOR)
  then Val(prmValue,FCData^[i],err)
  else FCData^[i]:= StrToFloat(prmValue);
 *)

 (*
 p:= POS(CURRENT_DECIMAL_SEPARATOR,prmValue);
 if (p>0) then prmValue[p]:= POINT_DECIMAL_SEPARATOR;
 Val(prmValue,FCData^[i],err);
 *)

 //bon, ben... on en reste l pour l'instant...
 FCData^[i]:= StrToFloat(prmValue);
end;

{ TMLDataset }

procedure TMLDataset.Copy(prmSource: TMLDataset);
begin
 self.CopyWithoutExamples(prmSource);
 //les individus
 FExamples.Copy(prmSource.Examples);
end;

procedure TMLDataset.CopyWithoutExamples(prmSource: TMLDataset);
var attstatus: TEnumAttStatus;
begin
 //les variables
 for attstatus:= low(TEnumAttStatus) to high(TEnumAttStatus) do
  self.LstAtts[attstatus].Assign(prmSource.LstAtts[attstatus]);
end;

constructor TMLDataset.CreateFromLstAtt(prmAtts: TLstAttributes);
begin
 inherited Create();
 InitializeLst(prmAtts.Size);
 FAllAtts.Assign(prmAtts);
 //traitement des individus
 FExamples:= TExamples.Create(prmAtts.Size);
 FExamples.Initialize();//rempli de numros successifs
end;

constructor TMLDataset.CreateFromMLDataset(prmSource: TMLDataset);
begin
 inherited Create();
 InitializeLst(prmSource.FAllAtts.Size);
 FExamples:= TExamples.Create(prmSource.Examples.Size);
 self.Copy(prmSource);
end;

destructor TMLDataset.Destroy;
begin
 FAllAtts.Free;
 FInputAtts.Free;
 FTargetAtts.Free;
 FIllusAtts.Free;
 inherited Destroy;
end;

function TMLDataset.GetLstAtts(
  prmAttStatus: TEnumAttStatus): TLstAttributes;
begin
 case prmAttStatus of
  asAll:    result:= FAllAtts;
  asTarget: result:= FTargetAtts;
  asInput:  result:= FInputAtts;
  asIllus:  result:= FIllusAtts
  else result:= NIL;//le pire des cas...
 end;
end;

procedure TMLDataset.InitializeLst(prmSize: integer);
begin
 {NOT owned !!!}
 FAllAtts:= TLstAttributes.Create(FALSE,prmSize);
 FInputAtts:= TLstAttributes.Create(FALSE,prmSize);
 FTargetAtts:= TLstAttributes.Create(FALSE,prmSize);
 FIllusAtts:= TLstAttributes.Create(FALSE,prmSize);
end;

initialization
 RegisterClass(TAttDiscrete);
 RegisterClass(TAttContinue);
 //rcuprer le point dcimal courant (paramtres rgionaux de Windows)
 //CURRENT_DECIMAL_SEPARATOR:= DecimalSeparator;
end.
