unit UXlsPageBreaks;

interface
uses UXlsBaseRecords, UXlsBaseList, XlsMessages, Classes, SysUtils;

type
  TBreakData= Packed Record
    Row: Word;
    Col1: Word;
    Col2: Word;
  end;

  THPageBreakRecord = class (TBaseRecord)
  public
    function Count: Word;
    function BreakData(const index: integer): TBreakData;
  end;

  TPageBreak=class
  public
    BreakData: TBreakData;

    constructor Create(const aBreakData: TBreakData);
    function CopyTo: TPageBreak;
  end;

  TPageBreakList= class (TBaseList)
  {$INCLUDE TPageBreakListHdr.inc}
  public
    procedure AddRecord(const aRecord: THPageBreakRecord);
    procedure AddBreak(const aRow: Longint);
    procedure CopyFrom(const aBreakList: TPageBreakList);
    procedure SaveToStream(const DataStream: TStream);
    function TotalSize: int64;

    procedure InsertRows(const DestRow: integer; const aCount: integer);
    procedure DeleteRows(const DestRow: integer; const aCount: integer);
  end;

implementation
{$INCLUDE TPageBreakListImp.inc}

{ THPageBreakRecord }

function THPageBreakRecord.BreakData(const index: integer): TBreakData;
begin
  Move(Data[2+index*SizeOf(TBreakData)],Result, SizeOf(TBreakData));
end;

function THPageBreakRecord.Count: Word;
begin
  Result:= GetWord(Data, 0);
end;

{ TPageBreak }

function TPageBreak.CopyTo: TPageBreak;
begin
  Result:= TPageBreak.Create(BreakData);
end;

constructor TPageBreak.Create(const aBreakData: TBreakData);
begin
  inherited Create;
  BreakData:= aBreakData;
end;

{ TPageBreakList }

procedure TPageBreakList.AddBreak(const aRow: Integer);
var
  Index: integer;
  BreakData: TBreakData;
begin
  BreakData.Row:=aRow;
  BreakData.Col1:=0;
  BreakData.Col2:=$FF;
  if Count>MaxHPageBreaks then raise Exception.Create(ErrTooManyPageBreaks);
  if not Find(aRow, Index) then Insert(Index, TPageBreak.Create(BreakData));
end;

procedure TPageBreakList.AddRecord(const aRecord: THPageBreakRecord);
var
  i, Index: integer;
begin
  for i:=0 to aRecord.Count - 1 do
    if not Find(aRecord.BreakData(i).Row, Index) then Insert(Index, TPageBreak.Create(aRecord.BreakData(i)));
  aRecord.Free;
end;

procedure TPageBreakList.CopyFrom(const aBreakList: TPageBreakList);
var
  i:integer;
begin
  if aBreakList=nil then exit;
  for i:=0 to aBreakList.Count-1 do Add(aBreakList[i].CopyTo);
end;


procedure TPageBreakList.DeleteRows(const DestRow, aCount: integer);
var
  Index: integer;
  i: integer;
begin
  Find(DestRow, Index);
  for i:=Count-1 downto Index do
    if Items[i].BreakData.Row<DestRow+aCount then Delete(i) else dec(Items[i].BreakData.Row, aCount);
end;

procedure TPageBreakList.InsertRows(const DestRow, aCount: integer);
var
  Index: integer;
  i: integer;
begin
  Find(DestRow, Index);
  for i:=Index to Count-1 do IncMax(Items[i].BreakData.Row, aCount, $FFFF);
end;

procedure TPageBreakList.SaveToStream(const DataStream: TStream);
var
  RecordHeader: TRecordHeader;
  i: integer;
  MyCount: word;
begin
  if Count > 0 then
  begin
    Sort; //Just in case...
    RecordHeader.Id:= xlr_HORIZONTALPAGEBREAKS;
    RecordHeader.Size:= SizeOf(MyCount)+Count*SizeOf(TBreakData);

    MyCount:=Count;
    if DataStream.Write(RecordHeader, Sizeof(RecordHeader)) <> Sizeof(RecordHeader) then raise Exception.Create(ErrCantWrite);
    if DataStream.Write(MyCount, Sizeof(MyCount)) <> Sizeof(MyCount) then raise Exception.Create(ErrCantWrite);
    for i:=0 to Count-1 do
    begin
      if DataStream.Write(Items[i].BreakData, SizeOf(Items[i].BreakData)) <> SizeOf(Items[i].BreakData) then
        raise Exception.Create(ErrCantWrite);
    end;
  end;
end;

function TPageBreakList.TotalSize: int64;
begin
  if Count=0 then Result:=0
  else Result:=sizeOf(TRecordHeader)+SizeOf(Word)+Count*SizeOf(TBreakData);
end;

end.
