(**********************************************************************)
(* UCalcStatDesPairedTest.pas - Copyright (c) 2005 Ricco RAKOTOMALALA *)
(**********************************************************************)

{
@abstract(Classes de calcul pour les tests sur chantillons apparis)
@author(Ricco)
@created(17/07/2005)

Principale rfrence :
----------------------
Siegel & Castellan (Nonparametric Statistics for the Behavioral Science -- dition de 1988) -- chapitre 5 et 7

}

unit UCalcStatDesPairedTest;

interface

USES
        UCalcStatDes,
        UDatasetDefinition,
        UDatasetImplementation,
        UDatasetExamples;

TYPE
   //classe gnrique pour les tests sur chantillons apparis -- forward
   TCalcSDPairedTest = class;

   //classe de classe test sur chantillons apparis
   TClassCalcSDPairedTest = class of TCalcSDPairedTest;

   //classe gnrique pour les tests sur chantillons apparis
   TCalcSDPairedTest = class(TCalcStatDesContinuous)
                       private
                       //le second attribut
                       FSecondAttribute: TAttribute;
                       //la stat. associe
                       FStatSecondAttribute: TCalcStatDesContinuous;
                       protected
                       //procdure de calcul interne des stats --  surcharger
                       procedure   computeStatPairedTest(examples: TExamples); virtual; abstract;
                       public
                       //constructeur
                       constructor create(prmOne, prmTwo: TAttribute; prmExamples: TExamples);
                       //destructeur
                       destructor  destroy(); override;
                       //la statistique utilise pour le tri --  surcharger
                       function    getStatForSorting(): double; virtual; abstract;
                       //la p-value utilise pour le tri --  surcharger
                       function    getPValueForSorting(): double; virtual; abstract;
                       //description des rsultats --  surcharger
                       function    getHTMLStatDescription(): string; virtual; abstract;
                       //envoyer le rapport complet pour le test courant
                       function    getHTMLResult(prmOption: integer = -1): string; override;
                       //appel du calcul -- ne plus surcharger
                       procedure   RefreshStat(prmExamples: TExamples); override;
                       //proprits
                       property    SecondAttribute: TAttribute read FSecondAttribute;
                       end;

   //classe gnrique pour la liste de stats
   TLstCalcSDPairedTest = class(TLstCalcStatDesContinuous)
                          public
                          //affichage de l'en-tte des stats
                          function    getHeaderHTML(): string; override;
                          //trier les stats selon les critres
                          procedure SortStats(); override;
                          end;

   //classe de classe liste de stats pour tests sur chantillons apparis
   TClassLstCalcSDPairedTest = class of TLstCalcSDPairedTest;

   //**********************************************************************
   //************** Test des signes ***************************************
   //**********************************************************************

   //classe de calcul pour le test des signes
   TCalcSDSignTest = class(TCalcSDPairedTest)
                     private
                     //effectif rellement utilis
                     FUsedExamples: integer;
                     //le comptage
                     FPlus, FMoins: integer;
                     //Z et p-value
                     FZValue, FPValue: double;
                     protected
                     //calcul
                     procedure   computeStatPairedTest(examples: TExamples); override;
                     public
                     //la statistique utilise pour le tri
                     function    getStatForSorting(): double; override;
                     //la p-value utilise pour le tri
                     function    getPValueForSorting(): double; override;
                     //description des rsultats
                     function    getHTMLStatDescription(): string; override;
                     end;

   //liste d'objets test des signes
   TLstCalcSDSignTest = class(TLstCalcSDPairedTest)
                        end;

   //*********************************************************************
   //************** Test de Student pour chantillons apparis ***********
   //*********************************************************************

   //classe de calcul du paired T-test (loi de Student)
   TCalcSDPairedTTest = class(TCalcSDPairedTest)
                        private
                        //moyenne des carts
                        FDMean: double;
                        //cart-type des carts
                        FDStdDev: double;
                        //statistique
                        FTTest: double;
                        //p-value
                        FTPValue: double;
                        protected
                        //calcul
                        procedure   computeStatPairedTest(examples: TExamples); override;
                        public
                        //la statistique utilise pour le tri
                        function    getStatForSorting(): double; override;
                        //la p-value utilise pour le tri
                        function    getPValueForSorting(): double; override;
                        //description des rsultats
                        function    getHTMLStatDescription(): string; override;
                        end;

   //liste d'objets test des signes
   TLstCalcSDPairedTTest = class(TLstCalcSDPairedTest)
                           end;

   //*********************************************************************
   //************ Test de Wilcoxon pour chantillons apparis ************
   //*********************************************************************

   //classe de calcul -- test des rangs signs
   TCalcSDPairedWilcoxonSignedRank = class(TCalcSDPairedTest)
                                     private
                                     //exemples utiliss
                                     FUsedExamples: integer;
                                     //les indicateurs -- somme des rangs positifs et ngatifs
                                     FTPlus, FTMoins: double;
                                     //la Z et sa p-value
                                     FZValue, FPValue: double;  
                                     protected
                                     //calcul
                                     procedure   computeStatPairedTest(examples: TExamples); override;
                                     public
                                     //la statistique utilise pour le tri
                                     function    getStatForSorting(): double; override;
                                     //la p-value utilise pour le tri
                                     function    getPValueForSorting(): double; override;
                                     //description des rsultats
                                     function    getHTMLStatDescription(): string; override;
                                     end;

   //liste pour les stats
   TLstCalcSDPairedWilcoxonSignedRank = class(TLstCalcSDPairedTest)
                                        end;   


implementation

USES
   FMath, Math,
   Classes, Sysutils,
   UConstConfiguration, UCalcStatDesRankForNonParametricStat;

{ TCalcSDPairedTest }

constructor TCalcSDPairedTest.create(prmOne, prmTwo: TAttribute;
  prmExamples: TExamples);
begin
 //brancher et les cas chant calculer les stats sur la second variables
 FSecondAttribute:= prmTwo;
 FStatSecondAttribute:= TCalcStatDesContinuous.Create(FSecondAttribute,prmExamples);
 //brancher la premire variable
 inherited Create(prmOne,prmExamples);
end;

{ TLstCalcSDPairedTest }

//***************************************
//procdure locales pour le tri des stats

{trier selon les noms d'attributs}
function ListSortCompareNameSecondAttribute(item1,item2: pointer): integer;
var st1,st2: TCalcSDPairedTest;
begin
 st1:= TCalcSDPairedTest(item1);
 st2:= TCalcSDPairedTest(item2);
 if (st1.SecondAttribute.Name<st2.SecondAttribute.Name)
  then result:= -1
  else
   if (st1.SecondAttribute.Name>st2.SecondAttribute.Name)
    then result:= +1
    else result:= 0;
end;

function ListSortCompareStat(item1,item2: pointer): integer;
var st1,st2: TCalcSDPairedTest;
begin
 st1:= TCalcSDPairedTest(item1);
 st2:= TCalcSDPairedTest(item2);
 //tri invers !!!
 if (st1.getStatForSorting()<st2.getStatForSorting())
  then result:= +1
  else
   if (st1.getStatForSorting()>st2.getStatForSorting())
    then result:= -1
    else result:= 0;
end;

function ListSortComparePValue(item1,item2: pointer): integer;
var st1,st2: TCalcSDPairedTest;
begin
 st1:= TCalcSDPairedTest(item1);
 st2:= TCalcSDPairedTest(item2);
 if (st1.getPValueForSorting()<st2.getPValueForSorting())
  then result:= -1
  else
   if (st1.getPValueForSorting()>st2.getPValueForSorting())
    then result:= +1
    else result:= 0;
end;
//***************************************

function TLstCalcSDPairedTest.getHeaderHTML: string;
begin
 result:= HTML_TABLE_COLOR_HEADER_GRAY+'<TH>Attribute_Y</TH><TH>Attribute_X</TH><TH>Statistical test</TH></TR>';
end;

procedure TLstCalcSDPairedTest.SortStats;
var funcCompare: TListSortCompare;
begin
 if (CompareMode>=0)
  then
   begin

    case CompareMode of
     0: funcCompare:= ListSortCompareName;
     1: funcCompare:= ListSortCompareNameSecondAttribute;
     2: funcCompare:= ListSortCompareStat;
     3: funcCompare:= ListSortComparePValue;
     else
      funcCompare:= NIL;
    end;

    if assigned(funcCompare)
     then LstStat.Sort(funcCompare);
   end;
end;

destructor TCalcSDPairedTest.destroy;
begin
 if assigned(FStatSecondAttribute) then FStatSecondAttribute.Free();
 inherited;
end;

function TCalcSDPairedTest.getHTMLResult(prmOption: integer): string;
var s: string;
begin
 //description de variable 1
 s:= '<TD valign="top">';
 s:= s+HTML_HEADER_TABLE_RESULT;
 s:= s+HTML_TABLE_COLOR_HEADER_GREEN+format('<TH colspan=2>%s</TH></TR>',[self.Attribute.Name]);
 s:= s+HTML_TABLE_COLOR_DATA_GREEN+format('<TD>%s</TD><TD align="right">'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',['Avg',self.Average]);
 s:= s+HTML_TABLE_COLOR_DATA_GREEN+format('<TD>%s</TD><TD align="right">'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',['Std-dev',self.StdDev]);
 s:= s+'</table></TD>';
 //description de la variable 2
 s:= s+'<TD valign="top">';
 s:= s+HTML_HEADER_TABLE_RESULT;
 s:= s+HTML_TABLE_COLOR_HEADER_GREEN+format('<TH colspan=2>%s</TH></TR>',[self.SecondAttribute.Name]);
 s:= s+HTML_TABLE_COLOR_DATA_GREEN+format('<TD>%s</TD><TD align="right">'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',['Avg',FStatSecondAttribute.Average]);
 s:= s+HTML_TABLE_COLOR_DATA_GREEN+format('<TD>%s</TD><TD align="right">'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD></TR>',['Std-dev',FStatSecondAttribute.StdDev]);
 s:= s+'</table></TD>';
 //appel de la description de la statistique et des rsultats associs
 result:= s+format('<td>%s%s</table></td>',[HTML_HEADER_TABLE_RESULT,self.getHTMLStatDescription()]);
end;

procedure TCalcSDPairedTest.RefreshStat(prmExamples: TExamples);
begin
 //stats sur le premier attribut
 inherited RefreshStat(prmExamples);
 //stats sur le second attribut
 FStatSecondAttribute.RefreshStat(prmExamples);
 //puis calcul de la stat avec appariement
 computeStatPairedTest(prmExamples);
end;

{ TCalcSDSignTest }

procedure TCalcSDSignTest.computeStatPairedTest(examples: TExamples);
var i: integer;
    example: integer;
    continuite: double;
begin
 //nombre de dpart
 FUsedExamples:= examples.Size;
 //comptage
 FPlus:= 0;
 FMoins:= 0;
 for i:= 1 to examples.Size do
  begin
   example:= examples.Number[i];
   if (Attribute.cValue[example] > SecondAttribute.cValue[example])
    then inc(FPlus)
    else
     if (Attribute.cValue[example] < SecondAttribute.cValue[example])
      then inc(FMoins)
      //dcrmenter le nombre d'exemples utiliss en cas d'galit des valeurs
      else dec(FUsedExamples);
  end;
 //calcul de la correction de continuit
 if (FPlus > 0.5*FUsedExamples)
  then continuite:= -1.0
  else if (FPlus < 0.5*FUsedExamples)
        then continuite:= +1.0
        //et oui, sinon a fausserait les rsultats
        else continuite:= 0.0;
 //calcul du Z avec correction de continuit
 if (FUsedExamples > 0)
  then FZValue:= (2.0 * FPlus + continuite - 1.0 * FusedExamples) / SQRT(FUsedExamples)
  else FZValue:= 0.0;
 //calcul de la p-value avec la loi normale
 FPValue:= PNorm(abs(FZValue));
end;

function TCalcSDSignTest.getHTMLStatDescription: string;
var s: string;
begin
 s:= HTML_TABLE_COLOR_HEADER_BLUE+'<TH>Measure</TH><TH>Value</TH></TR>';
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>%d</td></tr>',['Used examples',FUsedExamples]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>%d</td></tr>',['Positive',FPlus]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>%d</td></tr>',['Negative',FMoins]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</td></tr>',['Z',FZValue]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</td></tr>',['p-value',FPValue]);
 //and then...
 result:= s;
end;

function TCalcSDSignTest.getPValueForSorting: double;
begin
 result:= FPValue;
end;

function TCalcSDSignTest.getStatForSorting: double;
begin
 result:= abs(FZValue);
end;

{ TCalcSDPairedTTest }

procedure TCalcSDPairedTTest.computeStatPairedTest(examples: TExamples);
var i,example: integer;
    d,sum,sum2: double;
begin
 //assurer le coup
 FTTest:= 0.0;
 FTPValue:= 1.0;
 //si calcul possible
 if (examples.Size > 0)
  then
   begin
     //calculer les stats
     sum:= 0.0;
     sum2:= 0.0;
     for i:= 1 to examples.Size do
      begin
       example:= examples.Number[i];
       d:= Attribute.cValue[example] - SecondAttribute.cValue[example];
       sum:= sum + d;
       sum2:= sum2 + d*d;
      end;
     //calcul des stats sur les diffrences
     FDMean:= sum/(1.0*examples.Size);
     FDStdDev:= SQRT(abs(sum2 - 1.0 * examples.Size * FDMean * FDMean)/(-1.0 + examples.Size));
     //la stat. de test
     if (FDStdDev > 0) and (examples.Size > 0)
      then
       begin
        FTTest:= FDMean/(FDStdDev/sqrt(1.0*examples.Size));
        FTPValue:= PStudent(examples.Size - 1,FTTest);
       end;
   end;
end;

function TCalcSDPairedTTest.getHTMLStatDescription: string;
var s: string;
begin
 s:= HTML_TABLE_COLOR_HEADER_BLUE+'<TH>Measure</TH><TH>Value</TH></TR>';
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</td></tr>',['D avg.',FDMean]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</td></tr>',['D std-dev',FDStdDev]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</td></tr>',['T-test',FTTest]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</td></tr>',['p-value',FTPValue]);
 //and then...
 result:= s;
end;

function TCalcSDPairedTTest.getPValueForSorting: double;
begin
 result:= FTPValue;
end;

function TCalcSDPairedTTest.getStatForSorting: double;
begin
 result:= abs(FTTest);
end;

{ TCalcSDPairedWilcoxonSignedRank }

procedure TCalcSDPairedWilcoxonSignedRank.computeStatPairedTest(
  examples: TExamples);
var abs_attDif: TAttribute;
    signed_attDif: TAttribute;   
    tmpLst: TExamples;
    i,example: integer;
    d,correction: double;
    rank: TRankComputedPairedWilcoxon;
    vDiff, vRank: double;
    N,muT, varT: double;

begin
 //construire une var. intmdiaire pour calculer la diffrence (en valeur absolue)
 abs_attDif:= TAttContinue.Create('abs_diff',self.Attribute.Size);
 //la diffrence signe
 signed_attDif:= TAttContinue.Create('signed_diff',self.Attribute.Size);
 //les individus  rfrencer (pour ne pas tenir compte des individus avec un cart nul)
 FUsedExamples:= examples.Size;
 tmpLst:= TExamples.Create(examples.Size);
 tmpLst.BeginAdd();
 //assigner
 for i:= 1 to examples.Size do
  begin
   example:= examples.Number[i];
   d:= Attribute.cValue[example] - SecondAttribute.cValue[example];
   //les rangs sont calculs sur la valeur absolue des diffrences
   abs_attDif.cValue[example]:= abs(d);
   //mais on a besoin galement des valeurs signes (on aurait plus faire l'conomie des valeurs et ne garder qu'un indicateur de signe !)
   signed_attDif.cValue[example]:= d;
   if (abs(d) > 0)
    then tmpLst.AddExample(example)
    //corriger le nombre rel d'individus  utiliser
    else dec(FUsedExamples);
  end;
 //trier la liste des individus  utiliser
 tmpLst.EndAdd();
 tmpLst.QuickSortBy(abs_attDif);
 //construire les rangs
 rank:= TRankComputedPairedWilcoxon.create(abs_attDif);
 correction:= rank.computeRank(tmpLst);
 //calculer des statistiques -- Siegel, pp. 90
 FTPlus:= 0.0;
 FTMoins:= 0.0;
 for i:= 1 to tmpLst.Size do
  begin
   example:= tmpLst.Number[i];
   //valeur de la diffrence (signe)
   vDiff:= signed_attDif.cValue[example];
   //la valeur du rang associ
   vRank:= rank.getRank(example);
   //les diffrences gales  zro ont t limine ne peut pas tre gal  zro !
   //c'est ici que l'on tient compte du signe des rangs
   if (vDiff < 0)
    then FTMoins:= FTMoins + vRank
    else FTPlus:= FTPlus + vRank;
  end;
 //transtypage pour un calcul plus efficace
 N:= 1.0*FUsedExamples;
 //calcul de la moyenne -- Siegel, pp. 91
 muT:= N * (N + 1.0) / 4.0;
 //calcul de la variance -- formule corrige pour les ties -- Siegel, pp.94 (formule 5.6)
 varT:= N * (N + 1.0) * (2.0 * N + 1.0) / 24.0 - 0.5 * correction;
 //Z-value
 FZValue:= 0.0;
 FPValue:= 1.0;
 if (varT > 0.0)
  then
   begin
    FZValue:= (FTPlus - muT) / sqrt(varT);
    FPValue:= PNorm(abs(FZValue));
   end;
 //librer
 rank.Free();
 tmpLst.Free();
 abs_attDif.Free();
 signed_attDif.Free();
end;

function TCalcSDPairedWilcoxonSignedRank.getHTMLStatDescription: string;
var s: string;
begin
 s:= HTML_TABLE_COLOR_HEADER_BLUE+'<TH>Measure</TH><TH>Value</TH></TR>';
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>%d</td></tr>',['Used examples',FUsedExamples]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>%.1f</td></tr>',['Sum ranks +',FTPlus]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>%.1f</td></tr>',['Sum ranks -',FTMoins]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</td></tr>',['Z',FZValue]);
 s:= s+HTML_TABLE_COLOR_DATA_BLUE+format('<td>%s</td><td align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</td></tr>',['p-value',FPValue]);
 //and then...
 result:= s;
end;

function TCalcSDPairedWilcoxonSignedRank.getPValueForSorting: double;
begin
 result:= FPValue;
end;

function TCalcSDPairedWilcoxonSignedRank.getStatForSorting: double;
begin
 result:= abs(FZValue);
end;

end.
