(****************************************************************************)
(* UCalcStatDesConditionnalDesc.pas - Copyright (c) 2004 Ricco RAKOTOMALALA *)
(****************************************************************************)

{
@abstract(Calcul des rangs pour les stats non-paramtriques)
@author(Ricco)
@created(14/07/2005)

Utilisation standardise pour Mann-Whitney, Kruskal-Wallis, Spearman, etc.
Deux choses importantes :
(1) calcul en temps linaire des rangs moyens
(2) calcul  la vole de la correction pour les stats. 
}
unit UCalcStatDesRankForNonParametricStat;

interface

USES
   UDatasetDefinition,
   UDatasetExamples;

TYPE
   //type de l'indicateur de rang
   TTypeRankValue = single;

   //structure de calcul des rangs
   //Tableau temporaire de conservation des rangs
   TTypeRankTab = array of TTypeRankValue;

   //classe de calcul des rangs
   TRankComputed = class(TObject)
                   private
                   //attribut de rfrence
                   FAtt: TAttribute;
                   //tableau des rangs
                   FRankValue: TTypeRankTab;
                   //coefficient de correction
                   FTieCorrection: double;
                   //calculer le rang brut
                   procedure   computeRankNoTied(lstSorted: TExamples);
                   //corriger avec les rangs moyens
                   procedure   computeRankTied(lstSorted: TExamples); virtual;
                   protected
                   //la correction  calculer pour chaque groupe -- c'est ce qui sera surcharg dans les classes descendantes (e.g. Kendall)
                   function    computeCorrection(nbExAequo: double): double; virtual;
                   public
                   //passer l'attribut de rfrence
                   constructor create(prmAtt: TAttribute); virtual;
                   //dtruire
                   destructor  destroy(); override;
                   //calculer les rangs -- renvoie la valeur de la correction
                   //new -- 04/08/2005 -- on calcule mme pour les ex-aequos de taille 1 i.e. pas d'ex-aequos ?
                   function    computeRank(lstSorted: TExamples): double;
                   //rcuprer le rang d'une observation
                   function    getRank(example: integer): TTypeRankValue;
                   end;

   //classe de classe RankComputed
   TClassRankComputed = class of TRankComputed;

   //pour le tau de Kendall (corrlation)
   TRankComputedKendallTau = class(TRankComputed)
                             public
                             //la correction  calculer pour chaque groupe -- c'est ce qui sera surcharg dans les classes descendantes (e.g. Kendall)
                             function    computeCorrection(nbExAequo: double): double; override;
                             end;

   //le test de Wilcoxon pour chantillons apparis
   TRankComputedPairedWilcoxon = class(TRankComputed)
                                 public
                                 //la correction  calculer pour chaque groupe -- c'est ce qui sera surcharg dans les classes descendantes (e.g. Kendall)
                                 function    computeCorrection(nbExAequo: double): double; override;
                                 end;

   //le test de Friedman pour K related samples
   TRankComputedKRelatedFriedman = class(TRankComputed)
                                   protected
                                   //gestion trs trange chez Friedman, inclure les ex-aequos de taille 1
                                   procedure   computeRankTied(lstSorted: TExamples); override;
                                   public
                                   //la correction  calculer pour chaque groupe -- c'est ce qui sera surcharg dans les classes descendantes (e.g. Kendall)
                                   function    computeCorrection(nbExAequo: double): double; override;
                                   end;

   //le coefficient de concordance W de Kendall
   //la gestion des ex-aequos revient au standard -- ouf !
   TRankComputedKRelatedWKendall = class(TRankComputed)
                                   public
                                   //la correction  calculer pour chaque groupe -- c'est ce qui sera surcharg dans les classes descendantes (e.g. Kendall)
                                   function    computeCorrection(nbExAequo: double): double; override;
                                   end;



implementation

USES
   Math;

{ TRankComputed }

function TRankComputed.computeCorrection(nbExAequo: double): double;
begin
 //Mann-Whitney, Kruskal-Wallis, Spearman
 result:= power(nbExAequo,3) - nbExAequo;
end;

function TRankComputed.computeRank(lstSorted: TExamples): double;
begin
 //calculer les rangs bruts -- la liste est trie d'office
 self.computeRankNoTied(lstSorted);
 //calculer les rangs moyens et de la correction associe
 self.computeRankTied(lstSorted);
 //renvoyer la valeur de la correction
 result:= FTieCorrection;
end;

procedure TRankComputed.computeRankNoTied(lstSorted: TExamples);
var i: integer;
begin
 //affecter le rang de chaque individu -- la liste est considre trie
 for i:= 1 to lstSorted.Size do
  FRankValue[lstSorted.Number[i]]:= i;
end;

procedure TRankComputed.computeRankTied(lstSorted: TExamples);
var i,j,n,k: integer;
    nbExAequo,sumRank,meanRank: double;
    sortie: boolean;
begin
 n:= lstSorted.Size;
 //corriger avec les rangs moyens
 FTieCorrection:= 0.0;
 i:= 1;
 while (i < n) do
  begin
   //initialisation pour la recherche des ex-aequos
   j:= succ(i);
   nbExAequo:= 1;
   sumRank:= FRankValue[lstSorted.Number[i]];
   sortie:= false;
   //passer en revue vers le haut
   while (j <= n) and not(sortie) do
    begin
     if (FAtt.cValue[lstSorted.Number[i]] < FAtt.cValue[lstSorted.Number[j]])
      then sortie:= true
      else
       begin
        nbExAequo:= nbExAequo + 1;
        sumRank:= sumRank + FRankValue[lstSorted.Number[j]];
        inc(j);
       end;
    end;
   //y a-t-il eu des ex-aequos sur ce passage ?
   if (nbExAequo > 1)
    then
     begin
      dec(j);
      //calculer le rang moyen
      meanRank:= sumRank/nbExAequo;
      //corriger les rangs en affectant le rang moyen sur la portion tudie
      for k:= i to j do FRankValue[lstSorted.Number[k]]:= meanRank;
      //calculer (cumuler) la correction pour les stats.
      FTieCorrection:= FTieCorrection + self.computeCorrection(nbExAequo);
     end;
   //passer  l'observation suivante
   i:= j;
  end;
end;


constructor TRankComputed.create(prmAtt: TAttribute);
begin
 inherited Create();
 //rcupration de l'attribut
 FAtt:= prmAtt;
 //rservation de la place mmoire
 setlength(FRankValue,succ(prmAtt.Size));
end;

destructor TRankComputed.destroy;
begin
 Finalize(FRankValue);
 inherited;
end;

function TRankComputed.getRank(example: integer): TTypeRankValue;
begin
 result:= FRankValue[example];
end;

{ TRankComputedTau }

function TRankComputedKendallTau.computeCorrection(nbExAequo: double): double;
begin
 //Kendall -- Siegel, p.249
 result:= nbExAequo * (nbExAequo - 1.0);
end;

{ TRankComputedPairedWilcoxon }

function TRankComputedPairedWilcoxon.computeCorrection(
  nbExAequo: double): double;
begin
 //Wilcoxon matched-pairs test -- Siegel, p.94
 result:= nbExAequo * (nbExAequo - 1.0) * (nbExAequo + 1.0);
end;

{ TRankComputedKRelatedFriedman }

function TRankComputedKRelatedFriedman.computeCorrection(
  nbExAequo: double): double;
begin
 //Friedman -- Siegel, p.179
 result:= power(nbExAequo,3.0);
end;

procedure TRankComputedKRelatedFriedman.computeRankTied(
  lstSorted: TExamples);
Label FIN;
var i,j,n,k: integer;
    nbExAequo,sumRank,meanRank: double;
    sortie: boolean;
begin
 n:= lstSorted.Size;
 //corriger avec les rangs moyens
 FTieCorrection:= 0.0;
 i:= 1;
 //le dernier s'il n'est pas ex-aequo doit tre compt
 //donc infrieur ou gal
 while (i <= n) do
  begin
   //initialisation pour la recherche des ex-aequos
   j:= succ(i);
   nbExAequo:= 1;
   sumRank:= FRankValue[lstSorted.Number[i]];
   sortie:= false;
   //passer en revue vers le haut
   while (j <= n) and not(sortie) do
    begin
     if (FAtt.cValue[lstSorted.Number[i]] < FAtt.cValue[lstSorted.Number[j]])
      then sortie:= true
      else
       begin
        nbExAequo:= nbExAequo + 1;
        sumRank:= sumRank + FRankValue[lstSorted.Number[j]];
        inc(j);
       end;
    end;
   //calculer (cumuler) la correction pour les stats.
   //mme s'ils sont de taille 1 - Siegel -- p.179
   FTieCorrection:= FTieCorrection + self.computeCorrection(nbExAequo);
   //y a-t-il eu des ex-aequos sur ce passage ?
   if (nbExAequo > 1)
    then
     begin
      //************************************************************************
      //pas de retour en arrire pour j, c'est un peu superflu mme dans la mthode anctre mais a ssurait le coup
      //ici c'est totalement illicite car a xagre le nombre de passage sur les sous-ensembles  1 ex-aequo
      //et gonfle la statistique !!! mais on ne touche pas dans l'anctre !!! surtout pas introduire de nouvelles erreurs !
      //************************************************************************
      //calculer le rang moyen
      meanRank:= sumRank/nbExAequo;
      //corriger les rangs en affectant le rang moyen sur la portion tudie
      //c'est le pred(j) qui assure le calcul sur la bonne portion dans notre cas ici
      for k:= i to pred(j) do FRankValue[lstSorted.Number[k]]:= meanRank;
     end;
   //passer  l'observation suivante
   i:= j;
  end;
end;

{ TRankComputedKRelatedWKendall }

function TRankComputedKRelatedWKendall.computeCorrection(
  nbExAequo: double): double;
begin
 //siegel, p.266
 result:= power(nbExAequo,3.0) - nbExAequo;
end;

end.
