(******************************************************************)
(* UCompClusteringHAC.pas - Copyright (c) 2004 Ricco RAKOTOMALALA *)
(******************************************************************)

{
@abstract(CAH - Stratgie mixte dcrite dans Lebart et al. (2000) - Mthode WARD)
@author(Ricco)
@created(07/03/2004)
}
unit UCompClusteringHAC;

interface

USES
        Contnrs, Forms, Classes, IniFiles,
        UCompDefinition,
        UOperatorDefinition,
        UCompClusteringDefinition,
        UDatasetDefinition, UDatasetExamples,
        UCalcStatDes;

TYPE
       {gnrateur de composant}
        TGenClusHAC = class(TMLGenComp)
                      protected
                      procedure   GenCompInitializations(); override;
                      public
                      function    GetClassMLComponent: TClassMLComponent; override;
                      end;

        {composant CAH}
        TMLCompClusHAC = class(TMLCompClustering)
                         protected
                         function    getClassOperator: TClassOperator; override;
                         function    getGenericAttName(): string; override;
                         end;

        {oprateur CAH}
        TOpClusHAC = class(TOperatorClusteringContinue)
                     protected
                     function    getClassParameter: TClassOperatorParameter; override;
                     function    getClassCalcClustering(): TClassCalcClustering; override;
                     function    CheckAttributes(): boolean; override;
                     function    CheckOthers(): boolean; override;
                     end;

        {paramtre oprateur CAH}
        TOpPrmHAC = class(TOpPrmClustering)
                    protected
                    {Mode de dtermination de la coupure optimale : 0->automatique, 1->manuel}
                    FModeBestSelection: integer;
                    {si choix manuel, nombre de clusters}
                    FNbClusters: integer;
                    {normalisation pour le calcul des distances}
                    FNormalization: integer;
                    function    CreateDlgParameters(): TForm; override;
                    procedure   SetDefaultParameters(); override;
                    public
                    procedure   LoadFromStream(prmStream: TStream); override;
                    procedure   SaveToStream(prmStream: TStream); override;
                    procedure   LoadFromINI(prmSection: string; prmINI: TMemIniFile); override;
                    procedure   SaveToINI(prmSection: string; prmINI: TMemIniFile); override;
                    function    getHTMLParameters(): string; override;
                    property    ModeBestSelection: integer read FModeBestSelection write FModeBestSelection;
                    property    NbClusters: integer read FNbClusters write FNbClusters;
                    property    Normalization: integer read FNormalization write FNormalization;
                    end;

        {dclaration forward}
        TCalcHAC = class;

        {Un sommet de CAH}
        TNodeHAC = class(TObject)
                   private
                   {oprateur de calcul}
                   FCalc: TCalcHAC;
                   {pre}
                   FDad: TNodeHAC;
                   {enfants gauche et droite}
                   FChildL,FChildR: TNodeHAC;
                   {les informations locales}
                   FStats: TLstCalcStatDesContinuous;
                   {numro de sommet pour suivre les fusions}
                   FNumNode: integer;
                   {Distance d'agrgation}
                   FDist: double;
                   public
                   {construire un noeud}
                   constructor  Create(prmCalc: TCalcHAC);
                   {dtruire}
                   destructor   Destroy(); override;
                   {mj des donnes courantes}
                   procedure    Refresh(prmExamples:  TExamples); 
                   {calculer les distances d'agrgation avec un autre noeud}
                   function     MergeDistance(other: TNodeHAC): double;
                   {calculer la distance avec un individu}
                   function     CalcDistance(example: integer): double;
                   end;

        {Structure complte de CAH}
        TSetNodeHAC = class(TObject)
                      private
                      {Calculateur HAC}
                      FCalcHAC: TCalcHAC;  
                      {racine de l'arbre}
                      FRootNode: TNodeHAC;
                      {tous les noeuds - liste propritaire}
                      FLstAllNodes: TObjectList;
                      {les noeuds  fusionner - les noeuds orphelins - liste non propritaire}
                      FLstMergeNodes: TObjectList;
                      {les feuilles de l'arbre - liste non propritaire}
                      FLstLeaves: TObjectList;
                      {Indicateur global du dernier numro de noeud ajout}
                      FLastNumNode: integer;
                      {Tableau de suivi des fusions ascendantes}
                      FLstNewNodes: TObjectList;
                      protected
                      {ajouter un noeud dans la liste initiale}
                      procedure   AddLeaf(node: TNodeHAC);
                      public
                      {cration de la structure}
                      constructor Create(prmCalc: TCalcHAC);
                      {dtruire la structure}
                      destructor  Destroy(); override;
                      {initialiser les noeuds}
                      procedure   InitializeNodes(prmExamples: TExamples);
                      {fusionner deux noeuds et l'ajouter dans la structure}
                      procedure   MergeNodes(nodeL,nodeR: TNodeHAC; prmDist: double);
                      
                      end;

        {tableau dynamique pour la pondration}
        TTabHACAttWeight = array of double;

        {oprateur de calcul CAH}
        TCalcHAC = class(TCalcClusteringContinue)
                   private
                   {inertie totale}
                   FTSS: double;
                   {Indice du palier max.}
                   FIdMaxPalier: integer;
                   {Attribut dfinissant les noeuds de 1er niveau}
                   FAttRef: TAttribute;
                   {Tableau de pondration par la variance}
                   FVariances: TTabHACAttWeight;
                   {tableau des hauteurs des noeuds de l'arbre}
                   FTabIndex: array of double;
                   {tableau des diffrences x-imes pour la dtection de palliers}
                   FTabDifIndex: array of double;
                   {structure d'arbre}
                   FSetNodeHAC:TSetNodeHAC;
                   {les noeuds de la solution adopte}
                   FLstNodesSolution: TObjectList;
                   {initialiser les noeuds}
                   procedure   InitializeNodes(prmExamples: TExamples);
                   protected
                   function    getHTMLClustering(): string; override;
                   procedure   detectPalier();
                   procedure   buildBestNodes();
                   public
                   destructor  Destroy(); override;
                   procedure   BuildClusters(prmExamples: TExamples); override;
                   function    SetClusterExample(example: integer): TTypeDiscrete; override;
                   procedure   FillClusAttDef(); override;
                   property    AttRef: TAttribute read FAttRef;
                   property    SetNodeHAC: TSetNodeHAC read FSetNodeHAC;      
                   end;     

implementation

uses
        SysUtils, UConstConfiguration, ULogFile, UDlgOpClusHAC,
        UDlgBaseOperatorParameter;

const
        //nombre de feuilles min. pour commencer une CAH mixte
        MIN_LEAF_HAC = 5;

        

{ TGenClusHAC }

procedure TGenClusHAC.GenCompInitializations;
begin
 FMLComp:= mlcClustering;
end;

function TGenClusHAC.GetClassMLComponent: TClassMLComponent;
begin
 result:= TMLCompClusHAC;
end;

{ TMLCompClusHAC }

function TMLCompClusHAC.getClassOperator: TClassOperator;
begin
 result:= TOpClusHAC;
end;

function TMLCompClusHAC.getGenericAttName: string;
begin
 result:= 'HAC';
end;

{ TOpClusHAC }

function TOpClusHAC.CheckAttributes: boolean;
begin
 result:= inherited CheckAttributes()
          and (Targets.Count=1)
          and (Targets.isAllCategory(caDiscrete));
end;

function TOpClusHAC.CheckOthers: boolean;
var ok: boolean;
begin
 //vrifier le nombre min. de noeuds feuilles
 ok:= (self.Targets.Attribute[0].nbValues >= MIN_LEAF_HAC);
 //ok ?
 result:= ok;
end;

function TOpClusHAC.getClassCalcClustering: TClassCalcClustering;
begin
 result:= TCalcHAC;
end;

function TOpClusHAC.getClassParameter: TClassOperatorParameter;
begin
 result:= TOpPrmHAC;
end;

{ TCalcHAC }

procedure TCalcHAC.buildBestNodes;
var i: integer;
    node: TNodeHAC;
begin
 FLstNodesSolution:= TObjectList.Create(FALSE);
 FLstNodesSolution.Add(FSetNodeHAC.FRootNode);
 for i:= 1 to FIdMaxPalier+1 do
  begin
   node:= FSetNodeHAC.FLstNewNodes.Items[FSetNodeHAC.FLstNewNodes.Count-i] as TNodeHAC;
   FLstNodesSolution.Extract(node);
   FLstNodesSolution.Add(node.FChildL);
   FLstNodesSolution.Add(node.FChildR);
  end;
end;

procedure TCalcHAC.BuildClusters(prmExamples: TExamples);
var i,j,i_min,j_min: integer;
    nodeI,nodeJ: TNodeHAC;
    d,min_d: double;
begin
 inherited BuildClusters(prmExamples);
 //initialiser les noeuds de la structure
 self.InitializeNodes(prmExamples);
 //fusionner en remontant jusqu' la racine de l'arbre
 while (FSetNodeHAC.FLstMergeNodes.Count>1) do
  begin
   i_min:= -1;
   j_min:= -1;
   min_d:= 1.0e308;
   //tester le croisement
   for i:= 0 to FSetNodeHAC.FLstMergeNodes.Count-2 do
    begin
     nodeI:= FSetNodeHAC.FLstMergeNodes.Items[i] as TNodeHAC;
     for j:= succ(i) to FSetNodeHAC.FLstMergeNodes.Count-1 do
      begin
       nodeJ:= FSetNodeHAC.FLstMergeNodes.Items[j] as TNodeHAC;
       d:= nodeI.MergeDistance(nodeJ)/(1.0*prmExamples.Size);
       if (d<min_d)
        then
         begin
          i_min:= i;
          j_min:= j;
          min_d:= d;
         end;
      end;
    end;
   //rcuprer les plus proches et fusionner
   nodeI:= FSetNodeHAC.FLstMergeNodes.Items[i_min] as TNodeHAC;
   nodeJ:= FSetNodeHAC.FLstMergeNodes.Items[j_min] as TNodeHAC;
   FSetNodeHAC.MergeNodes(nodeI,nodeJ,min_d);
  end;
 FSetNodeHAC.FRootNode:= FSetNodeHAC.FLstMergeNodes.Items[0] as TNodeHAC;
 TraceLog.WriteToLogFile('CAH -> examples sur la racine : '+IntToStr(FSetNodeHAC.FRootNode.FStats.NbExamples));
 //structure arbre ok - rechercher maintenant le meilleur cluster
 self.detectPalier();
 //forcer la solution si demande par l'utilisateur
 if ((self.PrmCalc as TOpPrmHAC).FModeBestSelection = 1)
  then self.FIdMaxPalier:= (self.PrmCalc as TOpPrmHAC).FNbClusters-2;
 //dtecter les noeuds du clustering dfinitif
 self.buildBestNodes();
end;

destructor TCalcHAC.Destroy;
begin
 SetLength(FTabDifIndex,0);
 SetLength(FVariances,0);
 if assigned(FLstNodesSolution)
  then FLstNodesSolution.Free;
 if assigned(FSetNodeHAC)
  then FSetNodeHAC.Free;
 inherited;
end;

procedure TCalcHAC.detectPalier;
var i: integer;
    node: TNodeHAC;
    max: double;
begin
 setLength(FTabDifIndex,FSetNodeHAC.FLstNewNodes.Count);
 setLength(FTabIndex,FSetNodeHAC.FLstNewNodes.Count);
 //tableau des distances d'agrgation
 for i:= 0 to pred(FSetNodeHAC.FLstNewNodes.Count) do
  begin
   node:= FSetNodeHAC.FLstNewNodes.Items[pred(FSetNodeHAC.FLstNewNodes.Count)-i] as TNodeHAC;
   FTabDifIndex[i]:= node.FDist;
   FTabIndex[i]:= node.FDist;
  end;
 //calcul des diffrences 1res
 for i:= low(FTabDifIndex) to high(FTabDifIndex)-1 do
  FTabDifIndex[i]:= FTabDifIndex[i]-FTabDifIndex[succ(i)];
 //on peut dtecter le palier
 FIdMaxPalier:= -1;
 max:= -1.0e308;
 for i:= low(FTabDifIndex) to high(FTabDifIndex)-1 do
  begin
   if (i>0) and (FTabDifIndex[i]>max)
    then
     begin
      max:= FTabDifIndex[i];
      FIdMaxPalier:= i;
     end;
  end;
end;

procedure TCalcHAC.FillClusAttDef;
var k: integer;
begin
 FAttClus.LstValues.clear;
 for k:= 1 to FIdMaxPalier+2 do
  FAttClus.LstValues.getValue('c_hac_'+IntToStr(k));
end;

function TCalcHAC.getHTMLClustering: string;
var s: string;
    sIdMax: string;
    cumul: double;
    i: integer;
    node: TNodeHAC;
begin
 //affichage de la structure de l'arbre
 s:= '<P><H3>Tree structure</H3><BR>';
 s:= s+HTML_HEADER_TABLE_RESULT;
 s:= s+format('%s <th colspan="4">First level clusters</th></tr>',[HTML_TABLE_COLOR_HEADER_GRAY]);
 s:= s+format('%s <td>Attribute ref.</td><td align="right" colspan="3">%s</td></tr>',[HTML_TABLE_COLOR_DATA_GRAY,FAttRef.Name]);
 s:= s+format('%s <td align="left">Tree leaves</td><td align="right" colspan="3">%d</td></tr>',[HTML_TABLE_COLOR_DATA_GRAY,FSetNodeHAC.FLstLeaves.Count]);
 s:= s+format('%s <th>Node</th><th>Size</th><th>Childs</th><th>Index</th></tr>',[HTML_TABLE_COLOR_HEADER_GRAY]);
 for i:= 0 to pred(FSetNodeHAC.FLstLeaves.Count) do
  begin
   node:= FSetNodeHAC.FLstLeaves.Items[i] as TNodeHAC;
   s:= s+format('%s <td align="center">%d</td><td align="right">%d</td><td align="center">-</td><td align="center">-</td></tr>',[HTML_TABLE_COLOR_DATA_GRAY,node.FNumNode,node.FStats.NbExamples]);
  end;
 for i:= 0 to pred(FSetNodeHAC.FLstNewNodes.Count) do
  begin
   node:= FSetNodeHAC.FLstNewNodes.Items[i] as TNodeHAC;
   s:= s+format('%s <td align="center">%d</td><td align="right">%d</td><td align="right">(%d,%d)</td><td align="right">%8.4f</td></tr>',[HTML_TABLE_COLOR_DATA_GRAY,node.FNumNode,node.FStats.NbExamples,node.FChildL.FNumNode,node.FChildR.FNumNode,node.FDist]);
  end;
 s:= s+'</table>';
 //affichage de la slection du meilleur cluster
 s:= s+'<P><H3>Best cluster selection</H3><BR>';
 s:= s+HTML_HEADER_TABLE_RESULT;
 s:= s+format('%s <th>Clusters</th><th>BSS ratio</th><th>Gap</th></tr>',[HTML_TABLE_COLOR_HEADER_GRAY]);
 cumul:= 0.0;
 for i:= 0 to FSetNodeHAC.FLstNewNodes.Count-2 do
  begin
   //indiquer la bonne solution
   if (i=FIdMaxPalier)
    then sIdMax:= HTML_TABLE_COLOR_DATA_GREEN
    else sIdMAx:= HTML_TABLE_COLOR_DATA_GRAY;
   //cumul des inerties expliques
   cumul:= cumul+FTabIndex[i];
   //envoyer la sauce
   s:= s+format('%s <td>%d</td><td align="right">%6.4f</td><td align="right">%6.4f</td></tr>',
                [sIdMax,i+2,cumul/FTSS,FTabDifIndex[i]]);
  end;
 s:= s+'</table>';
 //zoo...
 result:= s;
end;

procedure TCalcHAC.InitializeNodes(prmExamples: TExamples);
var j: integer;
begin
 FAttRef:= NIL;
 if (self.Targets.Count>0)
  then FAttRef:= self.Targets.Attribute[0] as TAttribute;
 if not(assigned(FAttRef))
  then RAISE Exception.Create('no attribute reference for HAC')
  else
   begin
    setLength(FVariances,self.Inputs.Count);
    //calculer la variance utilise pour la normalisation
    if ((self.PrmCalc as TOpPrmHAC).FNormalization = 1)
     then
      begin
       For j:= 0 to pred(self.Inputs.Count) do
        FVariances[j]:= TCalcStatDesContinuous(self.StatsInputs.Stat(j)).Variance;
       //donnes normalise, l'inertie totale est gale au nombre de variables
       FTSS:= self.Inputs.Count;
      end
     else
      begin
       //non normalises - inertie totale = somme des variances avec un distance euclidienne simple
       FTSS:= 0.0;
       for j:= 0 to pred(self.Inputs.Count) do
        begin
         FVariances[j]:= 1.0;
         FTSS:= FTSS+TCalcStatDesContinuous(self.StatsInputs.Stat(j)).Variance; 
        end;
      end;
   end;
 //construire le 1er niveau de la structure d'arbre
 FSetNodeHAC:= TSetNodeHAC.Create(self);
 FSetNodeHAC.InitializeNodes(prmExamples);
 FSetNodeHAC.FLastNumNode:= FSetNodeHAC.FLstLeaves.Count;
end;

function TCalcHAC.SetClusterExample(example: integer): TTypeDiscrete;
var i,i_min: integer;
    d,min_d: double;
    node: TNodeHAC;
begin
 i_min:= -1;
 min_d:= 1.0e308;
 //trouver le noeud le plus proche
 for i:= 0 to pred(self.FLstNodesSolution.Count) do
  begin
   node:= self.FLstNodesSolution.Items[i] as TNodeHAC;
   d:= node.CalcDistance(example);
   if (d<min_d)
    then
     begin
      min_d:= d;
      i_min:= i;
     end;
  end;
 result:= succ(i_min);
end;

{ TSetNodeHAC }

procedure TSetNodeHAC.AddLeaf(node: TNodeHAC);
begin
 FLstAllNodes.Add(node);
 FLstMergeNodes.Add(node);
 FLstLeaves.Add(node);
 node.FNumNode:= FLstLeaves.Count;
 node.FDist:= 0.0;
end;

constructor TSetNodeHAC.Create(prmCalc: TCalcHAC);
begin
 inherited Create();
 FCalcHAC:= prmCalc;
 //les listes internes
 FLstAllNodes:= TObjectList.Create(TRUE);
 FLstMergeNodes:= TObjectList.Create(FALSE);
 FLstLeaves:= TObjectList.Create(FALSE);
 FLstNewNodes:= TObjectList.Create(FALSE);
end;

destructor TSetNodeHAC.Destroy;
begin
 FLstNewNodes.Free;
 FLstLeaves.Free;
 FLstMergeNodes.Free;
 FLstAllNodes.Free;
 inherited Destroy();
end;

procedure TSetNodeHAC.InitializeNodes(prmExamples: TExamples);
var dispEx: TObjectList;
    ex: TExamples;
    j: integer;
    node: TNodeHAC;
begin
 //lister les individus par cluster de 1er niveau
 dispEx:= prmExamples.DispatchExamples(FCalcHAC.AttRef);
 //construire les noeuds
 for j:= 0 to pred(dispEx.Count) do
  begin
   ex:= dispEx.Items[j] as TExamples;
   if (ex.Size>0)
    then
     begin
      //crer
      node:= TNodeHAC.Create(FCalcHAC);
      node.Refresh(ex);
      //ajouter
      self.AddLeaf(node);
     end;
  end;
end;

procedure TSetNodeHAC.MergeNodes(nodeL, nodeR: TNodeHAC; prmDist: double);
var node: TNodeHAC;
    j: integer;
    st,stL,stR: TCalcStatDesContinuous;
    min,max,sum,sqsum: double;
    nb: integer;
begin
 node:= TNodeHAC.Create(self.FCalcHAC);
 //Numro du dernier sommet ajout et sa hauteur
 inc(self.FLastNumNode);
 node.FNumNode:= self.FLastNumNode;
 node.FDist:= prmDist;
 //affecter les donnes simples pour chaque variable j
 for j:= 0 to pred(node.FStats.Count) do
  begin
   //brancher
   st:= node.FStats.Stat(j) as TCalcStatDesContinuous;
   stL:= nodeL.FStats.Stat(j) as TCalcStatDesContinuous;
   stR:= nodeR.FStats.Stat(j) as TCalcStatDesContinuous;
   //calculer les stats locales
   if (stL.Min<stR.Min)
    then min:= stL.Min
    else min:= stR.Min;
   if (stL.Max>stR.Max)
    then max:= stL.Max
    else max:= stR.Max;
   sum:= stL.Sum+stR.Sum;
   sqsum:= stL.SumSquare+stR.SumSquare;
   nb:= stL.NbExamples+stR.NbExamples;
   st.SetValues(min,max,sum,sqsum,nb);
  end;
 //mj des listes internes et des branchements
 node.FChildL:= nodeL;
 node.FChildR:= nodeR;
 nodeL.FDad:= node;
 nodeR.FDad:= node;
 self.FLstAllNodes.Add(node);
 self.FLstNewNodes.Add(node);
 self.FLstMergeNodes.Add(node);
 self.FLstMergeNodes.Remove(nodeL);
 self.FLstMergeNodes.Remove(nodeR);
end;

{ TNodeHAC }

function TNodeHAC.CalcDistance(example: integer): double;
var s,d: double;
    j: integer;
begin
 s:= 0.0;
 for j:= 0 to pred(self.FStats.Count) do
  begin
   d:= SQR(self.FCalc.Inputs.Attribute[j].cValue[example]-(self.FStats.Stat(j) as TCalcStatDesContinuous).Average);
   d:= d/self.FCalc.FVariances[j];
   s:= s+d;
  end;
 result:= s;
end;

constructor TNodeHAC.Create(prmCalc: TCalcHAC);
begin
 inherited Create();
 FCalc:= prmCalc;
 FDad:= NIL;
 FChildL:= NIL;
 FChildR:= NIL;
 FStats:= TLstCalcStatDesContinuous.Create(FCalc.Inputs,NIL);
end;

destructor TNodeHAC.Destroy;
begin
 FStats.Free;
 inherited Destroy();
end;

function TNodeHAC.MergeDistance(other: TNodeHAC): double;
var s,d: double;
    j: integer;
begin
 s:= 0.0;
 //distance brute
 for j:= 0 to pred(self.FStats.Count) do
  begin
   d:= SQR((self.FStats.Stat(j) as TCalcStatDesContinuous).Average-(other.FStats.Stat(j) as TCalcStatDesContinuous).Average);
   d:= d/self.FCalc.FVariances[j];
   s:= s+d;
  end;
 //pondration par le poids des noeuds
 result:= (1.0*self.FStats.NbExamples*other.FStats.NbExamples)/(1.0*(self.FStats.NbExamples+other.FStats.NbExamples))*s;
end;

procedure TNodeHAC.Refresh(prmExamples: TExamples);
begin
 FStats.RefreshStat(prmExamples);
end;

{ TOpPrmHAC }

function TOpPrmHAC.CreateDlgParameters: TForm;
begin
 result:= TDlgOpPrmClusHAC.CreateFromOpPrm(self);
end;

function TOpPrmHAC.getHTMLParameters: string;
begin

end;

procedure TOpPrmHAC.LoadFromINI(prmSection: string; prmINI: TMemIniFile);
begin
 inherited;
 FModeBestSelection:= prmINI.ReadInteger(prmSection,'clus_selection',FModeBestSelection);
 FNbClusters:= prmINI.ReadInteger(prmSection,'user_nb_clus',FNbClusters);
 FNormalization:= prmINI.ReadInteger(prmSection,'normalization',FNormalization);
end;

procedure TOpPrmHAC.LoadFromStream(prmStream: TStream);
begin
 inherited;
 prmStream.ReadBuffer(FModeBestSelection,sizeof(FModeBestSelection));
 prmStream.ReadBuffer(FNbClusters,sizeof(FNbClusters));
 prmStream.ReadBuffer(FNormalization,sizeof(FNormalization));
end;

procedure TOpPrmHAC.SaveToINI(prmSection: string; prmINI: TMemIniFile);
begin
 inherited;
 prmINI.WriteInteger(prmSection,'clus_selection',FModeBestSelection);
 prmINI.WriteInteger(prmSection,'user_nb_clus',FNbClusters);
 prmINI.WriteInteger(prmSection,'normalization',FNormalization);
end;

procedure TOpPrmHAC.SaveToStream(prmStream: TStream);
begin
 inherited;
 prmStream.WriteBuffer(FModeBestSelection,sizeof(FModeBestSelection));
 prmStream.WriteBuffer(FNbClusters,sizeof(FNbClusters));
 prmStream.WriteBuffer(FNormalization,sizeof(FNormalization));
end;

procedure TOpPrmHAC.SetDefaultParameters;
begin
 inherited SetDefaultParameters();
 FModeBestSelection:= 0;
 FNbClusters:= 3;
 FNormalization:= 1;
end;

initialization
 RegisterClass(TGenClusHAC);
end.
