(*********************************************************************)
(* UCompClusteringKMeans.pas - Copyright (c) 2004 Ricco RAKOTOMALALA *)
(*********************************************************************)

{
@abstract(Implmentation des K-Means -- Trs simple)
@author(Ricco)
@created(12/01/2004)

Elments cls dans l'algo implment : on effectue plusieurs essais avec (1) des points de dparts
alatoires diffrents et (2) une perturbation alatoire de l'ordre des individus  passer.

Pour chaque essai, le nombre d'itrations max est limit pour viter l'explosion des calculs,
on remarquera que McQueen converge nettement plus vite que Forgy.

Si la classif est effectue  partir d'axes factoriels, il est impratif de ne pas normaliser la distance
utilise, la variance indique clairement le poids de la variable (l'axe) utilise.
}
unit UCompClusteringKMeans;

interface

USES
        Forms,Classes,
        IniFiles,
        UCompDefinition,
        UOperatorDefinition,
        UCompClusteringDefinition,
        UDatasetDefinition,
        UDatasetExamples,
        UCalcStatDes,
        contnrs;

TYPE
        {gnrateur de rgression multiple}
        TGenClusKMeans = class(TMLGenComp)
                          protected
                          procedure   GenCompInitializations(); override;
                          public
                          function    GetClassMLComponent: TClassMLComponent; override;
                          end;

        {composant K-Means}
        TMLCompClusKMeans = class(TMLCompClustering)
                            protected
                            function    getClassOperator: TClassOperator; override;
                            function    getGenericAttName(): string; override;    
                            end;    

        {oprateur K-Means}
        TOpClusKMeans = class(TOperatorClusteringContinue)
                        protected
                        function    getClassParameter: TClassOperatorParameter; override;
                        function    getClassCalcClustering(): TClassCalcClustering; override;
                        end;

        {paramtre oprateur K-Means}
        TOpPrmClusKMeans = class(TOpPrmClustering)
                           private
                           {nombre de clusters dsir}
                           FNbClusters: integer;
                           {normalisation dans le calcul de la distance,
                           0 -> aucune,
                           1 -> pondr par la variance}
                           FNormalization: integer;
                           {mode de mise  jour des moyennes,
                           0 ->  une fois tous les individus passs (Forgy),
                           1 -> au fur et  mesure de l'ajout d'un individu (McQueen)}
                           FAvgComputation: integer;
                           {nombre d'itrations maximum sur une session}
                           FMaxIter: integer;
                           {nombre de sessions afin de trouver le partitionnement le plus adquat}
                           FNbTrials: integer;
                           protected
                           function    CreateDlgParameters(): TForm; override;
                           procedure   SetDefaultParameters(); override;
                           public
                           procedure   LoadFromStream(prmStream: TStream); override;
                           procedure   SaveToStream(prmStream: TStream); override;
                           procedure   LoadFromINI(prmSection: string; prmINI: TMemIniFile); override;
                           procedure   SaveToINI(prmSection: string; prmINI: TMemIniFile); override;
                           function    getHTMLParameters(): string; override;
                           property NbClusters: integer read FNbClusters write FNbClusters;
                           property Normalization: integer read FNormalization write FNormalization;
                           property AvgComputation: integer read FAvgComputation write FAvgComputation;
                           property MaxIter: integer read FMaxIter write FMaxIter;
                           property NbTrials: integer read FNBTrials write FNBTrials; 
                           end;

        {tableau de coordonnes d'un cluster}
        TTabClusterCoordKMeans = array of double;

        {un cluster de K-Means}
        TOneClusterKMeans = class
                            private
                            {centre de gravit du cluster}
                            FCurAvg: TTabClusterCoordKMeans;
                            {somme des valeurs des exemples ajouts}
                            FSumValues: TTabClusterCoordKMeans;
                            {tableau des variances}
                            FVariances: TTabClusterCoordKMeans;
                            {nombre d'individus dans le cluster}
                            FNbExamples: double;
                            {paramtre de l'oprateur}
                            FPrmAlgo: TOpPrmClusKMeans;
                            {les Targets}
                            FTargets: TLstAttributes;
                            {initialiser le centre de gravit}
                            procedure   InitializeFirstExample(example: integer);
                            public
                            constructor Create(prmTargets: TLstAttributes; prmAlgo: TOpPrmClusKMeans; prmStats: TLstCalcStatDesContinuous; idFirstExample: integer);
                            destructor  Destroy; override;
                            function  CalcDistance(example: integer): double;
                            procedure BeginUpdate();
                            procedure AddExample(example: integer);
                            procedure EndUpdate();
                            property  NbExamples: double read FNbExamples;
                            property  CurAvg: TTabClusterCoordKMeans read FCurAvg;
                            end;

        {Structure d'un cluster de K-Means - liste de TOneClusterKMeans}
        TStrucClustersKMeans = class
                               private
                               {ratio de l'inertie explique}
                               FExplainedRatio: double;
                               {nombre d'itrations pour obtenir ce rsultat}
                               FNbIterations: integer;
                               {liste interne des clusters}
                               FLstClusters: TObjectList;
                               {nombre de clusters}
                               function    GetNbClusters(): integer;
                               {accder au cluster ni}
                               function    GetCluster(i: integer): TOneClusterKMeans;
                               public
                               {construire la liste}
                               constructor Create();
                               {dtruire la liste}
                               destructor  Destroy; override;
                               {ajouter un cluster dans la liste}
                               procedure   AddCluster(prmClus: TOneClusterKMeans);
                               {accs  un cluster}
                               property    Cluster[i: integer]: TOneClusterKMeans read GetCluster;
                               {nombre de clusters}
                               property    Count: integer read GetNbClusters;
                               {ratio inertie explique}
                               property    ExplainedRatio: double read FExplainedRatio write FExplainedRatio;
                               {nombre d'irations}
                               property    NbIterations: integer read FNbIterations write FNbIterations;
                               end;


        {calculateur K-Means}
        TCalcClusKMeans = class(TCalcClusteringContinue)
                          private
                          {clusters de la solution courante}
                          FCurClusters: TStrucClustersKMeans;
                          {ensemble des solutions envisages}
                          FLstClustersSolutions: TObjectList;
                          {identifant de cluster pour chaque individu}
                          FIdClusters: array of integer;
                          {nombre d'itrations effectues}
                          FNbIter: integer;
                          {Inertie totale}
                          FGlobalInertia: double;
                          {tableau des inerties intra}
                          FWSS: array of double;
                          {le dernier ratio calcul}
                          FLastRatio: double;
                          {n de la meilleure solution}
                          FIdMaxRatio: integer;
                          {ajouter la solution courante}
                          procedure   AddSolution();
                          {preparer les tableaux internes}
                          procedure   InitializeClusters(prmExamples: TExamples);
                          {affecter un exemple  un cluster, renvoie TRUE si une nouvelle affecation a t effectue, donc pas de stabilit}
                          function    AffectExample(i,example: integer): boolean;
                          {commencer une nouvelle session}
                          procedure   BeginUpdate();
                          {finaliser la session}
                          procedure   EndUpdate();
                          {calculer l'inertie totale, n est la taille de l'chantillon}
                          procedure   ComputeGlobalInertia(n: double);
                          protected
                          function    getHTMLClustering(): string; override;
                          public
                          destructor  Destroy; override;
                          procedure   BuildClusters(prmExamples: TExamples); override;
                          procedure   FillClusAttDef(); override;
                          function    SetClusterExample(example: integer): TTypeDiscrete; override;
                          end;

implementation

uses
        Sysutils, UConstConfiguration, UDlgOpClusKMeans, UStringsResources,
  UCalcRndGenerator;

const
        // mettre en paramtre ?
        thresold_epsilon_ratio = 0.001;

{ TGenClusKMeans }

procedure TGenClusKMeans.GenCompInitializations;
begin
 FMLComp:= mlcClustering;
 //FMLNumIcon:= 24;
 //FMLCompName:= str_comp_name_k_means;
 //FMLBitmapFileName:= 'MLClusteringKMeans.bmp';
end;

function TGenClusKMeans.GetClassMLComponent: TClassMLComponent;
begin
 result:= TMLCompClusKMeans;
end;

{ TOpClusKMeans }

function TOpClusKMeans.getClassCalcClustering: TClassCalcClustering;
begin
 result:= TCalcClusKMeans;
end;

function TOpClusKMeans.getClassParameter: TClassOperatorParameter;
begin
 result:= TOpPrmClusKMeans;
end;

{ TMLCompClusKMeans }

function TMLCompClusKMeans.getClassOperator: TClassOperator;
begin
 result:= TOpClusKMeans;
end;

function TMLCompClusKMeans.getGenericAttName: string;
begin
 result:= 'KMeans';
end;

{ TOneClusterKMeans }

procedure TOneClusterKMeans.AddExample(example: integer);
var j: integer;
begin
 FNbExamples:= FNbExamples+1.0;
 for j:= 0 to pred(FTargets.Count) do
  FSumValues[j]:= FSumValues[j]+FTargets.Attribute[j].cValue[example];
 //mise  jour du centre de gravit ?
 if (FPrmAlgo.FAvgComputation = 1)
  then
   begin
    for j:= 0 to pred(FTargets.Count) do
     FCurAvg[j]:= FSumValues[j]/FNbExamples;
   end;
end;

procedure TOneClusterKMeans.BeginUpdate;
var i: integer;
begin
 FNbExamples:= 0.0;
 for i:= 0 to pred(FTargets.Count) do
  FSumValues[i]:= 0.0;
end;

function TOneClusterKMeans.CalcDistance(example: integer): double;
var s: double;
    j: integer;
begin
 s:= 0.0;
 for j:= 0 to pred(FTargets.Count) do
  s:= s+1.0/FVariances[j]*SQR(FCurAvg[j]-FTargets.Attribute[j].cValue[example]);
 result:= s;
end;

constructor TOneClusterKMeans.Create(prmTargets: TLstAttributes;
  prmAlgo: TOpPrmClusKMeans; prmStats: TLstCalcStatDesContinuous;
  idFirstExample: integer);
var j: integer;
begin
 inherited Create();
 FPrmAlgo:= prmAlgo;
 FTargets:= prmTargets;
 //les tableaux locaux
 FNbExamples:= 0.0;
 SetLength(FCurAvg,FTargets.Count);
 SetLength(FSumValues,FTargets.Count);
 SetLength(FVariances,FTargets.Count);
 case FprmAlgo.Normalization of
  1 : begin
       for j:= 0 to pred(FTargets.Count) do
        if (TCalcStatDesContinuous(prmStats.Stat(j)).Variance>0)
         then FVariances[j]:= TCalcStatDesContinuous(prmStats.Stat(j)).Variance
         //de toute manire elle ne psera jamais dans les calculs
         else FVariances[j]:= 1.0;
      end
  else
    begin
     for j:= 0 to pred(FTargets.Count) do
      FVariances[j]:= 1.0;
    end;
 end;
 //initialiser le centre de gravit
 self.InitializeFirstExample(idFirstExample);
end;

destructor TOneClusterKMeans.Destroy;
begin
 SetLength(FCurAvg,0);
 SetLength(FSumValues,0);
 SetLength(FVariances,0);
 inherited;
end;

procedure TOneClusterKMeans.EndUpdate;
var j: integer;
begin
 //si c'est pas le cas, il y a un srieux problme !!!
 if (FNbExamples>0)
  then
   begin
    for j:= 0 to pred(FTargets.Count) do
     FCurAvg[j]:= FSumValues[j]/FNbExamples;
   end;
end;

procedure TOneClusterKMeans.InitializeFirstExample(example: integer);
var j: integer;
begin
 for j:= 0 to pred(FTargets.Count) do
  FCurAvg[j]:= FTargets.Attribute[j].cValue[example];
end;

{ TCalcClusKMeans }

procedure TCalcClusKMeans.AddSolution();
begin
 //rcuprer les caractristiques de la solution
 FCurClusters.ExplainedRatio:= FLastRatio;
 FCurClusters.NbIterations:= FNbIter;
 FLstClustersSolutions.Add(FCurClusters);
 FCurClusters:= NIL;
end;

function TCalcClusKMeans.AffectExample(i,example: integer): boolean;
var idK: integer;
    d: double;
    clus: TOneClusterKMeans;
begin
 result:= FALSE;
 idK:= self.SetClusterExample(example);
 //si ok, on a trouv, ce qui devrait toujours arriver
 if (idK>0)
  then
   begin
    clus:= FCurClusters.Cluster[pred(idK)];
    //la distance entre le cluster et le centre de gravit
    d:= clus.CalcDistance(example);
    //mettre  jour le cluster, y compris son centre de gravit dans le cas des K-Means
    clus.AddExample(example);
    //ajouter pour l'inertie intra classes
    FWSS[pred(idK)]:= FWSS[pred(idK)]+d;
    //puis tester si modif de l'attribution
    if (FIdClusters[i]<>idK)
     then
      begin
       FIdClusters[i]:= idK;
       //nouvelle affectation
       result:= TRUE;
      end;
   end;
end;

procedure TCalcClusKMeans.BeginUpdate;
var k: integer;
begin
 for k:= 0 to pred(FCurClusters.Count) do
  begin
   FCurClusters.Cluster[k].BeginUpdate;
   FWSS[k]:= 0.0;
  end;
end;

procedure TCalcClusKMeans.BuildClusters(prmExamples: TExamples);
var prm: TOpPrmClusKMeans;
    NewAffectExist: boolean;
    i,t: integer;
    exRandomized: TExamples;
    oldRatio,maxRatio: double;
begin
 //initialiser la liste des solutions
 FLstClustersSolutions:= TObjectList.Create(TRUE);

 //calcul maintenant des caractristiques de l'ensemble des individus
 inherited BuildClusters(prmExamples);
 //caluler l'inertie totale
 self.ComputeGlobalInertia(1.0*prmExamples.Size);
 //rcupration des paramtres typs
 prm:= PrmCalc as TOpPrmClusKMeans;
 //tester plusieurs fois pour obtenir une bonne solution
 maxRatio:= -1.0e308;
 FIdMaxRatio:= -1;
 //crer la liste de travail
 exRandomized:= TExamples.Create(prmExamples.Size);
 exRandomized.Copy(prmExamples);
 for t:= 1 to prm.NbTrials do
  begin
   //perturber l'ordre d'arrive pour amoindrir la dpendance au fichier
   exRandomized.procRandomizeExamples(PrmCalc.ModeRndGenerator);
   //crer la structure de clusters courante
   self.InitializeClusters(exRandomized);
   //lancer le calcul
   FNbIter:= 0;
   FLastRatio:= -1.0e308;
   REPEAT
    inc(FNbIter);
    NewAffectExist:= FALSE;
    //commencer la session
    self.BeginUpdate();
    //pour chaque exemple
    for i:= 1 to exRandomized.Size do
     //l'ordre dans la condition est trs important
     NewAffectExist:= self.AffectExample(i,exRandomized.Number[i]) OR NewAffectExist;
    //finaliser la session
    oldRatio:= FLastRatio;
    self.EndUpdate();
   UNTIL NOT(NewAffectExist) OR (FNbIter>=prm.MaxIter) OR ((FLastRatio-oldRatio)<thresold_epsilon_ratio);
   //ajouter la solution dans la liste des solutions
   Self.AddSolution();
   //tester si cette solution est optimale
   if (FLastRatio>maxRatio)
    then
     begin
      maxRatio:= FLastRatio;
      FIdMaxRatio:= t;
     end;
  end;
 //supprimer la liste des individus randomiss
 FreeAndNil(exRandomized);
 //rcuprer alors la meilleure solution
 FCurClusters:= FLstClustersSolutions.Items[pred(FIdMaxRatio)] as TStrucClustersKMeans;
end;

procedure TCalcClusKMeans.ComputeGlobalInertia(n: double);
var prm: TOpPrmClusKMeans;
    j: integer;
begin
 prm:= PrmCalc as TOpPrmClusKMeans;
 if (prm.Normalization = 1)
  //pondre par l'inverse de la variance => inertie totale = dimension x n
  then FGlobalInertia:= n*FInputs.Count
  //non pondre => somme des TSS, ou n x somme des variances
  else
   Begin
    FGlobalInertia:= 0.0;
    for j:= 0 to pred(StatsInputs.Count) do
     FGlobalInertia:= FGlobalInertia+TCalcStatDesContinuous(StatsInputs.Stat(j)).TSS;
   end;
end;

destructor TCalcClusKMeans.Destroy;
begin
 SetLength(FIdClusters,0);
 SetLength(FWSS,0);
 FLstClustersSolutions.Free;
 inherited destroy;
end;

procedure TCalcClusKMeans.EndUpdate;
var k: integer;
    wss,ratio: double;
begin
 wss:= 0.0;
 for k:= 0 to pred(FCurClusters.Count) do
  begin
   FCurClusters.Cluster[k].EndUpdate();
   wss:= wss+FWSS[k];
  end;
 //ratio explained
 ratio:= (self.FGlobalInertia-wss)/Self.FGlobalInertia;
 FLastRatio:= ratio;
end;

procedure TCalcClusKMeans.FillClusAttDef;
var k: integer;
begin
 FattClus.LstValues.clear;
 for k:= 1 to (PrmCalc as TOpPrmClusKMeans).NbClusters do
  //le fonctionnement particulier de getValue permet cette manip.
  FAttClus.LstValues.getValue('c_kmeans_'+IntToStr(k));
end;

function TCalcClusKMeans.getHTMLClustering: string;
var s: string;
    i: integer;
    strucClus: TStrucClustersKMeans;
    prm: TOpPrmClusKMeans;
begin
 //paramtre de l'algo
 prm:= PrmCalc as TOpPrmClusKMeans;
 s:= '<P><H3>Ratio explained evolution</H3>';
 s:= s+HTML_HEADER_TABLE_RESULT;
 s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>Number of trials</TD><TD align=right>%d</TD></TR>',[prm.NbTrials]);
 s:= s+HTML_TABLE_COLOR_HEADER_GRAY+'<TH>Trial</TH><TH>Ratio explained</TH></TR>';
 for i:= 1 to prm.NbTrials do
  begin
   strucClus:= FLstClustersSolutions.Items[pred(i)] as TStrucClustersKMeans;
   if (i<>FIdMaxRatio)
    then s:= s+HTML_TABLE_COLOR_DATA_GRAY
    else s:= s+HTML_TABLE_COLOR_DATA_GREEN;
   s:= s+format('<TD>%d</TD><TD align=right>%.6f</TD></TR>',[i,strucClus.ExplainedRatio]);
  end;
 s:= s+'</table>';
 result:= s;
end;

procedure TCalcClusKMeans.InitializeClusters(prmExamples: TExamples);
var k,idExample: integer;
    prm: TOpPrmClusKMeans;
    clus: TOneClusterKMeans;
begin
 prm:= PrmCalc as TOpPrmClusKMeans;
 SetLength(FIdClusters,succ(prmExamples.Size));//tous  zro systmatiquement
 FCurClusters:= TStrucClustersKMeans.Create();
 for k:= 1 to prm.NbClusters do
  begin
   //vu le mode de gnration de valeur alatoire, un doublon peut difficilement survenir
   idExample:= prmExamples.Number[succ(FRndGenClustering.IRanMarRange(prmExamples.Size))];
   //crer le cluster
   clus:= TOneClusterKMeans.Create(Inputs,prm,StatsInputs,idExample);
   FCurClusters.AddCluster(clus);
  end;
 //tableau des inerties intra
 setLength(FWSS,prm.NbClusters);
end;

function TCalcClusKMeans.SetClusterExample(
  example: integer): TTypeDiscrete;
var k,idK: TTypeDiscrete;
    d,dMin: double;
begin
 idK:= 0;
 dMin:= 1e308;
 for k:= 1 to FCurClusters.Count do
  begin
   d:= FCurClusters.Cluster[pred(k)].CalcDistance(example);
   if (d<dMin)
    then
     begin
      dMin:= d;
      idK:= k;
     end;
  end;
 //si on a trouv un min
 if (idK>0)
  then result:= idK
  //sinon affectation au hasard
  else result:= succ(FRndGenClustering.IRanMarRange(FCurClusters.Count));
end;

{ TOpPrmClusKMeans }

function TOpPrmClusKMeans.CreateDlgParameters: TForm;
begin
 result:= TDlgOpPrmClusKMeans.CreateFromOpPrm(self);
end;

function TOpPrmClusKMeans.getHTMLParameters: string;
var s,sPrm: string;
begin
 s:= HTML_HEADER_TABLE_RESULT;
 s:= s+HTML_TABLE_COLOR_HEADER_GRAY+'<TH colspan=2>K-Means parameters</TH></TR>';
 s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>%s</TD><TD align="right">%d</TD></TR>',['Clusters',FNbClusters]);
 s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>%s</TD><TD align="right">%d</TD></TR>',['Max Iteration',FMaxIter]);
 s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>%s</TD><TD align="right">%d</TD></TR>',['Trials',FNbTrials]);
 case FNormalization of
  0: sPrm:= 'none';
  1: sPrm:= 'variance';
 end;
 s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>%s</TD><TD align="right">%s</TD></TR>',['Distance normalization',sPrm]);
 case FAvgComputation of
  0: sPrm:= 'Forgy';
  1: sPrm:= 'McQueen';
 end;
 s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>%s</TD><TD align="right">%s</TD></TR>',['Average computation',sPrm]);
 //random generator
 s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>%s</TD><TD align="right">%s</TD></TR>',['Seed random generator',StartSeedDescription[self.FModeRndGenerator]]);
 s:= s+'</table>';
 result:= s;
end;

procedure TOpPrmClusKMeans.LoadFromINI(prmSection: string;
  prmINI: TMemIniFile);
begin
 inherited;
 //si mauvaise lecture, les valeurs par dfaut sont conservs
 FNbClusters:= prmINI.ReadInteger(prmSection,'nb_clusters',FNbClusters);
 FNormalization:= prmINI.ReadInteger(prmSection,'normalization',FNormalization);
 FAvgComputation:= prmINI.ReadInteger(prmSection,'avg_computation',FAvgComputation);
 FMaxIter:= prmINI.ReadInteger(prmSection,'max_iter',FMaxIter);
 FNbTrials:= prmINI.ReadInteger(prmSection,'nb_trials',FNbTrials);
end;

procedure TOpPrmClusKMeans.LoadFromStream(prmStream: TStream);
begin
 inherited;
 prmStream.ReadBuffer(FNbClusters,sizeof(FNbClusters));
 prmStream.ReadBuffer(FNormalization,sizeof(FNormalization));
 prmStream.ReadBuffer(FAvgComputation,sizeof(FAvgComputation));
 prmStream.ReadBuffer(FMaxIter,sizeof(FMaxIter));
 prmStream.ReadBuffer(FNbTrials,sizeof(FNbTrials));
end;

procedure TOpPrmClusKMeans.SaveToINI(prmSection: string;
  prmINI: TMemIniFile);
begin
 inherited;
 prmINI.WriteInteger(prmSection,'nb_clusters',FNbClusters);
 prmINI.WriteInteger(prmSection,'normalization',FNormalization);
 prmINI.WriteInteger(prmSection,'avg_computation',FAvgComputation);
 prmINI.WriteInteger(prmSection,'max_iter',FMaxIter);
 prmINI.WriteInteger(prmSection,'nb_trials',FNbTrials);
end;

procedure TOpPrmClusKMeans.SaveToStream(prmStream: TStream);
begin
 inherited;
 prmStream.WriteBuffer(FNbClusters,sizeof(FNbClusters));
 prmStream.WriteBuffer(FNormalization,sizeof(FNormalization));
 prmStream.WriteBuffer(FAvgComputation,sizeof(FAvgComputation));
 prmStream.WriteBuffer(FMaxIter,sizeof(FMaxIter));
 prmStream.WriteBuffer(FNbTrials,sizeof(FNbTrials));
end;

procedure TOpPrmClusKMeans.SetDefaultParameters;
begin
 inherited SetDefaultParameters();
 FNbClusters:= 3;
 FNormalization:= 1;
 FAvgComputation:= 1;
 FMaxIter:= 10;
 FNbTrials:= 5;
end;

{ TStrucClustersKMeans }

procedure TStrucClustersKMeans.AddCluster(prmClus: TOneClusterKMeans);
begin
 FLstClusters.Add(prmClus);
end;

constructor TStrucClustersKMeans.Create;
begin
 inherited Create();
 FLstClusters:= TObjectList.Create(TRUE);
end;

destructor TStrucClustersKMeans.Destroy;
begin
 FLstClusters.Free;
 inherited;
end;

function TStrucClustersKMeans.GetCluster(i: integer): TOneClusterKMeans;
begin
 result:= FLstClusters.Items[i] as TOneClusterKMeans;
end;

function TStrucClustersKMeans.GetNbClusters: integer;
begin
 result:= FLstClusters.Count;
end;

initialization
 RegisterClass(TGenClusKMeans);
end.
