(**********************************************************************)
(* UCalcStatDesPairedTest.pas - Copyright (c) 2005 Ricco RAKOTOMALALA *)
(**********************************************************************)

{
@abstract(Classes de calcul pour les tests sur K chantillons apparis)
@author(Ricco)
@created(04/08/2005)

Principale rfrence :
----------------------
Siegel & Castellan (Nonparametric Statistics for the Behavioral Science -- dition de 1988) -- chapitre 7 et 9

Deux mthodes seront implments :
(1) FRIEDMAN 2-way ANOVA by RANKS
(2) KENDALL COEFFICIENT OF CONCORDANCE

}

unit UCalcStatDesKRelatedSamples;

interface

USES
    UDatasetDefinition,
    UDatasetExamples,
    UCalcStatDesRankForNonParametricStat;

TYPE
    //sa structure est trs diffrente des autres, il n'y a pas une variable en particulier  traiter
    //on recommence  zro donc
    TCalcSDKRelated = class(TObject)
                      protected
                      //liste des attributs
                      FLstAtt : TLstAttributes;
                      //tableau des rangs -- la dimension est k ou N selon la mthode
                      FTabRanks:array of double;
                      //correction pour les ex-aequos
                      FTieCorrection: double;
                      //la statistique
                      FStat: double;
                      //la p-value
                      FPValue: double;
                      //ddl
                      FDDL: integer;
                      //nombre d'observations
                      FNbExamples: integer;
                      //classe de calcul des rangs
                      function    getClassComputingRanks(): TClassRankComputed; virtual; abstract;
                      public
                      //passer la liste de variables
                      constructor create(prmLst: TLstAttributes); virtual;
                      //destructeur
                      destructor  destroy(); override;
                      //calcule les stats, renvoie TRUE si OK
                      function    computeStats(prmExamples: TExamples): boolean; virtual; abstract;
                      //renvoyer les rsultats sous forme HTML
                      function    getHTMLResults(): string; virtual; abstract;
                      end;

    //test de Friedman -- p.174
    TCalcSDKRelatedFriedman = class(TCalcSDKRelated)
                              protected
                              //calcul effectif de la statistique de Friedman
                              procedure   calcStatistics(prmExamples: TExamples); virtual;
                              //classe de calcul des rangs
                              function    getClassComputingRanks(): TClassRankComputed; override;
                              public
                              //calcule les stats, renvoie TRUE si OK
                              function    computeStats(prmExamples: TExamples): boolean; override;
                              //sortie
                              function    getHTMLResults(): string; override;
                              end;

    //W de concordance de Kendall -- p.262
    //exploiter la similitude avec le test de Friedman
    TCalcSDKRelatedWKendall = class(TCalcSDKRelatedFriedman)
                              protected
                              //la stat. du chi-2 associe
                              FChi2: double;
                              //moyenne du rho de Spearman
                              FAvgRho: double;
                              //calcul effectif de la statistique de Friedman
                              procedure   calcStatistics(prmExamples: TExamples); override;
                              //calcul des rangs
                              function    getClassComputingRanks(): TClassRankComputed; override;  
                              public
                              //sortie
                              function    getHTMLResults(): string; override;
                              end;


implementation

uses
    Math,
    Sysutils,
    UDatasetImplementation,
    FMath, UConstConfiguration;

{ TCalcSDKRelated }

constructor TCalcSDKRelated.create(prmLst: TLstAttributes);
begin
 inherited Create();
 FLstAtt:= prmLst;
end;

destructor TCalcSDKRelated.destroy;
begin
 if (FTabRanks <> nil) then Finalize(FTabRanks);
 inherited destroy();
end;

{ TCalcSDKRelatedFriedman }

procedure TCalcSDKRelatedFriedman.calcStatistics(prmExamples: TExamples);
var cK,cN: double;
    s2Ranks,num,denom: double;
    k: integer;
begin
 //taille de tableau -- on se rfre  la terminologie de Siegel, p.179
 cN:= 1.0 * prmExamples.Size;
 ck:= 1.0 * FLstAtt.Count;
 //la somme au carr des rangs
 s2Ranks:= 0.0;
 for k:= 0 to pred(FLstAtt.Count) do
  s2Ranks:= s2Ranks + FTabRanks[k] * FTabRanks[k];
 //ddl.
 FDDL:= FLstAtt.Count - 1;
 //si tout va bien, on calcule la statistique
 //numrateur
 num:= 12.0 * s2Ranks - 3.0 * power(cN,2.0) * cK * power(cK+1.0,2.0);
 //dnominateur
 denom:= cN* cK * (cK + 1.0) + 1.0/(cK - 1.0) * (cN * cK - FTieCorrection);
 //la stat.
 if (denom > 0.0)
  then FStat:= num / denom
  else FStat:= 0.0;
 //p-value
 FPValue:= PKhi2(FDDL,FStat);
end;

function TCalcSDKRelatedFriedman.computeStats(
  prmExamples: TExamples): boolean;
var attRanks: TAttContinue;
    lstSorted: TExamples;
    ranks: TRankComputed;
    i,k: integer;
    correction: double;
begin
 FNbExamples:= prmExamples.Size;
 //tableau -- somme des rangs (s'assurer qu'il sera vide et rempli de zros !)
 if (FTabRanks <> nil) then Finalize(FTabRanks);
 setLength(FTabRanks,FLstAtt.Count);
 //variables de calculs -- un peu tir par les cheveux mais a a le mrite de profiter  fond des structures existantes
 //!\on prend le tableau de donnes de travers ( l'horizontale)
 lstSorted:= TExamples.Create(FLstAtt.Count);
 attRanks:= TAttContinue.Create('_att_',FLstAtt.Count);
 ranks:= getClassComputingRanks.create(attRanks);
 TRY
  result:= TRUE;
  TRY
   FTieCorrection:= 0.0;
   //pour chaque observation
   for i:= 1 to prmExamples.Size do
    begin
     //remplir la variable temporaire
     for k:= 0 to pred(FLstAtt.Count) do
      attRanks.cValue[succ(k)]:= FLstAtt.Attribute[k].cValue[prmExamples.Number[i]];
     //recalculer le tableau de rangs
     lstSorted.fillValues();
     lstSorted.QuickSortBy(attRanks);
     //mme pour les ex-aequos de taille 1, on corrige (!) -- Siegel, p.179 -- dernier paragraphe
     //calcul en 2 temps pour dbogage et affichage dans le log.
     correction:= ranks.computeRank(lstSorted);
     //
     FTieCorrection:= FTieCorrection + correction;
     //faire la somme des rangs
     for k:= 0 to pred(FLstAtt.Count) do
       FTabRanks[k]:= FTabRanks[k] + ranks.getRank(succ(k));
    end;
   //calculer les statistiques --> Stat, DDL, et p-value
   self.calcStatistics(prmExamples);
  EXCEPT
   result:= FALSE;
  END;
 FINALLY
 ranks.Free();
 lstSorted.Free();
 attRanks.Free();
 END;
end;

function codeCouleurKRelatedSamples(alpha: double): string;
begin
 if (alpha < SIGNIFICANCE_LEVEL_FOR_TESTS)
  then result:= HTML_BGCOLOR_DATA_RED
  else result:= HTML_BGCOLOR_DATA_GREEN;
end;

function TCalcSDKRelatedFriedman.getClassComputingRanks: TClassRankComputed;
begin
 result:= TRankComputedKRelatedFriedman; 
end;

function TCalcSDKRelatedFriedman.getHTMLResults: string;
var s: string;
    k: integer;
begin
 s:= '<H3>Results</H3>';
 s:= s + HTML_HEADER_TABLE_RESULT;
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY + '<TH>RANKS</TH><TH>Friedman Statistic</TH></TR>';
 s:= s + HTML_TABLE_COLOR_DATA_GRAY;
 //description des rangs
 s:= s + '<TD valign="top">';
 s:= s + HTML_HEADER_TABLE_RESULT;
 s:= s + HTML_TABLE_COLOR_HEADER_BLUE + '<TH>Att.</TH><TH>Sum(Ranks)</TH><TH>Mean(Ranks)</TH></TR>';
 for k:= 0 to pred(FLstAtt.Count) do
  s:= s + HTML_TABLE_COLOR_HEADER_GRAY + format('<TD>%s</TD><TD align=right>%.1f</TD><TD align="right">%.4f</TD></TR>',
                                        [FLstAtt.Attribute[k].Name,FTabRanks[k],FTabRanks[k]/(1.0*FNbExamples)]);
 s:= s + '</table>';
 s:= s + '</TD>';
 //descritpion de la statistique
 s:= s + '<TD valign="top">';
 s:= s + HTML_HEADER_TABLE_RESULT;
 s:= s + HTML_TABLE_COLOR_HEADER_BLUE + '<TH>Stat.</TH><TH>Value</TH></TR>';
 s:= s + HTML_TABLE_COLOR_DATA_BLUE + format('<TD>Frideman Fr</TD><TD align="right">%.5f</TD></TR>',[FStat]);
 s:= s + HTML_TABLE_COLOR_DATA_BLUE + format('<TD>d.f.</TD><TD align="right">%d</TD></TR>',[FDDL]);
 s:= s + HTML_TABLE_COLOR_DATA_BLUE + format('<TD>p-value</TD><TD align="right" %s>%.5f</TD></TR>',[codeCouleurKRelatedSamples(FPValue),FPValue]);
 s:= s + '</table>';
 s:= s+'</TD>';
 //finaliser le tableau
 s:= s + '</TR>';
 s:= s + '</table>';
 //
 result:= s;
end;

{ TCalcSDKRelatedWKendall }

procedure TCalcSDKRelatedWKendall.calcStatistics(prmExamples: TExamples);
var cK,cN: double;
    s2Ranks,num,denom: double;
    k: integer;
begin
 //terminologie Siegel, p.262 -- on prend le tableau dans l'autre sens !
 cK:= 1.0 * prmExamples.Size;  //nombre de ligne = k
 cN:= 1.0 * FLstAtt.Count; //nombre de colonnes = N
 //la somme au carr des rangs
 s2Ranks:= 0.0;
 for k:= 0 to pred(FLstAtt.Count) do
  s2Ranks:= s2Ranks + FTabRanks[k] * FTabRanks[k];
 //la statistique, p.266 (9.18b)
 num:= 12.0 * s2Ranks - 3.0 * power(cK,2.0) * cN * power(cN + 1.0,2.0);
 denom:= power(cK,2.0) * cN * (power(cN,2.0) - 1.0) - cK * FTieCorrection;
 if (denom > 0.0)
  then FStat:= num / denom
  else FStat:= 0.0;
 //moyenne des rho de Spearman (calculs 2  2) -> Siegel, p. 262 , eq. 9.16
 FAvgRho:= (cK * FStat - 1.0) / (cK - 1.0);
 //la stat transforme -> chi-2 pour le test (eq. 9.19)
 FChi2:= cK * (cN - 1.0) * FStat;
 //ddl -> (N-1)
 FDDL:= FLstAtt.Count - 1;
 //p-value
 FPValue:= PKhi2(FDDL,FChi2);
end;

function TCalcSDKRelatedWKendall.getClassComputingRanks: TClassRankComputed;
begin
 result:= TRankComputedKRelatedWKendall;
end;

function TCalcSDKRelatedWKendall.getHTMLResults: string;
var s: string;
    k: integer;
begin
 s:= '<H3>Results</H3>';
 s:= s + HTML_HEADER_TABLE_RESULT;
 s:= s + HTML_TABLE_COLOR_HEADER_GRAY + '<TH>RANKS</TH><TH>Concordance Statistic</TH></TR>';
 s:= s + HTML_TABLE_COLOR_DATA_GRAY;
 //description des rangs
 s:= s + '<TD valign="top">';
 s:= s + HTML_HEADER_TABLE_RESULT;
 s:= s + HTML_TABLE_COLOR_HEADER_BLUE + '<TH>Att.</TH><TH>Sum(Ranks)</TH><TH>Mean(Ranks)</TH></TR>';
 for k:= 0 to pred(FLstAtt.Count) do
  s:= s + HTML_TABLE_COLOR_HEADER_GRAY + format('<TD>%s</TD><TD align=right>%.1f</TD><TD align="right">%.4f</TD></TR>',
                                        [FLstAtt.Attribute[k].Name,FTabRanks[k],FTabRanks[k]/(1.0*FNbExamples)]);
 s:= s + '</table>';
 s:= s + '</TD>';
 //descritpion de la statistique
 s:= s + '<TD valign="top">';
 s:= s + HTML_HEADER_TABLE_RESULT;
 s:= s + HTML_TABLE_COLOR_HEADER_BLUE + '<TH>Stat.</TH><TH>Value</TH></TR>';
 s:= s + HTML_TABLE_COLOR_DATA_BLUE + format('<TD>Kendall W</TD><TD align="right">%.5f</TD></TR>',[FStat]);
 s:= s + HTML_TABLE_COLOR_DATA_BLUE + format('<TD>Average(Spearman''s Rho)</TD><TD align="right">%.5f</TD></TR>',[FAvgRho]);
 s:= s + HTML_TABLE_COLOR_DATA_BLUE + format('<TD>Chi-2</TD><TD align="right">%.5f</TD></TR>',[FChi2]);
 s:= s + HTML_TABLE_COLOR_DATA_BLUE + format('<TD>d.f.</TD><TD align="right">%d</TD></TR>',[FDDL]);
 s:= s + HTML_TABLE_COLOR_DATA_BLUE + format('<TD>p-value</TD><TD align="right" %s>%.5f</TD></TR>',[codeCouleurKRelatedSamples(FPValue),FPValue]);
 s:= s + '</table>';
 s:= s+'</TD>';
 //finaliser le tableau
 s:= s + '</TR>';
 s:= s + '</table>';
 //
 result:= s;
end;

end.
