(********************************************************************)
(* UCompFactAnalysisPCA.pas - Copyright (c) 2004 Ricco RAKOTOMALALA *)
(********************************************************************)

{
@abstract(Analyse en Composantes Principales)
@author(Ricco)
@created(12/01/2004)
}
unit UCompFactAnalysisPCA;

interface

USES
        Forms,Classes,
        IniFiles,
        Matrices,
        UCalcStatDes,
        UCompDefinition,
        UCompFADefinition,
        UOperatorDefinition;

TYPE
        {gnrateur de composant ACP}
        TMLGenCompFactPCA = class(TMLGenComp)
                            protected
                            procedure   GenCompInitializations(); override;
                            public
                            function    GetClassMLComponent: TClassMLComponent; override;
                            end;

        {composant ACP}
        TMLCompFactPCA = class(TMLCompFactAnalysis)
                         protected
                         function    getClassOperator: TClassOperator; override;
                         end;

        {oprateur ACP, on travaille sur les target}
        TOpFactPCA = class(TOpFactAnalysis)
                     private
                     {StatDes}
                     FLstStats: TLstCalcStatDesContinuous;
                     {dimension des matrices}
                     FDimAnalysis: integer;
                     {matrice de corrlation}
                     FCorrMat: PMatrix;
                     {vecteur des valeurs propres}
                     FEgVal: PVector;
                     {matrice des vecteurs propres}
                     FEgVec: PMatrix;
                     {calculer les stats descriptives}
                     procedure   CalcStats();
                     {construire la matrice de corrlation}
                     procedure   BuildVCV();
                     {calculer les valeurs et vecteurs propres}
                     procedure   BuildEigen();
                     {effectuer les projections sur les axes factoriels}
                     procedure   SetProjections();
                     protected
                     {dtruire les matrices et vecteurs}
                     procedure   DelPrivateMatrix(); override;
                     function    getClassParameter: TClassOperatorParameter; override;
                     function    CheckAttributes(): boolean; override;
                     function    CoreExecute(): boolean; override;
                     public
                     constructor Create(AOwner: TObject); override;
                     destructor  Destroy; override;
                     function    getHTMLResultsSummary(): string; override;
                     end;

        {paramtre oprateur ACP}
        TOpPrmFactPCA = class(TOpPrmFactAnalysis)
                        private
                        {nombre d'axes  crer, si -1 on les gnre tous}
                        FNbAxisCreated: integer;
                        protected
                        function    CreateDlgParameters(): TForm; override;
                        procedure   SetDefaultParameters(); override;
                        public
                        function    getHTMLParameters(): string; override;
                        procedure   LoadFromStream(prmStream: TStream); override;
                        procedure   SaveToStream(prmStream: TStream); override;
                        procedure   LoadFromINI(prmSection: string; prmINI: TMemIniFile); override;
                        procedure   SaveToINI(prmSection: string; prmINI: TMemIniFile); override;
                        property    NbAxisCreated: integer read FNbAxisCreated write FNbAxisCreated;
                        end;

implementation

USES
        SysUtils,Math,
        UDatasetImplementation, UCalcMatrixToAttributes,
        UCalcMatrixAdditionalFunctions, UConstConfiguration, UDatasetDefinition,
        UDlgOpPrmPCA, UStringsResources, UStringAddBuffered;

{ TMLGenCompFactPCA }

procedure TMLGenCompFactPCA.GenCompInitializations;
begin
 FMLComp:= mlcFactorialAnalysis;
 //FMLNumIcon:= 10;
 //FMLCompName:= str_comp_name_fact_pca;
 //FMLBitmapFileName:= 'MLFactorialPCA.bmp';
end;

function TMLGenCompFactPCA.GetClassMLComponent: TClassMLComponent;
begin
 result:= TMLCompFactPCA;
end;

{ TMLCompFactPCA }

function TMLCompFactPCA.getClassOperator: TClassOperator;
begin
 result:= TOpFactPCA;
end;

{ TOpFactPCA }

procedure TOpFactPCA.BuildEigen;
begin
 GetEigensFromSymetricMatrix(FCorrMat,FDimAnalysis,FEgVal,FEgVec);
end;

procedure TOpFactPCA.SetProjections();
var i,j,k: integer;
    newAtt: TAttContinue;
    att: TAttribute;
    stat: TCalcStatDesContinuous;
    attSize: integer;
    s: TTypeContinue;
    nbAxis: integer;
begin
 GenAtts.Clear;
 attSize:= WorkData.LstAtts[asAll].Size;
 //dterminer le nombre d'axes  gnrer
 nbAxis:= (PrmOp as TOpPrmFactPCA).NbAxisCreated;
 if (nbAxis<0)
  then nbAxis:= FDimAnalysis
  else nbAxis:= min(nbAxis,FDimAnalysis);
 //ok on peut lancer les calculs
 for k:= 1 to nbAxis do
  begin
   newAtt:= TAttContinue.Create(format('PCA_%d_Axis_%d',[(MLOwner as TMLCompFactPCA).Number,k]),attSize);
   //pour chaque individu i
   for i:= 1 to attSize do
    begin
     s:= 0.0;
     for j:= 1 to FDimAnalysis do
      begin
       att:= WorkData.LstAtts[asInput].Attribute[pred(j)];
       stat:= TCalcStatDesContinuous(FLstStats.Stat(pred(j)));
       s:= s+FEgVec^[j]^[k]*(att.cValue[i]-stat.Average)/stat.StdDev;
      end;
     newAtt.cValue[i]:= s;
    end;
   GenAtts.Add(newAtt);
  end;
end;

procedure TOpFactPCA.BuildVCV;
begin
 FCorrMat:= BuildMatVCV(workdata.Examples,Workdata.lstAtts[asInput],vcvNormNormalized,FLstStats);
end;

function TOpFactPCA.CoreExecute: boolean;
begin
 result:= TRUE;
 TRY
 self.CalcStats();
 self.BuildVCV();
 self.BuildEigen();
 self.SetProjections();
 EXCEPT
 self.DelPrivateMatrix();
 result:= FALSE;
 END;
end;

procedure TOpFactPCA.DelPrivateMatrix;
begin
 //dtruire les matrices internes
 delMatrix(FCorrMat,FDimAnalysis,FDimAnalysis);
 delMatrix(FEgVec,FDimAnalysis,FDimAnalysis);
 delVector(FEgVal,FDimAnalysis);
 FLstStats.FreeAll;
end;

function TOpFactPCA.getClassParameter: TClassOperatorParameter;
begin
 result:= TOpPrmFactPCA;
end;

function TOpFactPCA.getHTMLResultsSummary: string;
var s: string;
    sum: double;
    tot: double;
    i,j: integer;
    buf: TBufString;
    nbAxis: integer;
begin
 nbAxis:= (self.PrmOp as TOpPrmFactPCA).NbAxisCreated;
 //buffriser l'ajout -- 23/06/2004
 buf:= TBufString.Create();
 buf.BeginUpdate();
 //#ToDo1 - affichage des rsultats cf. SPAD et STATISTICA
 //afficher les stats des.
 //afficher la matrice de corrlation
 //afficher les valeurs propres
 //la trace de la matrice
 s:= '<P><H3>Eigen values</H3>';
 s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY+
       '<TH>Axis</TH><TH>Eigen value</TH><TH>% explained</TH><TH width="200">Histogram</TH><TH>% cumulated</TH></TR>';
 buf.AddStr(s);
 tot:= 1.0*FDimAnalysis;
 sum:= 0.0;
 for i:= 1 to FDimAnalysis do
  begin
   sum:= sum+FEgVal^[i];
   if (tot>0)
    then
     begin
      s:= HTML_TABLE_COLOR_DATA_GRAY+
         format('<TH>%d</TH><TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY_HIGH+'</TD><TD align=right>%.2f%s</TD><TD>%s</TD><TD align=right>%.2f%s</TD></TR>',
                [i,FEgVal^[i],100.0*FEgVal^[i]/tot,'%',getHtmlHistogram(TRUNC(100.0*FEgVal^[i]/tot)),100.0*sum/tot,'%'])+'</TR>';
      buf.AddStr(s);
     end;
  end;
 buf.AddStr('</table>');
 //**
 //new -- 13/02/2005 -- afficher les vecteurs propres
 //**
 s:= '<P><H3>Eigen vectors (coefficients on standardized dataset)</H3>';
 //en-tte pour chaque axe factoriel demand
 s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY+'<TH width=100>Attribute</TH>';
 for j:= 1 to MIN(nbAxis,FDimAnalysis) do
  s:= s+format('<TH width=60 align=right>Axis_%d</TH>',[j]);
 s:= s+'</TR>';
 //le contenu
 for i:= 1 to FDimAnalysis do
  begin
   s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TH>%s</TH>',[WorkData.LstAtts[asInput].Attribute[pred(i)].Name]);
   for j:= 1 to MIN(nbAxis,FDimAnalysis) do
    begin
      TRY
      s:= s+format('<TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</TD>',[FEgVec^[i]^[j]]);
      EXCEPT
      END;
    end;
   s:= s+'</TR>';
  end;
 s:= s+'</table>';
 //dans le buffer...
 buf.AddStr(s);
 //**
 //afficher la corrlation des attributs avec les axes
 //**
 s:= '<P><H3>Attributes correlation with PC Axis</H3>';
 s:= s+HTML_HEADER_TABLE_RESULT+HTML_TABLE_COLOR_HEADER_GRAY+'<TH width=100>Attribute</TH>';
 buf.AddStr(s);
 for j:= 1 to MIN(nbAxis,FDimAnalysis) do
  buf.AddStr(format('<TH width=60 align=right>Axis_%d</TH>',[j]));
 buf.AddStr('</TR>');
 for i:= 1 to FDimAnalysis do
  begin
   buf.AddStr(HTML_TABLE_COLOR_DATA_GRAY+format('<TH>%s</TH>',[WorkData.LstAtts[asInput].Attribute[pred(i)].Name]));
   for j:= 1 to MIN(nbAxis,FDimAnalysis) do
    begin
      TRY
      buf.AddStr(format('<TD align=right>'+STR_FORMAT_VIEW_STAT_ACCURACY+'</TD>',[sqrt(FEgVal^[j])*FEgVec^[i]^[j]]));
      EXCEPT
      END;
    end;
   buf.AddStr('</TR>');
  end;
 buf.AddStr('</table>');
 //envoyer la sauce
 buf.EndUpdate();
 result:= buf.BufS;
 buf.Free();
end;

constructor TOpFactPCA.Create(AOwner: TObject);
begin
 inherited Create(AOwner);
 FLstStats:= TLstCalcStatDesContinuous.Create(NIL,NIL);
end;

procedure TOpFactPCA.CalcStats;
begin
 FDimAnalysis:= WorkData.LstAtts[asInput].Count;
 FLstStats.RebuildStatDes(WorkData.LstAtts[asInput],Workdata.Examples);
end;

destructor TOpFactPCA.Destroy;
begin
 inherited Destroy;
 FLstStats.Free;
end;

function TOpFactPCA.CheckAttributes: boolean;
begin
 result:= (self.WorkData.LstAtts[asInput].Count>0) and (self.WorkData.LstAtts[asInput].isAllCategory(caContinue));
end;

{ TOpPrmFactPCA }

function TOpPrmFactPCA.CreateDlgParameters: TForm;
begin
 result:= TDlgOpPrmPCA.CreateFromOpPrm(self);
end;

function TOpPrmFactPCA.getHTMLParameters: string;
var s: string;
begin
 s:= '<P><B>Generated axis : </B>';
 if (FNbAxisCreated<0)
  then s:= s+'all'
  else s:= s+IntToStr(FNbAxisCreated);
 result:= s;
end;

procedure TOpPrmFactPCA.LoadFromINI(prmSection: string;
  prmINI: TMemIniFile);
begin
 FNbAxisCreated:= prmINI.ReadInteger(prmSection,'nb_axis',FNbAxisCreated);
end;

procedure TOpPrmFactPCA.LoadFromStream(prmStream: TStream);
begin
 prmStream.ReadBuffer(FNBAxisCreated,sizeof(FNBAxisCreated));
end;

procedure TOpPrmFactPCA.SaveToINI(prmSection: string; prmINI: TMemIniFile);
begin
 prmINI.WriteInteger(prmSection,'nb_axis',FNbAxisCreated);
end;

procedure TOpPrmFactPCA.SaveToStream(prmStream: TStream);
begin
 prmStream.WriteBuffer(FNBAxisCreated,sizeof(FNBAxisCreated));
end;

procedure TOpPrmFactPCA.SetDefaultParameters;
begin
 FNbAxisCreated:= 10;
end;

initialization
 RegisterClass(TMLGenCompFactPCA);
end.
