(************************************************************)
(* UCompPlsBase.pas - Copyright (c) 2005 Ricco RAKOTOMALALA *)
(************************************************************)

{
@abstract(PLS Base)
@author(Ricco)
@created(13/05/2005)

Composant de base PLS qui calcule les moyennes et ecart-type des variables en TARGET et INPUT.
Ce composant sert de maquette de dpart pour les implmentations futures des mthodes PLS dans TANAGRA.

Il suffit de reproduire la squence ci-dessous en surchargeant chaque classe pour crer un nouveau composant PLS dans TANAGRA. Ne pas
oublier galement de crer l'icne graphique et de modifier le fichier de configuration des composants
(cf. "tanagra_components.xml")



}

unit UCompPlsBase;

interface

USES
       //units std delphi
       Classes, IniFiles, Forms, Sysutils,
       //gestion des composants
       UCompDefinition, UOperatorDefinition,
       UCompManageDataset;

TYPE
       //***********************************************************************
       //** gnrateur de composant, son rle est de crer une instance de composant qui est insr dans le diagramme de traitements
       //***********************************************************************
       TMLGenCompPLSBase = class(TMLGenComp)
                           protected
                           //spcifier dans quel onglet de la palette doit apparatre l'icne -- ne pas surcharger dans les hritiers
                           procedure   genCompInitializations(); override;
                           public
                           //spcifier quel est la classe du composant  gnrer sur le diagramme --  surcharger obligatoirement chez les hritiers
                           function    getClassMLComponent: TClassMLComponent; override;
                           end;

       //***********************************************************************
       //** classe du composant graphique insr dans le diagramme
       //***********************************************************************
       TMLCompPLSBase = class(TMLCompLocalData)
                        protected
                        //spcifier l'oprateur de calcul --  surcharger obligatoirement
                        function    getClassOperator: TClassOperator; override;
                        end;

       //***********************************************************************
       //** classe oprateur de calcul : prend en charge le calcul et l'affichage des rsultats
       //***********************************************************************
       TOpPlsBase = class(TOpLocalData)
                    private
                    //tableau des moyennes TARGET et INPUT
                    FTabAvg: array of double;
                    //tableau des cart-types TARGET et INPUT
                    FTabStdDev: array of double;
                    protected
                    //spcifier la classe de paramtrage -- surcharge obligatoire
                    function    getClassParameter: TClassOperatorParameter; override;
                    //contrle de cohrence des attributs avant les calculs --  surcharger selon les mthodes
                    function    checkAttributes(): boolean; override;
                    //lancer effectivement les calculs -- surcharge obligatoire
                    function    coreExecute(): boolean; override;
                    public
                    //libration de la mmoire alloue -- surcharge selon les mthodes, appel de la mthode anctre avec "inherited" obligatoire
                    destructor  destroy(); override;
                    //affichage des rsultats au format HTML -- surcharge obligatoire
                    function    getHTMLResultsSummary(): string; override;
                    end;

       //***********************************************************************
       //** classe paramtrage de l'oprateur de calcul : gre la bote de paramtrage et l'E/S des paramtres
       //** si la mthode de requiert pas de paramtres, il suffit de crer une classe vide hritire de "TOperatorParameter"
       //***********************************************************************
       TOpPrmPlsBase = class(TOperatorParameter)
                       private
                       //un paramtre qui permet de spcifier si l'on doit calculer l'cart-type chantillon ou population
                       //0 --> population (division par "n"); 1 --> chantillon (division par "n-1")
                       FTypeStdDev: integer;
                       protected
                       //cration de la bote de dialogue de paramtrage -- surcharge obligatoire (doit renvoyer "nil" si pas de bote  gnrer)
                       function    CreateDlgParameters(): TForm; override;
                       //spcifier les paramtres par dfaut de la mthode -- surcharge obligatoire
                       procedure   SetDefaultParameters(); override;
                       public
                       //affichage HTML des paramtres -- surcharge obligatoire
                       function    getHTMLParameters(): string; override;
                       //sauvegarde des paramtres dans diffrents formats -- flux de donnes et fichiers INI -- surcharge obligatoire
                       procedure   SaveToStream(prmStream: TStream); override;
                       procedure   SaveToINI(prmSection: string; prmINI: TMemIniFile); override;
                       //chargement des paramtres dans diffrents formats -- flux de donnes et fichiers INI -- surcharge obligatoire -- doit tre "mirroir" des save
                       procedure   LoadFromStream(prmStream: TStream); override;
                       procedure   LoadFromINI(prmSection: string; prmINI: TMemIniFile); override;
                       //proprits -- mettre en accs public les proprits prives
                       property    TypeStdDev: integer read FTypeStdDev write FTypeStdDev;
                       end;

implementation

uses
       //unit de dfinition des donnes
       UDatasetDefinition, UDatasetImplementation, UDatasetExamples,
       //rfrence  l'unit dfinissant la bote de dialogue de paramtrage
       UDlgOpPrmPlsBase;

{ TMLGenCompPLSBase }

procedure TMLGenCompPLSBase.genCompInitializations;
begin
 //onglet PLS
 FMLComp:= mlcPLS;
end;

function TMLGenCompPLSBase.getClassMLComponent: TClassMLComponent;
begin
 //classe du composant associ
 result:= TMLCompPLSBase;
end;

{ TMLCompPLSBase }

function TMLCompPLSBase.getClassOperator: TClassOperator;
begin
 //oprateur de calcul associ
 result:= TOpPlsBase;
end;

{ TOpPlsBase }

function TOpPlsBase.checkAttributes: boolean;
var ok: boolean;
begin
 //premier test, il y a des target et ils sont continus ou binaires
 ok:= (self.WorkData.LstAtts[asTarget].Count > 0) and self.WorkData.LstAtts[asTarget].isAllCategory(caQuasiContinue);
 //second test -- idem pour les input
 ok:= ok and ((self.WorkData.LstAtts[asInput].Count > 0) and self.WorkData.LstAtts[asInput].isAllCategory(caQuasiContinue));
 //renvoyer la rponse
 result:= ok;
end;

function TOpPlsBase.coreExecute: boolean;
var nbAttTarget, nbAttInput: integer;
    nbObservations: integer;
    stdDevDiv: double;
    j: integer;
    att: TAttribute;

    //***********************************************
    //procedure locale de calcul pour chaque variable
    //***********************************************
    procedure calculStat(attribute: TAttribute; column: integer);
    var sum,sumSq,value: double;
        i: integer;
    begin
     //initialisation
     sum:= 0.0;
     sumSq:= 0.0;
     //pour chaque individu -- l'indice commence  1 pour les individus (!)
     for i:= 1 to nbObservations do
      begin
       //rcuprer la valeur -- noter le drfrencement avec la proprit "Number"
       value:= attribute.cValue[self.WorkData.Examples.Number[i]];
       //additionner
       sum:= sum + value;
       sumSq:= sumSq + value * value;
      end;
     //moyenne
     FTabAvg[column]:= sum/(1.0*nbObservations);
     //somme des carrs des carts
     sumSq:= sumSq - 1.0*nbObservations*FTabAvg[column]*FTabAvg[Column];
     //division pour la variance (population ou chantillon)
     sumSq:= sumSq/stdDevDiv;
     //racine carre -- attention si risque de racine sur une valeur ngative
     FTabStdDev[column]:= SQRT(ABS(sumSq));
    end;

begin
 //************************************************
 //*** rcupration des paramtres de l'opration *
 //************************************************

 //rcuprer le nombre de variables
 nbAttTarget:= self.WorkData.LstAtts[asTarget].Count;
 nbAttInput := self.WorkData.LstAtts[asInput].Count;
 //allouer les tableaux (ou modifier leur taille si dj allous une premire fois)
 setLength(FTabAvg,nbAttTarget+nbAttInput);
 setLength(FTabStdDev,nbAttTarget+nbAttInput);
 //rcuprer le nombre d'individus slectionns pour l'analyse
 nbObservations:= self.WorkData.Examples.Size;
 //selon le paramtrage -- le diviseur de std-dev
 if ((self.PrmOp as TOpPrmPlsBase).TypeStdDev = 0)
  //population -- divison par "n"
  then stdDevDiv:= nbObservations
  //chantillon -- division par "n-1"
  else stdDevDiv:= -1.0+nbObservations;

 //************
 //*** calcul *
 //************

 //utiliser un gestionnaire d'exceptions pour prvenir le plantage
 TRY

 //calcul sur les target
 //les indices de variables commencent  zro
 for j:= 0 to nbAttTarget - 1 do
  begin
   //rcuprer la variable
   att:= self.WorkData.LstAtts[asTarget].Attribute[j];
   //calculer
   calculStat(att,j);
  end;

 //calcul sur les input
 //les indices de variables commencent  zro
 for j:= 0 to nbAttInput - 1 do
  begin
   //rcuprer la variable
   att:= self.WorkData.LstAtts[asInput].Attribute[j];
   //calculer
   calculStat(att,j+nbAttTarget);
  end;

 //si tout va bien
 result:= TRUE;
 
 EXCEPT

 //en cas de plantage
 result:= false;
 
 END;

end;

destructor TOpPlsBase.destroy;
begin
 //librer les tableaux internes s'ils sont encore allous
 if (FTabAvg <> nil) then Finalize(FTabAvg);
 if (FTabStdDev <> nil) then Finalize(FTabStdDev);
 //ne pas oublier !!!
 inherited destroy();
end;

function TOpPlsBase.getClassParameter: TClassOperatorParameter;
begin
 //classe de paramtrage
 result:= TOpPrmPlsBase;
end;

function TOpPlsBase.getHTMLResultsSummary: string;
var s: string;
    j: integer;
    att: TAttribute;
begin
 //afficher les rsultats dans un tableau
 s:= '<h3>Test architecture de classes PLS</H3>';
 s:= s+'<table border=1>';
 s:= s+'<tr><th>Attribute</th><th width=50>Average</th><th width=50>Std-dev</th></tr>';

 //pour chaque attribut target
 s:= s+'<tr><th colspan=3>TARGET</th></tr>';
 for j:= 0 to self.WorkData.LstAtts[asTarget].Count - 1 do
  begin
   att:= self.WorkData.LstAtts[asTarget].Attribute[j];
   //sortie formatte
   s:= s+format('<tr><td>%s</td><td align="right">%.4f</td><td align="right">%.4f</td></tr>',[att.Name,FTabAvg[j],FTabStdDev[j]]);
  end;

 //pour chaque attribut input
 s:= s+'<tr><th colspan=3>INPUT</th></tr>';
 for j:= 0 to self.WorkData.LstAtts[asInput].Count - 1 do
  begin
   att:= self.WorkData.LstAtts[asInput].Attribute[j];
   //sortie formatte -- attention au dcalage pour les colonnes soient synchrones ici
   s:= s+format('<tr><td>%s</td><td align="right">%.4f</td><td align="right">%.4f</td></tr>',
         [att.Name,FTabAvg[j+self.WorkData.LstAtts[asTarget].Count],FTabStdDev[j+self.WorkData.LstAtts[asTarget].Count]]);
  end;

 //finaliser la table
 s:= s+'</table>';

 //renvoyer le tout
 result:= s;
end;

{ TOpPrmPlsBase }

function TOpPrmPlsBase.CreateDlgParameters: TForm;
begin
 //instanciation de la bote de dialogue de paramtrage adquat -- on passe l'instance courante en rfrence
 result:= TdlgOpPrmPlsBase.CreateFromOpPrm(self);
end;

function TOpPrmPlsBase.getHTMLParameters: string;
var s: string;
begin
 //affichage du paramtre dans un tableau -- trs simple pour l'instant,  voir plus tard les formattages types TANAGRA
 s:= '<table border=1>';
 s:= s+'<tr><th colspan=2>PLS parameters</th></tr>';
 //la fonction format est trs pratique pour le formattage de chanes de caractres avec des paramtres (ex. l'entier)
 s:= s+format('<tr><td>Std dev estimate</td><td align="right" width=50>%d</td></tr>',[FTypeStdDev]);
 s:= s+'</table>';
 //renvoyer le rsultat
 result:= s;
end;

procedure TOpPrmPlsBase.LoadFromINI(prmSection: string;
  prmINI: TMemIniFile);
begin
 //charger  partir du fichier INI
 //mettre la valeur par dfaut si paramtre non-trouv
 FTypeStdDev:= prmINI.ReadInteger(prmSection,'type_std_dev',FTypeStdDev);
end;

procedure TOpPrmPlsBase.LoadFromStream(prmStream: TStream);
begin
 //charger  partir du flux (fichier binaire)
 prmStream.ReadBuffer(FTypeStdDev,sizeof(FTypeStdDev));
end;

procedure TOpPrmPlsBase.SaveToINI(prmSection: string; prmINI: TMemIniFile);
begin
 //sauver l'entier avec un nom de paramtre "type_std_dev" dans le fichier INI
 prmINI.WriteInteger(prmSection,'type_std_dev',FTypeStdDev);
end;

procedure TOpPrmPlsBase.SaveToStream(prmStream: TStream);
begin
 //sauver la valeur en spcifiant sa taille mmoire
 prmStream.WriteBuffer(FTypeStdDev,sizeof(FTypeStdDev));
end;

procedure TOpPrmPlsBase.SetDefaultParameters;
begin
 //par dfaut, on ralise le calcul de l'cart-type sur la population
 FTypeStdDev:= 0;
end;

initialization
 //enregistrer la classe gnratrice de composant dans l'application -- obligatoire
 RegisterClass(TMLGenCompPLSBase);
end.
