(***********************************************************)
(* UCalcSpvCN2.pas - Copyright (c) 2005 Ricco RAKOTOMALALA *)
(***********************************************************)

{
@abstract(dfinition du composant de calcul d'un arbre de dcision - algo ID3)
@author(Ricco)
@created(21/04/2005)

Algorithme CN2 -- DECISION LIST (Article originel de CLARCK et al., 1989)
Classe de calcul
}

unit UCalcSpvCN2;

interface

USES
   UCompSpvLDefinition,
   UCalcDistributionEvaluation, UCalcRulesDefinition,
   UDatasetExamples, UDatasetDefinition, UCalcDistribution, UCalcRulesImplementation,
   UCalcAssocStructure, EZDSLBar, UCalcSpvStructScore;

TYPE
   {********************************************}
   {*** types de mesures utilises dans CN2 ****}
   TEnumMeasureCN2 = (mCN2Shannon, mCN2JMeasure);
   {********************************************}

CONST
   STR_MEASURE_CN2 : array[TEnumMeasureCN2] of string = ('Shannon entropy','J-Measure');

TYPE
   {***************************************************}
   {* algorithme d'induction des "listes de dcision" *}
   {***************************************************}
   TCalcSpvCN2 = class(TCalcSpvLearning)
                 private
                 //risque critique
                 FSignifLevel: double;
                 //mesure  utiliser pour l'valuation des rgles
                 FMeasure: TFuncEvalDistribution;
                 //support min de la rgle
                 FSupportMin: integer;
                 //liste des rgles
                 FRules: TRuleKBSProductionRule;
                 //liste de tous les items -- une par modalit de descripteur -- rcupre des structures des rgles d'association
                 FDescriptorsItems: TLstAssocItem;
                 //items par modalit de conclusion
                 FClassItems: TLstAssocItem;
                 //les individus en cours de traitement
                 FCurExamples: TBooleanArray;
                 //la distribution sur la racine
                 FRootDistribution: TTabFrequence;
                 //la distribution de la rgle par dfaut
                 FDefaultDistribution: TTabFrequence;
                 //la conclusion par dfaut
                 FDefaultConclusion: TRuleProductionRule;
                 //rcuprer les paramtres de calcul
                 procedure   getParameters();
                 //construire tous les items -- candidats  la construction des rgles
                 procedure   buildAllItems(examples: TExamples);
                 //identifier les individus sur la racine
                 procedure   getRootExamples(examples: TExamples);
                 //calculer la distribution sur un ensemble d'individus
                 procedure computeClassDistribution(cov: TBooleanArray; var dist: TTabFrequence);
                 //obtenir la conclusion sur un sommet ET en profiter pour calculer les distributions
                 function getRuleConclusion(cov: TBooleanArray; var dist: TTabFrequence): TTypeDiscrete;
                 protected
                 //chercher la rgle suivante -- renvoie NIL si pas de rgle possible
                 function    searchNextRule(): TRuleProductionRule;
                 //
                 procedure   createStructures(); override;
                 procedure   destroyStructures(); override;
                 {apprentissage proprement dit}
                 function    coreLearning(examples: TExamples): boolean; override;
                 public
                 procedure   getScore(example: integer; var postProba: TTabScore); override;
                 function    getHTMLResults(): string; override;
                 end;

implementation

uses
       Math, Sysutils,
       UCompSpvCN2, ULogFile, UStringAddBuffered, UConstConfiguration;

{ TCalcSpvCN2 }

procedure TCalcSpvCN2.buildAllItems(examples: TExamples);
var j: integer;
begin
 //pour les descripteurs
 for j:= 0 to pred(self.Descriptors.Count) do
  FDescriptorsItems.buildItems(self.Descriptors.Attribute[j],examples);
 //pour les modalits de la variable  prdire
 FClassItems.buildItems(self.ClassAttribute,examples);
end;

procedure TCalcSpvCN2.getScore(example: integer; var postProba: TTabScore);
var rule: TRuleProductionRule;
    i: integer;
begin
 rule:= nil;
 //rcuprer la premire rgle dclenche
 for i:= 0 to pred(FRules.Count()) do
  begin
   if FRules.getRule(i).testExample(example)
    then
     begin
      rule:= FRules.getRule(i);
      BREAK;
     end;
  end;
 //si pas de rgle dclenche -- utilisation de la rgle par dfaut
 if not(assigned(rule))
  then rule:= FDefaultConclusion;
 //puis calculer la distribution
 postProba.recupFromTabFrequence(rule.Distribution);
 //si vide, prendre la racine
 if (postProba[0] = 0)
  then postProba.recupFromTabFrequence(FRootDistribution);
end;

procedure TCalcSpvCN2.computeClassDistribution(cov: TBooleanArray;
  var dist: TTabFrequence);
var k: integer;
    covK: TBooleanArray;
begin
  //dommage pour le temps de calcul mais ce serait trop confus si on le met en var. globale
  covK:= TBooleanArray.Create(cov.Capacity);
  dist.ReInitialization();
  //compter les individus pour chaque modalit de la classe
  for k:= 1 to self.ClassAttribute.nbValues do
   begin
    covK.CopyFrom(cov);
    covK.AndArray(FClassItems.Item[pred(k)].BoolArray);
    covK.RefreshCount();
    dist.Value[k]:= covK.Count;
   end;
  //et donc...
  covK.Free();
end;

function TCalcSpvCN2.coreLearning(examples: TExamples): boolean;
var rule: TRuleProductionRule;
    conclusion: TTypeDiscrete;
begin
 //rcuprer les paramtres de calcul
 self.getParameters();
 //construire tous les items
 self.buildAllItems(examples);
 //construire la racine -- les exemples  traiter
 self.getRootExamples(examples);
 //debug.
 TraceLog.WriteToLogFile(format('[CN2-DL] available examples for learning = %d',[FCurExamples.Count]));
 TRY
  REPEAT
   rule:= self.searchNextRule();
   if assigned(rule)
    //si OK, on ajoute la rgle
    then FRules.addRule(rule)
    //si pas(OK) c'est la fin, on dtecte la rgle par dfaut
    else
     begin
      //debug.
      TraceLog.WriteToLogFile(format('[CN2-DL] available examples for default rule = %d',[FCurExamples.Count]));
      //
      if assigned(FDefaultConclusion) then FDefaultConclusion.Free();
      conclusion:= self.getRuleConclusion(FCurExamples,FDefaultDistribution);
      if (conclusion > 0)
       then
        begin
         FDefaultConclusion:= TRuleProductionRule.Create(FALSE,TRuleCondDiscrete.Create(self.ClassAttribute,1,conclusion));
         FDefaultConclusion.setDistribution(FDefaultDistribution);
        end;
     end;

  UNTIL (rule = nil);
 //and then...
 result:= true;
 EXCEPT
 result:= false;
 END;
end;

procedure TCalcSpvCN2.createStructures;
begin
 FRules:= TRuleKBSProductionRule.Create();
 FDescriptorsItems:= TLstAssocItem.Create();
 FClassItems:= TLstAssocItem.Create();
 FRootDistribution:= TTabFrequence.CreateFromAtt(self.ClassAttribute);
 FDefaultDistribution:= TTabFrequence.CreateFromAtt(self.ClassAttribute);
end;

procedure TCalcSpvCN2.destroyStructures;
begin
 if assigned(FCurExamples) then FCurExamples.Free();
 if assigned(FDefaultConclusion) then FDefaultConclusion.Free();
 FDefaultDistribution.Free();
 FRootDistribution.Free();
 FDescriptorsItems.Free();
 FClassItems.Free();
 FRules.Free();
end;

function TCalcSpvCN2.getHTMLResults: string;
var s: string;
    i: integer;
    rule: TRuleProductionRule;
    buf: TBufString;
begin
 buf:= TBufString.Create();
 buf.BeginUpdate();
 buf.AddStr(format('<H3>Number of rules = %d</H3>',[FRules.Count]));
 buf.AddStr('<H3>Knowledge-based system</H3>');
 buf.AddStr(HTML_HEADER_TABLE_RESULT);
 buf.AddStr(HTML_TABLE_COLOR_HEADER_GRAY+'<TH>Antecedent</TH><TH>Consequent</TH><TH>Distribution</TH></TR>');
 //chaque rgle
 for i:= 0 to pred(FRules.count) do
  begin
   rule:= FRules.getRule(i);
   //
   s:= HTML_TABLE_COLOR_DATA_GRAY;
   //premisse
   s:= s+'<TD>';
   if (i>0) then s:= s+'ELSE ';
   s:= s+'IF '+rule.Premisse.getHTMLDescription()+'</TD>';
   //conclusion
   s:= s+'<TD>'+rule.Conclusion.getHTMLDescription()+'</TD>';
   //distribution
   s:= s+'<TD align="right">'+rule.Distribution.getHTMLResultSimplified()+'</TD>';
   s:= s+'</TR>';
   //
   buf.AddStr(s);
  end;
 //rgle par dfaut
 if assigned(FDefaultConclusion)
  then buf.AddStr(HTML_TABLE_COLOR_DATA_GRAY+'<TD>ELSE (DEFAULT RULE)</TD><TD>'+
                  FDefaultConclusion.Conclusion.getHTMLDescription()+'</TD><TD align="right">'+FDefaultDistribution.getHTMLResultSimplified()+'</TD></TR>');
 buf.AddStr('</table>');
 buf.EndUpdate();
 result:= buf.BufS;
 buf.Free();
end;

procedure TCalcSpvCN2.getParameters;
begin
 FSignifLevel:= (OpPrmSpv as TOpPrmCN2).SignifLevel;
 case (OpPrmSpv as TOpPrmCN2).Measurement of
  mCN2JMeasure: FMeasure:= JMeasure
  else FMeasure:= Shannon;//on fixe la Shannon comme mesure par dfaut
 end;
 FSupportMin:= (OpPrmSpv as TOpPrmCN2).SupportMin;
end;

procedure TCalcSpvCN2.getRootExamples(examples: TExamples);

begin
 if assigned(FCurExamples) then FCurExamples.Free();
 FCurExamples:= TBooleanArray.Create(examples.Size);
 //tous sont couverts bien sr
 FCurExamples.SetAllTrue();
 FCurExamples.RefreshCount();
 //distribution sur la racine
 FRootDistribution.Refresh(examples);
end;

//recherche hill-climbing to simple !!!
function TCalcSpvCN2.getRuleConclusion(cov: TBooleanArray;
  var dist: TTabFrequence): TTypeDiscrete;
var kMax: integer;
begin
 //calculer d'abord la distribution
 computeClassDistribution(cov,dist);
 //si pas d'individus couverts, on passe par la conclusion du classifieur par dfaut
 if (dist.Value[0] = 0)
  then kMax:= self.StatClassAtt.TabFreq.getIndexMaxValue()
  else kMax:= dist.getIndexMaxValue();
 result:= kMax;
end;

function TCalcSpvCN2.searchNextRule: TRuleProductionRule;
var covP,covC: TBooleanArray;
    rule: TRuleProductionRule;
    stopSearch: boolean;
    predDist,curDist: TTabFrequence;
    item: TAssocItem;
    j,jMin: integer;
    minFit,Fit: double;
begin
 //individus de travail, copie de travail
 covP:= TBooleanArray.CreateFrom(FCurExamples);
 covP.RefreshCount();
 covC:= TBooleanArray.Create(FCurExamples.Capacity);
 curDist:= TTabFrequence.CreateFromAtt(self.ClassAttribute,NIL);
 predDist:= TTabFrequence.CreateFromAtt(self.ClassAttribute,NIL);
 computeClassDistribution(covP,predDist);
 //rgle de travail
 rule:= TRuleProductionRule.create(TRUE,NIL);
 //boucler
 stopSearch:= false;
 minFit:= MATH.MaxDouble;
 //TraceLog.WriteToLogFile(format('[CN2-DL] Add rule number == %d',[FRules.Count()]));
 while not(stopSearch) do
  begin
   jMin:= -1;
   //pour chaque couple "attribut-valeur" candidat
   for j:= 0 to pred(FDescriptorsItems.Count) do
    begin
     item:= FDescriptorsItems.Item[j];
     //vrifier si l'attribut est dj dans la rgle
     if rule.isDescriptorPresent(item.Attribute)
      //passer  l'attribut suivant
      then CONTINUE
      else
       begin
        //calculer les individus couverts
        covC.CopyFrom(covP);
        covC.AndArray(item.BoolArray);
        covC.RefreshCount();
        //et la distribution sur la classe
        computeClassDistribution(covC,curDist);
        //tester d'abord si a passe les rgles d'arrt (pre-pruning)
        {$B-}
        if (curDist.Value[0]>=FSupportMin) and (CompareDistribCHI2(predDist,curDist,NIL)<FSignifLevel)
         then
          //est-ce qu'il est meilleur alors ?
          begin
           Fit:= FMeasure(FRootDistribution,curDist,NIL);
           if (Fit<minFit)
            then
             begin
              minFit:= Fit;
              jMin:= j;
             end;
          end;
       end;
    end;
   //alors ?
   if (jMin>=0)
    then
     begin
      //ajouter l'item  la rgle
      item:= FDescriptorsItems.Item[jMin];
      rule.addCondition(TRuleCondDiscrete.Create(item.Attribute,1,item.Value));
      //refresh des individus couverts
      covP.AndArray(item.BoolArray);
      covP.RefreshCount();//on a les individus couverts par la prmisse ici...
      //et on est prt  continuer...
      //TraceLog.WriteToLogFile(format('[CN2-DL] add %s, covered example = %d',[item.Description,covP.Count]));
      //mj de la distribution
      computeClassDistribution(covP,predDist);
     end
    //sinon, il est temps de s'arrter
    else stopSearch:= TRUE;
  end;
 //mettre  jour les distributions
 if rule.isEmptyPremisse()
  then
   begin
    FreeAndNil(rule);
    curDist.Free();
   end
  else
   begin
    //calculer la conclusion de la rgle
    rule.setConclusion(TRuleCondDiscrete.Create(self.ClassAttribute,1,getRuleConclusion(covP,curDist)));
    rule.setDistribution(curDist);//et ne pas dtruire curDist surtout !!!
    //******
    //*** mj des individus couverts dans les individus courants -- il s'agit des individus qui sont dans FCurExamples mais pas dans CovP
    //inverser la slection dans covP
    covP.ToggleAll();
    covP.RefreshCount();

    FCurExamples.AndArray(covP);
    FCurExamples.RefreshCount();
    //******
   end;
 //renvoyer l rgle (qui est  nil si rien n'a t ajout
 result:= rule;
 //rz
 covP.Free();
 covC.Free();
end;

end.
