(***********************************************************)
(* UCompSpvRBF.pas - Copyright (c) 2004 Ricco RAKOTOMALALA *)
(***********************************************************)

{
@abstract(Radial basis function)
@author(Ricco)
@created(12/01/2004)
Apprentissage en mode off-line - les noyaux sont dfinis par une procdure externe,
par exemple on aura effectu un K-Means ou un Kohonen au pralable, un attribut sert
 les dcrire.
}
unit UCompSpvRBF;

interface

USES
        Forms,Classes,IniFiles,
        UCompDefinition,
        UCompSpvLDefinition,
        UOperatorDefinition,
        UCalcSpvMLPStructure,
        UDatasetDefinition,
        UCalcStatDes,
        UCalcStatDesConditionnalDesc,
        UDatasetExamples,
        UCalcDistribution,
        UCompSpvMLPerceptron;


TYPE
        {le gnrateur de composant Supervised RBF}
        TMLGCompSpvRBF = class(TMLGenCompSpvLearning)
                         protected
                         procedure   GenCompInitializations(); override;
                         public
                         function    GetClassMLComponent: TClassMLComponent; override;
                         end;

        {le composant SpvMLP}
        TMLCompSpvRBF = class(TMLCompSpvLearning)
                        protected
                        function    getClassOperator: TClassOperator; override;
                        end;

        {l'oprateur}
        TOpSpvRBF = class(TOpSpvLearningContinuous)
                    protected
                    function    getClassParameter: TClassOperatorParameter; override;
                    function    getClassSpvLearning(): TClassCalcSpvLearning; override;
                    end;

        {paramtres de l'oprateur -- !!! hritier du MLP, certains paramtres sont fixs}
        TOpPrmSpvRBF = class(TOpPrmSpvMLP)
                       private
                       {nom de l'attribut dfinissant les noyaux}
                       FAttRBFNodeName: string;
                       protected
                       procedure   SetDefaultParameters(); override;
                       function    CreateDlgParameters(): TForm; override;
                       function    getHTMLArchitectureDescription(): string; override;
                       public
                       procedure   LoadFromStream(prmStream: TStream); override;
                       procedure   LoadFromINI(prmSection: string; prmINI: TMemIniFile); override;
                       procedure   SaveToStream(prmStream: TStream); override;
                       procedure   SaveToINI(prmSection: string; prmINI: TMemIniFile); override;
                       property    AttRBFNodeName: string read FAttRBFNodeName write FAttRBFNodeName;
                       end;

        {classe du calculateur de RBF}
        TCalcSpvRBF = class(TCalcSpvMLP)
                      private
                      {attribut de noyaux RBF}
                      FAttRBFNode: TAttribute;
                      {statistiques de noyaux}
                      FStatsRBF: TLstStatDesCondANOVA;
                      {nombre de victoire des noyaux  chaque passage}
                      FTabWins: array of double;
                      {nombre de passage}
                      FNbProcessed: integer;
                      protected
                      procedure   beforeProcessPatterns(); override;
                      procedure   afterProcessPatterns(); override;
                      procedure   createStructures(); override;
                      procedure   destroyStructures(); override;
                      function    beforeLearning(examples: TExamples): boolean; override;
                      {le neurone d'entre est maintenant un noyau}
                      procedure   InputPattern(example: integer); override;
                      {on est oblig de tout recalculer entirement l'input pattern dans ce cas}
                      procedure   SetAttToBlank(example: integer; prmAttEval: integer); override;
                      public
                      function    getHTMLResults(): string; override;
                      end;



implementation

uses
        UStringsResources, UDlgOpPrmSpvRBF, UConstConfiguration,
        SysUtils, Math,
        ULogFile;

CONST
        //cette contrainte est trs trange -- cf. RBFNetwork.java de WEKA
        RBF_MIN_STD_DEV_NOYAU_GAUSSIEN : double = 0.1;

        //normalisation
        RBF_GAMMA_NOYAU_GAUSSIEN : double = 1.0;

        //echelle de redistribution des donnes
        RBF_DATA_SCALE : double = 6.0;


{ TMLGCompSpvRBF }

procedure TMLGCompSpvRBF.GenCompInitializations;
begin
 FMLComp:= mlcSpvLearning;
 //FMLNumIcon:= 30;
 //FMLCompName:= str_comp_name_spvl_rbf;
 //FMLBitmapFileName:= 'MLSpvRBF.bmp';
end;

function TMLGCompSpvRBF.GetClassMLComponent: TClassMLComponent;
begin
 result:= TMLCompSpvRBF;
end;

{ TMLCompSpvRBF }

function TMLCompSpvRBF.getClassOperator: TClassOperator;
begin
 result:= TOpSpvRBF;
end;

{ TOpSpvRBF }

function TOpSpvRBF.getClassParameter: TClassOperatorParameter;
begin
 result:= TOpPrmSpvRBF;
end;

function TOpSpvRBF.getClassSpvLearning: TClassCalcSpvLearning;
begin
 result:= TCalcSpvRBF;
end;

{ TOpPrmSpvRBF }

function TOpPrmSpvRBF.CreateDlgParameters: TForm;
begin
 result:= TDlgOpPrmSpvRBF.CreateFromOpPrm(self);
end;

function TOpPrmSpvRBF.getHTMLArchitectureDescription: string;
var s: string;
begin
 s:= HTML_TABLE_COLOR_HEADER_GRAY+'<TH colspan=2>RBF architecture</TH></TR>';
 s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>Cluster attribute</TD><TD>%s</TD>',[FAttRBFNodeName]);
 result:= s;
end;

procedure TOpPrmSpvRBF.LoadFromINI(prmSection: string;
  prmINI: TMemIniFile);
begin
 inherited LoadFromINI(prmSection,prmINI);
 FAttRBFNodeName:= prmINI.ReadString(prmSection,'att_rbf_clus',FAttRBFNodeName);
end;

procedure TOpPrmSpvRBF.LoadFromStream(prmStream: TStream);
var l: integer;
begin
 inherited LoadFromStream(prmStream);
 prmStream.ReadBuffer(l,sizeof(l));
 if (l>0)
  then
   begin
    setLength(FAttRBFNodeName,l);
    prmStream.ReadBuffer(FAttRBFNodeName[1],l);
   end;
end;

procedure TOpPrmSpvRBF.SaveToINI(prmSection: string; prmINI: TMemIniFile);
begin
 inherited SaveToIni(prmSection,prmINI);
 prmINI.WriteString(prmSection,'att_rbf_clus',FAttRBFNodeName);
end;

procedure TOpPrmSpvRBF.SaveToStream(prmStream: TStream);
var l: integer;
begin
 inherited SaveToStream(prmStream);
 l:= length(FAttRBFNodeName);
 prmStream.WriteBuffer(l,sizeof(l));
 if (l>0)
  then prmStream.WriteBuffer(FAttRBFNodeName[1],l);
end;

procedure TOpPrmSpvRBF.SetDefaultParameters;
begin
 // on rcupre les paramtres par dfaut de l'anctre
 inherited SetDefaultParameters();
 //on fixe les paramtres courants (les autres restent donc valables)
 self.UseHiddenLayer:= FALSE;//pas de couche cache
 self.AttTransform:= 0;//pas de transformation pralable des attributs
 self.MaxIteration:= 10;//10 itrations max.
 self.UseTestErrStagnation:= FALSE;//dsactiver le test d'arrt sur la stagnation de l'erreur
 //le nom de variable de noyaux est vide au dpart
 FAttRBFNodeName:= '';
end;

{ TCalcSpvRBF }

procedure TCalcSpvRBF.afterProcessPatterns;
var i: integer;
    s: string;
begin
 s:= 'wins [';
 for i:= 1 to FInputLayerSize do
  s:= s+format('%.0f,',[FTabWins[i]]);
 s:= copy(s,1,pred(length(s)))+']';
 s:= s+format(' -> processed : %d',[FNbProcessed]);
 //TraceLog.WriteToLogFile(s);
end;

function TCalcSpvRBF.beforeLearning(examples: TExamples): boolean;
var ok: boolean;
    j: integer;
    att: TAttribute;
    stat: TCalcSDCondDesc;
    s: string;
    k: integer;
begin
 if assigned(FAttRBFNode)
  then
   begin
    ok:= inherited beforeLearning(examples);
    //puis les stats conditionnelles
    if ok
     then
      begin
       FStatsRBF:= TLstStatDesCondANOVA.Create(nil,nil);
       TRY
       for j:= 0 to pred(Descriptors.Count) do
        begin
         att:= descriptors.Attribute[j];
         stat:= TCalcSDCondDescANOVA.Create(att,FAttRBFNode,examples);
         FStatsRBF.AddStat(stat);
         s:= 'RBF node stat: '+att.Name+'(';
         for k:= 0 to pred(stat.AttDescription.nbValues) do
          s:= s+format('%.2f (%.2f)',[stat.StatCond[k].Average,stat.StatCond[k].Variance])+'|';
         s:= copy(s,1,pred(length(s)))+')';
         //TraceLog.WriteToLogFile(s);
        end;
       EXCEPT
        ok:= FALSE;
       END;
      end;
    result:= ok;
   end
  else result:= FALSE;
end;

procedure TCalcSpvRBF.beforeProcessPatterns;
var i: integer;
begin
 //vider le tableau
 for i:= 1 to FInputLayerSize do
  FTabWins[i]:= 0.0;
 FNbProcessed:= 0;
end;

procedure TCalcSpvRBF.createStructures;
begin
 //vrifier si l'attribut dsign est bien utilisable
 FAttRBFNode:= FAllAttributes.GetFromName((prmMLP as TOpPrmSpvRBF).AttRBFNodeName);
 //si tout est ok, on peut crer le rseau
 if assigned(FAttRBFNode)
  then
   begin
    FInputLayerSize:= FAttRBFNode.nbValues;//le nombre de neurones dans la couche d'entre est modifie
    //et on construit la structure
    inherited createStructures();
   end;
 //aux fins de tests
 setLength(FTabWins,succ(FInputLayerSize));
 FNbProcessed:= 0;
end;

procedure TCalcSpvRBF.destroyStructures;
begin
 inherited destroyStructures();
 if assigned(FStatsRBF)
  then FStatsRBF.Free;
 setLength(FTabWins,0);
end;

function TCalcSpvRBF.getHTMLResults: string;
var s: string;
    i,j: integer;
    stat: TCalcSDCondDesc;
begin
 s:= '<H3>Clusters characteristics - Average </H3>';
 s:= s+HTML_HEADER_TABLE_RESULT;
 s:= s+HTML_TABLE_COLOR_HEADER_GRAY+'<TH>Attribute</TH>';
 for i:= 1 to FNbNeuronsPerLayer[1] do
  s:= s+format('<TH>Cluster n%d</TH>',[i]);
 s:= s+'</TR>';
 //pour chaque attribut
 for j:= 0 to pred(descriptors.Count) do
  begin
   stat:= TCalcSDCondDesc(FStatsRBF.Stat(j));
   s:= s+HTML_TABLE_COLOR_DATA_GRAY+format('<TD>%s</TD>',[stat.Attribute.Name]);
   //pour chaque noyau
   for i:= 1 to stat.AttDescription.nbValues do
    s:= s+format('<TD align=right>%.4f</TD>',[stat.StatCond[pred(i)].Average]);
   s:= s+'</TR>';
  end;
 s:= s+'</table>';
 s:= s+ inherited getHTMLResults();
 result:= s;
end;

procedure TCalcSpvRBF.InputPattern(example: integer);
var i,j: integer;
    sOut,outValue,Value: double;
    stat: TCalcSDCondDesc;
    statRBF: TCalcStatDesContinuous;
    //statGlob: TCalcStatDesContinuous;
    minValue: double;
    iMin: integer;

    //s: string;
    vDiff,vStd,logValue: double;

    minOut,maxOut: double;
    rangeOut: double;
begin
 //debug.
 //s:= format('example = %d ',[example]);
 //
 sOut:= 0.0;
 iMin:= 0;
 minValue:= Math.MaxDouble;

 minOut:= Math.MaxDouble;
 maxOut:= -1.0*Math.MaxDouble;

 //pour chaque noyau
 for i:= 1 to FInputLayerSize do
  begin
   //debug.
   //s:= s+format('>> id = %d << ',[i]);
   //calculer l'entre du noyau
   //distance entre l'individu et le centre du noyau
   value:= 0.0;
   for j:= 0 to pred(descriptors.Count) do
    begin
     //pour l'attribut courant
     stat:= TCalcSDCondDesc(FStatsRBF.Stat(j));
     //pour le noyau courant
     statRBF:= stat.StatCond[pred(i)];
     //la statistique globale pour l'attribut
     //statGlob:= TCalcStatDesContinuous(FStatsDescriptors.Stat(j));
     
     //***********************************************************
     //calculer la distance
     //pourquoi doit-on pondrer avec la variance locale ?
     //et non pas avec la variance globale ?
     //ou ne pas pondrer du tout ?

     //>>distance pondre par la variance locale
     //choix de l'cart-type  utiliser
     if (StatRBF.StdDev>RBF_MIN_STD_DEV_NOYAU_GAUSSIEN)
      then vStd:= statRBF.StdDev
      else vStd:= RBF_MIN_STD_DEV_NOYAU_GAUSSIEN;

     //cart par rapport au centre de gravit du noyau 
     vDiff:= descriptors.Attribute[j].cValue[example]-statRBF.Average;

     //normaliser avec l'cart-type global
     //vStd:= (FStatsDescriptors.Stat(j) as TCalcStatDesContinuous).StdDev;
     //vDiff:= vDiff/vStd;

     //log. nprien
     logValue:= -0.5*SQR(vDiff / vStd) - 0.5*LN(2.0*Pi) - LN(vStd);
     //logValue:= SQR(vDiff / vStd);

     //additionner
     value:= value+logValue;

     //>> distance non-pondre
     //Value:= Value+SQR(descriptors.Attribute[j].cValue[example]-statRBF.Average)
     //tracelog.WriteToLogFile(format('noyau %d -- att:%s -- value:%.2f -- avg:%.4f -- variance: %.4f',
     //                        [i,descriptors.Attribute[j].Name,descriptors.Attribute[j].cValue[example],statRBF.Average,statRBF.Variance]));

     //>> distance pondre par la variance globale... les donnes sont donc considres rduites
     //value:= value + vDiff * vDiff;

     //***********************************************************
    end;

   //normaliser par rapport au nombre de variables -- l'ide est d'tre le moins dpendant possible de l'chelle du problme (le nombre de variables ici)
   value:= value/(1.0*descriptors.Count);

   //test - contre-propagation
   if (Value<minValue)
    then
     begin
      minValue:= Value;
      iMin:= i;
     end;
     
   //noyau gaussien donc
   outValue:= exp(RBF_GAMMA_NOYAU_GAUSSIEN*value);

   //test -- autrement ?
   //outValue:= 1.0/(value * value);

   //sortie du RBF
   MLP^[1]^[i].OutputValue:= outValue;

   //param. de normalisation
   if (outValue<minOut)
    then minOut:= outValue;
   if (outValue>maxOut)
    then maxOut:= outValue;
    
   //debug.
   //s:= s+Format('[%.4f -> %.4f] --',[value,outValue]);
   //
   sOut:= sOut+outValue;
  end;

 //s:= format('minOut : %.4f, maxOut : %.4f',[minOut,maxOut]);
 //if (example = 1)
  //then TraceLog.WriteToLogFile(s);

 //normalisation des sorties
 //ou mise en 0/1 des sorties - cf. la contre-propagation

 
 if (sOut > 0.0)
  then
   begin
     inc(FNbProcessed);
     FTabWins[iMin]:= FTabWins[iMin]+1.0;

     //utiliser la victoire - dfaite ?
     //For i:= 1 to FAttRBFNode.nbValues do
     // begin
     //  if (i=iMin)
     //   then MLP^[1]^[i].OutputValue:= 1.0
     //   else MLP^[1]^[i].OutputValue:= 0.0;
     // end;


     //normalisation
     //For i:= 1 to FAttRBFNode.nbValues do
     // MLP^[1]^[i].OutputValue:= MLP^[1]^[i].OutputValue/sOut;

     //normalisation -x.xx - +x.xx
     rangeOut:= maxOut-minOut;
     if (rangeOut>0)
      then
       begin
        //s:= '';
        For i:= 1 to FAttRBFNode.nbValues do
         begin
          //faire varier entre -3.0 et +3.0
          value:= -0.5*RBF_DATA_SCALE+RBF_DATA_SCALE*(MLP^[1]^[i].OutputValue-minOut)/rangeOut;
          //s:= s+format('[%d = %.4f]',[i,value]);
          MLP^[1]^[i].OutputValue:= value;
         end;
        //TraceLog.WriteToLogFile(format('example %d --> %s',[example,s]));
       end
      else
       begin
        For i:= 1 to FAttRBFNode.nbValues do
         MLP^[1]^[i].OutputValue:= 0.0;
       end;


     //normalisation par la somme -- cf. formule (6.30) livre Hastie et al. (2001)
     (*
     for i:= 1 to FAttRBFNode.nbValues do
      MLP^[1]^[i].OutputValue:= MLP^[1]^[i].OutputValue/sOut;
     *)
   end;
 
end;

procedure TCalcSpvRBF.SetAttToBlank(example: integer; prmAttEval: integer);
var i,j: integer;
    outValue,Value,sOut: double;
    stat: TCalcSDCondDesc;
    statRBF: TCalcStatDesContinuous;
begin
 sOut:= 0.0;
 //pour chaque noyau
 for i:= 1 to FNbNeuronsPerLayer[1] do
  begin
   //calculer l'entre du noyau
   Value:= 0.0;
   for j:= 0 to pred(descriptors.Count) do
    begin
     //pour l'attribut courant
     stat:= TCalcSDCondDesc(FStatsRBF.Stat(j));
     //pour le noyau courant
     statRBF:= stat.StatCond[pred(i)];
     //calculer la distance
     if (prmAttEval = j)
      //on remplace la valeur par la moyenne sur l'chantillon
      //cf. commentaire plus haut sur la pondration (on devrait diviser par l'cart-type du noyau ???)
      then Value:= Value+SQR(1.0*(TCalcStatDesContinuous(FStatsDescriptors.Stat(j)).Average-statRBF.Average)/1.0)
      else Value:= Value+SQR(1.0*(descriptors.Attribute[j].cValue[example]-statRBF.Average)/1.0);
    end;
   //noyau gaussien donc
   outValue:= exp(-1.0*Value);
   //affectation  la premire couche du rseau, i.e. les sorties du noyau
   MLP^[1]^[i].OutputValue:= outValue;
   sOut:= sOut+outValue;
  end;
 //normaliser
 if (sOut>0.0)
  then
   begin
    for i:= 1 to FNbNeuronsPerLayer[1] do
     MLP^[1]^[i].OutputValue:= MLP^[1]^[i].OutputValue/sOut;
   end;
end;

initialization
 RegisterClass(TMLGCompSpvRBF);
end.
