(*******************************************************************)
(* UOperatorDefinition.pas - Copyright (c) 2004 Ricco RAKOTOMALALA *)
(*******************************************************************)

{
@abstract(L'oprateur associe  un composant visuel)
@author(Ricco)
@created(12/01/2004)
Cette unit contient l'oprateur de calcul associ  un composant
visuel. Il est subdivis en deux parties : (1) le paramtrage de l'oprateur,
cr systmatiquement; (2) l'oprateur de calcul cr systmatiquement galement.
}
unit UOperatorDefinition;

interface

USES
        Forms, Classes, IniFiles;

TYPE

        {Types d'erreur prcedent l'excution}
        TEnumErrCanExecute = (errCanExecNoError, errCanExecExamples, errCanExecAttributes, errCanExecOthers);

        {Types d'erreur pourvant survenir  l'excution -  voir si opportun de l'ajouter}
        TEnumErrExec = (errExeNoError,errExeCannotExecute,errExeCoreExecute);

        {forward}
        TOperatorParameter = class;

        {les classes de classes}
        TClassOperatorParameter = class of TOperatorParameter;

        {l'oprateur de calcul lui mme, cr  chaque demande de calcul}
        TOperator = class(TObject)
                    private
                    {l'outil qui permet de paramtrer l'oprateur de calcul}
                    FPrmOp: TOperatorParameter;
                    {Indicateur permettant de savoir s'il est possible de visualiser un rsultat}
                    FIsAvailable: Boolean;
                    {Fiche associe  l'oprateur}
                    FForm: TForm;
                    {Le composant associ  l'oprateur - TObject pour viter les rfs. croiss}
                    FMLOwner: TObject;
                    protected
                    {renvoie la classe du paramtrage,  surcharger obligatoirement pour chaque hritier}
                    function    getClassParameter: TClassOperatorParameter; virtual; abstract;
                    {rinitialisation de l'oprateur de calcul, prt pour une nouvelle session - peut tre surcharg,
                    toujours avec inherited}
                    procedure   ReInitialize(); virtual;
                    {modifier l'tat dispo ou pas...}
                    procedure   SetAvailable(prmAv: boolean);
                    {crer une nouvelle instance de la fiche de paramtrage}
                    function    getNewForm(): TForm; virtual;
                    {checker les individus  utiliser}
                    function    CheckExamples(): boolean; virtual;
                    {checker les attributs  utiliser}
                    function    CheckAttributes(): boolean; virtual;
                    {checker autre chose avant de pouvoir excuter}
                    function    CheckOthers(): boolean; virtual;
                    {toutes les condition sont runies pour l'excution ?}
                    function    CanExecute(): TEnumErrCanExecute; virtual;
                    {l'execution proprement dite -  surcharger obligatoirement chez les descendants !!!}
                    function    CoreExecute(): boolean; virtual; abstract;
                    {post-excution en cas de succs, trs peu de surcharge}
                    procedure   AfterSuccessExecute(); virtual;
                    {sauver les infos locales dans le flux - surchargeable}
                    procedure   SaveOperatorInfoToStream(prmStream: TStream); virtual;
                    {sauver les infos locales dans un fichier INI - surchargeable}
                    procedure   SaveOperatorInfoToINI(prmSection: string; prmINI: TMemIniFile); virtual;
                    {charger les infos internes  partir du stream}
                    procedure   LoadOperatorInfoFromStream(prmStream: TStream); virtual;
                    {charger les infos internes  partir d'un fichier INI}
                    procedure   LoadOperatorInfoFromINI(prmSection: string; prmINI: TMemIniFile); virtual;
                    public
                    {construction du paramtrage entre autres est ralis ici}
                    constructor Create(AOwner: TObject); virtual;
                    {destruction, incluant la fiche et les paramtres}
                    destructor  Destroy; override;
                    {montrer la fiche}
                    procedure   ShowForm();
                    {cacher et supprimer la fiche}
                    procedure   ReleaseForm();
                    {lancer une execution}
                    function    Execute(): boolean; virtual;
                    {charger  partir d'un flux}
                    procedure   LoadFromStream(prmStream: TStream);
                    {charger  partir d'un fichier INI}
                    procedure   LoadFromINI(prmSection: string; prmINI: TMemIniFile);
                    {sauver dans un flux}
                    procedure   SaveToStream(prmStream: TStream);
                    {sauver dans un fichier INI}
                    procedure   SaveToINI(prmSection: string; prmINI: TMemIniFile);
                    {rcupration des paramtres en HTML}
                    function    getHTMLParameters(): string;
                    {rcupration du rsum des rsultats -  surcharger absolument}
                    function    getHTMLResultsSummary(): string; virtual;
                    {envoyer un message d'avertissement aux utilisateurs}
                    procedure   SendUserMessage(prmStr: string);                    
                    {indique en public si le composant est dispo pour les affichages, cd qu'un calcul a t ralis et que le rsultat est dispo}
                    property   IsAvailable: Boolean read FIsAvailable write SetAvailable;
                    property   PrmOp: TOperatorParameter read FPrmOp;
                    property   Form: TForm read FForm write FForm;
                    property   MLOwner: TObject read FMLOwner;
                    end;

        {la classe d'oprateur}
        TClassOperator = class of TOperator;

        {le paramtrage d'un oprateur de calcul, il est cr systmatiquement avec le composant visuel}
        TOperatorParameter = class(TObject)
                             private
                             FOperator: TOperator;
                             protected
                             {fixer les paramtres par dfaut de l'analyse,  surcharger toujours, sans inherited}
                             procedure   SetDefaultParameters(); virtual; abstract;
                             {fait afficher une bote de dialogue dans laquelle on peut saisir/modifier les paramtres d'analyse
                              surcharger absolument}
                             function    CreateDlgParameters(): TForm; virtual; abstract;
                             {demande effectivement les paramtres,  surcharger si on doit manipuler les modifs utilisateurs}
                             procedure   GetParameters(var prmModif: boolean); virtual;
                             {prepare une mise  jour avant de faire apparatre la bote de dialogue des paramtres,
                             rien au dpart mais surcharger trs probable}
                             procedure   PrepareDlgParametersShow(); virtual;
                             public
                             {son propritaire est pass en paramtre : l'oprateur}
                             constructor Create(prmOp: TOperator); virtual;
                             {demander la bote de paramtres, renvoie vrai si clic sur OK}
                             function    AskParameter(): boolean;
                             {chargement  partir d'un flux}
                             procedure   LoadFromStream(prmStream: TStream); virtual;
                             {chargement  partir d'un fichier INI}
                             procedure   LoadFromINI(prmSection: string; prmINI: TMemIniFile); virtual;
                             {sauvegarde dans un flux}
                             procedure   SaveToStream(prmStream: TStream); virtual;
                             {sauvegarde dans un fichier INI}
                             procedure   SaveToINI(prmSection: string; prmINI: TMemIniFile); virtual;
                             {rcupration de la description des paramtres -  surcharger absolument}
                             function    getHTMLParameters(): string; virtual;
                             {oprateur associ}
                             property    Operator: TOperator read FOperator;
                             end;



implementation

uses
        SysUtils,
        UDlgBaseOperatorParameter, UFrmBaseOperator, UCompDefinition,
  ULogFile;

{ TOperator }

constructor TOperator.Create;
begin
 inherited Create();
 //les initialisations
 FPrmOp:= self.getClassParameter.Create(self);
 FMLOwner:= AOwner;
 FIsAvailable:= FALSE;
end;

destructor TOperator.Destroy;
begin
 self.ReInitialize();//inclut la destruction de la fiche de visualisation
 if assigned(FPrmOp)
  then FreeAndNil(FPrmOp);
 inherited Destroy();
end;

function TOperator.Execute: boolean;
var ok: boolean;
    errCanExec: TEnumErrCanExecute;
    sTmp: string;
begin
 ok:= FALSE;
 TRY
 errCanExec:= self.CanExecute;
 if (errCanExec=errCanExecNoError)
  then
   begin
    ok:= self.CoreExecute();
    if ok
     then self.AfterSuccessExecute()
     else self.SendUserMessage('error at CoreExecute, execution is not finalized');
   end
  else
   begin
    sTmp:= '';
    case errCanExec of
     errCanExecExamples: sTmp:= 'EXAMPLES';
     errCanExecAttributes: sTmp:= 'ATTRIBUTES';
     errCanExecOthers: sTmp:= 'OTHERS';
    end;
    sTmp:= format('STOP at CAN_EXECUTE, please check %s selection',[sTmp]);
    self.SendUserMessage(sTmp);
   end;
 except
 on e: exception do
  begin
   ok:= false;
   TraceLog.WriteToLogFile(format('%s.execute exception >> %s',[self.className,e.Message]));
  end;
 end;
 result:= ok;
end;

procedure TOperator.AfterSuccessExecute;
begin
 FIsAvailable:= TRUE;
end;

procedure TOperator.ReleaseForm;
begin
 if assigned(FForm)
  then
   begin
    FForm.Release();
    FForm:= nil;
   end;
end;

procedure TOperator.SetAvailable(prmAv: boolean);
begin
 //le rsultat prcdent est invalid
 //on rinitialise compltement le modle
 if (prmAv = false)
  then self.ReInitialize();
 //puis, on modifie le flag quoiqu'il en soit
 FIsAvailable:= prmAv;
end;

procedure TOperator.ShowForm;
begin
 if assigned(FForm)
  then
   begin
    FForm.WindowState:= wsNormal;
    FForm.Show;
   end
  else
   begin
    //#ToDo1 Il faudrait s'assurer que les rsultats sont montrables - donc qu'une excutaion a t ralise !!!
    //if self.IsAvailable
    FForm:= getNewForm();
   end;
end;

procedure TOperator.ReInitialize;
begin
 //fermeture de la fiche de visualisation si visible
 if assigned(FForm)
  then self.ReleaseForm();
end;

procedure TOperator.LoadFromStream(prmStream: TStream);
begin
 self.LoadOperatorInfoFromStream(prmStream);
 FPrmOp.LoadFromStream(prmStream);
end;

procedure TOperator.SaveToStream(prmStream: TStream);
begin
 self.SaveOperatorInfoToStream(prmStream);
 FPrmOp.SaveToStream(prmStream);
end;

function TOperator.getHTMLParameters: string;
begin
 result:= self.PrmOp.getHTMLParameters();
end;

function TOperator.getHTMLResultsSummary: string;
begin
 result:= Format('<H3>%s.getHTMLResultsSummary</H3>',[self.ClassName]);
end;

function TOperator.CanExecute: TEnumErrCanExecute;
var err: TEnumErrCanExecute;
begin
 err:= errCanExecNoError;
 if not(CheckExamples())
  then err:= errCanExecExamples
  else if not(CheckAttributes())
        then err:= errCanExecAttributes
        else if not(CheckOthers())
              then err:= errCanExecOthers;
 result:= err;
end;

function TOperator.getNewForm: TForm;
begin
 result:= TFrmBaseOperator.CreateFromOperator(self);
end;

procedure TOperator.SendUserMessage(prmStr: string);
begin
 (MLOwner as TMLComponent).SendUserMessage(prmStr);
end;

procedure TOperator.LoadOperatorInfoFromStream(prmStream: TStream);
begin
 //nothing
end;

procedure TOperator.SaveOperatorInfoToStream(prmStream: TStream);
begin
 //nothing
end;

procedure TOperator.SaveToINI(prmSection: string; prmINI: TMemIniFile);
begin
 self.SaveOperatorInfoToINI(prmSection,prmINI);
 FPrmOp.SaveToINI(prmSection,prmINI);
end;

procedure TOperator.SaveOperatorInfoToINI(prmSection: string;
  prmINI: TMemIniFile);
begin
 //nothing
end;

procedure TOperator.LoadFromINI(prmSection: string; prmINI: TMemIniFile);
begin
 self.LoadOperatorInfoFromINI(prmSection,prmINI);
 FPrmOp.LoadFromINI(prmSection,prmINI);
end;

procedure TOperator.LoadOperatorInfoFromINI(prmSection: string;
  prmINI: TMemIniFile);
begin
 //nothing
end;

function TOperator.CheckAttributes: boolean;
begin
 result:= TRUE;
end;

function TOperator.CheckExamples: boolean;
begin
 result:= TRUE;
end;

function TOperator.CheckOthers: boolean;
begin
 result:= TRUE;
end;

{ TOperatorParameter }

function TOperatorParameter.AskParameter: boolean;
var modif: boolean;
begin
 modif:= false;
 self.PrepareDlgParametersShow();
 self.GetParameters(modif);
 result:= modif;
end;

constructor TOperatorParameter.Create(prmOp: TOperator);
begin
 inherited Create();
 FOperator:= prmOp;
 //fixer les paramtres par dfaut
 self.SetDefaultParameters();
end;

function TOperatorParameter.getHTMLParameters: string;
begin
 //result:= Format('<H3>%s.getHTMLParameters </H3>',[self.ClassName]);
 result:= '';
end;

procedure TOperatorParameter.GetParameters(var prmModif: boolean);
var DlgPrm: TDlgBaseOperatorParameter;
begin
  dlgPrm:= self.CreateDlgParameters() as TDlgBaseOperatorParameter;
  if assigned(dlgPrm)
   then dlgPrm.GetPrmModification(prmModif)
   else self.FOperator.SendUserMessage('No parameters linked to this component');
end;

procedure TOperatorParameter.LoadFromINI(prmSection: string;
  prmINI: TMemIniFile);
begin
 //nothing
end;

procedure TOperatorParameter.LoadFromStream(prmStream: TStream);
begin
 //nothing
end;

procedure TOperatorParameter.PrepareDlgParametersShow;
begin
 //rien au dpart
end;

procedure TOperatorParameter.SaveToINI(prmSection: string;
  prmINI: TMemIniFile);
begin
 //nothing
end;

procedure TOperatorParameter.SaveToStream(prmStream: TStream);
begin
 //nothing
end;

end.
